import os
import shutil
import time

import utils.common.log as logger
from plugins.CSBS_VBS.common.upgrade.console_upgrade import ConsoleUpgrade
from plugins.CSBS_VBS.common.util import unzip_file
from plugins.CSBS_VBS.job.upgrade.base import BaseSubJob
from utils.common.exception import FCUException
from utils.common.message import Message

logger.init("CSBS-VBS")
RETRY_TIME_LIMIT = 3


class UpgradeConsole(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(UpgradeConsole, self).__init__(project_id, pod_id, regionid_list)
        self.console_upgrade = ConsoleUpgrade(self.params_store)

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            if self.upgrade_path.target_version \
                    != self.upgrade_path.console_pkg_version:
                raise Exception("The console package is not the expected one."
                                "Please check and upload console package "
                                "of the correct version.")

            # unzip packages
            pkg = os.path.join(self.upgrade_path.pkg_path,
                               self.upgrade_path.console_pkg_name)
            unzip_pkg_path = os.path.realpath(
                self.upgrade_path.pkg_path + "/csbs_console_temp")
            console_pkg_name = \
                "cbs-{}.tar.gz".format(self.upgrade_path.console_pkg_version)
            console_pkg_path = "{}/{}".format(unzip_pkg_path, console_pkg_name)
            # clear and unzip
            if os.path.exists(unzip_pkg_path):
                shutil.rmtree(unzip_pkg_path)
            unzip_file(pkg, unzip_pkg_path)
        except Exception as ex:
            logger.error(ex)
            return Message(500, FCUException(645000))

        try:
            # upload cf2 pkg into primary region ManageOne-Tenant01/02 nodes
            # For the primary region, the value of region_flag is false.
            # For other regions, the value is true.
            region_flag = False
            if not self.project_scene.is_primary_region:
                region_flag = True
            result = \
                self.console_upgrade.upload_console_pkg(
                    pod_id=pod_id,
                    console_file=console_pkg_path,
                    region_flag=region_flag)
            if result:
                logger.info("Upload primary region console upgrade "
                            "pkg to Console VM successfully.")
            else:
                logger.error("Failed to upload primary region console "
                             "upgrade pkg to VM.")
                return Message(500, FCUException(645005))

            logger.info("Start upgrade current console, "
                        "pkg is:{}".format(console_pkg_name))
            retry_times = 0
            while True:
                result = \
                    self.console_upgrade.upgrade_cbs_console(
                        pod_id=self.pod_id,
                        console_pkg_name=console_pkg_name,
                        region_flag=region_flag,
                        region_id=self.regionid_list[0],
                        pkg_version=self.upgrade_path.console_pkg_version)
                if not result and retry_times < RETRY_TIME_LIMIT:
                    retry_times += 1
                    logger.info("Failed to upgrade console, starting to retry,"
                                " and retry_times is {}.".format(retry_times))
                    time.sleep(10)
                else:
                    break
            if result:
                logger.info("Upgrade current Console successfully.")
            else:
                logger.error("Failed to upgrade current console, "
                             "after retrying {} times.".format(retry_times))
                return Message(500, FCUException(645015))

            # delete local temp console files
            if os.path.exists(unzip_pkg_path):
                shutil.rmtree(unzip_pkg_path)
            return Message(200)
        except Exception as e:
            err_msg = "Failed to upgrade console, reason: {}".format(str(e))
            logger.error(err_msg)
            return Message(500, FCUException(645015))
