import time

import utils.common.log as logger
from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.job.upgrade.base import BaseSubJob
from utils.business.manageone_gateway_domainname import MOCertManager
from utils.common.exception import FCUException
from utils.common.message import Message
from utils.common.ssh_util import Ssh

logger.init("CSBS-VBS")
RETRY_TIME_LIMIT = 3
SLEEP_SECONDS = 30


class UpgradeMOCerts(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(UpgradeMOCerts, self).__init__(project_id, pod_id, regionid_list)
        self.ssh_client = SshClient()
        self.mo_cert_manager = MOCertManager(pod_id)
        self.home_dir = "/home/djmanager"

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Upgrade iam, sc and oc certs on karbor nodes
        """
        logger.info("Start upgrade karbor MO certs.")
        karbor_nodes = [self.karbor_params.cbs_node0,
                        self.karbor_params.cbs_node1]
        for node in karbor_nodes:
            ssh_client = self.ssh_client.get_sshclient_user_su_root(node)
            self._upload_mo_apimlb_certs(node, ssh_client)
            retry_times = 0
            while retry_times < RETRY_TIME_LIMIT:
                try:
                    if not self._update_mo_certs(ssh_client):
                        retry_times += 1
                        logger.error("Update mo certs failed on node:{}, "
                                     "sleep {}s, retry "
                                     "times:{}.".format(node.ip,
                                                        SLEEP_SECONDS,
                                                        retry_times))
                        time.sleep(SLEEP_SECONDS)
                        continue
                    else:
                        self.ssh_client.ssh_close(ssh_client)
                        logger.info("Succeed update mo certs on "
                                    "node:{}.".format(node.ip))
                        break
                except Exception as e:
                    self.ssh_client.ssh_close(ssh_client)
                    logger.error("Execute error:{}".format(str(e)))
                    return Message(500, FCUException(645018))
            if retry_times >= RETRY_TIME_LIMIT:
                self.ssh_client.ssh_close(ssh_client)
                logger.error("Update MO certs failed, after retry 3 times.")
                return Message(500, FCUException(645018))
        return Message(200)

    def _upload_mo_apimlb_certs(self, node, karbor_ssh_client):
        """Upload mo apimlb certs to karbor node.

        :return:
        """
        mo_certs = self.mo_cert_manager.get_mo_apimlb_certs()
        ssh_client = Ssh()
        rm_cert_cmd = "rm -f /home/djmanager/trust_apimlb_*"
        result = \
            self.ssh_client.ssh_exec_command_return(karbor_ssh_client,
                                                    rm_cert_cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception(
                "Failed delete old certs:{}.".format(rm_cert_cmd))

        for key in mo_certs.keys():
            mo_cert = mo_certs.get(key)
            result = ssh_client.put_file(node.ip, node.user, node.user_pwd,
                                         mo_cert, self.home_dir)
            if not result:
                raise Exception("Upload {} cert {} to karbor node {} "
                                "failed.".format(key, mo_cert, node.ip))
            else:
                logger.info("Upload {} cert {} to karbor "
                            "successfully.".format(key, mo_cert))

    def _update_mo_certs(self, ssh_client):
        """Update mo new certs in karbor.

        :return:
        """
        mo_certs_dir = "/opt/huawei/dj/DJSecurity/mo_certs"
        dir_cmd = "mkdir -p {}".format(mo_certs_dir)
        result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                         dir_cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            logger.error('Failed executing: {}.'.format(dir_cmd))
            return False
        cert_names_suffix = ["iam", "sc", "oc"]
        for name_suffix in cert_names_suffix:
            cert_cmd = "/usr/bin/cp -rf /home/djmanager/" \
                       "trust_apimlb_{name}.cer " \
                       "{dir}/trust_{name}.cer".format(name=name_suffix,
                                                       dir=mo_certs_dir)
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             cert_cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error('Failed executing: {}.'.format(cert_cmd))
                return False
        chmod_mo_certs_dir = "chmod 750 {}".format(mo_certs_dir)
        chmod_mo_certs_subdir = "chmod 600 {}/*".format(mo_certs_dir)
        chown_mo_certs_dir = \
            "chown openstack:openstack -R {}".format(mo_certs_dir)
        for cmd in [chmod_mo_certs_dir, chmod_mo_certs_subdir,
                    chown_mo_certs_dir]:
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error('Failed executing: {}.'.format(cmd))
                return False
        return True
