import os

import utils.common.log as logger
from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.common.upgrade.config_files import ConfigFiles
from plugins.CSBS_VBS.common.upgrade.dmk import DMKClient
from plugins.CSBS_VBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS_VBS.common.upgrade.models import CBSNode
from plugins.CSBS_VBS.common.upgrade.params import ParamsStore
from plugins.CSBS_VBS.common.upgrade.params import ParamsTools
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

logger.init("CSBS-VBS")


class BaseSubJob(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        super(BaseSubJob, self).__init__(project_id, pod_id, regionid_list)

        self.params_store = ParamsStore(project_id, pod_id, regionid_list)
        self.dmk_params = self.params_store.dmk_params
        self.karbor_params = self.params_store.karbor_params
        self.project_scene = self.params_store.project_scene
        self.upgrade_path = self.params_store.upgrade_path

    def pre_check(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform resource pre-check before installation,
        this interface is called by the execute interface,
        The tool framework does not directly call this interface.
        """
        return Message(200)

    def execute(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        return Message(200)

    def rollback(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform job job failed rollback.
        """
        return Message(200)

    def retry(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform job failed retry.
        """
        self.rollback(engine_id, pod_id, regionid_list)
        return self.execute(engine_id, pod_id, regionid_list)

    def check(self, engine_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Check before task execution.
        """
        return Message(200)


class KarborUpgradeSubJobBase(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(KarborUpgradeSubJobBase, self).__init__(project_id, pod_id,
                                                      regionid_list)
        self.karbor_operation = KarborOperation(self.params_store)
        self.db_params_dict = self.params_store.db_params_dict
        self.db_params_dict['check_karbor'] = 'false'
        self.db_params_dict['karbor_version'] = \
            self.params_store.upgrade_path.karbor_pkg_version
        self.dmk_client = DMKClient(self.params_store)
        config = ConfigFiles(self.params_store, self.db_params_dict)
        self.dmk_config_content = config.get_karbor_config_files()
        self.ssh_client = SshClient()

    def upload_karbor_pkg2dmk(self):
        pkg = os.path.join(self.upgrade_path.pkg_path,
                           self.upgrade_path.karbor_pkg_name)
        self.dmk_client.upload_pkg_to_dmk_for_upgrade(pkg, self.pod_id)

    def upload_karbor_pkg2vm(self):
        action = "[Upload] CSBS-VBS Karbor Package"
        result = self._execute_dmk_action(
            action,
            self.upgrade_path.karbor_pkg_version)
        return result

    def check_karbor(self):
        action = "[Check] CSBS-VBS Karbor"
        result = self._execute_dmk_action(
            action,
            self.upgrade_path.karbor_pkg_version
        )
        return result

    def upgrade_karbor(self):
        action = "[Upgrade] CSBS-VBS Karbor"
        result = self._execute_dmk_action(
            action,
            self.upgrade_path.karbor_pkg_version)
        return result

    def rollback_karbor(self):
        action = "[Rollback] CSBS-VBS Karbor"
        result = self._execute_dmk_action(
            action,
            self.upgrade_path.karbor_pkg_version)
        return result

    def _execute_dmk_action(self, action, version):
        result = self.dmk_client.execute_dmk_deploy(
            app_name="Karbor",
            app_version=version,
            deploy_action=action,
            hosts_content=self.dmk_config_content["host"],
            vars_content=self.dmk_config_content["config"],
            user="djmanager"
        )
        return result


class ConsoleSoloUpgradeSubJobBase(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(ConsoleSoloUpgradeSubJobBase, self).__init__(project_id, pod_id,
                                                           regionid_list)
        self.params_tools = ParamsTools(project_id)
        self.ssh_client = SshClient()
        self.dmk_client = DMKClient(self.params_store)
        self.db_params_dict = self.params_store.db_params_dict
        self.console_param_dict = self.set_config_dic()
        config = ConfigFiles(self.params_store, self.console_param_dict)
        self.dmk_config_content = config.get_console_config_files()

    def upload_console_pkg2dmk(self):
        pkg = os.path.join(self.upgrade_path.pkg_path,
                           self.upgrade_path.console_pkg_name)
        self.dmk_client.upload_pkg_to_dmk_for_upgrade(pkg, self.pod_id)

    def check_console(self):
        action = "8 Service Check"
        result = self._execute_dmk_action(
            action, self.upgrade_path.console_pkg_version)
        return result

    def upload_console_pkg2vm(self):
        action = "2 Upload Package"
        result = self._execute_dmk_action(
            action, self.upgrade_path.console_pkg_version)
        return result

    def upgrade_console(self):
        action = "7 Upgrade console"
        result = self._execute_dmk_action(action,
                                          self.upgrade_path.target_version)
        return result

    def rollback_console(self):
        action = "7 Upgrade console"
        result = self._execute_dmk_action(action,
                                          self.upgrade_path.original_version)
        return result

    def _execute_dmk_action(self, action, version):
        result = self.dmk_client.execute_dmk_deploy(
            app_name="CBS-Console",
            app_version=version,
            deploy_action=action,
            hosts_content=self.dmk_config_content["host"],
            vars_content=self.dmk_config_content["config"],
            user="cbs_admin"
        )
        return result

    def backup_cert(self):
        console_cbs_nodes = self._get_console_cbs_nodes()
        for cbs_node in console_cbs_nodes:
            cmd = "ll /home/iamKeyStore.jks"
            ret = self._ex_cmd_by_karbor_client(cbs_node, cmd)
            if not ret:
                logger.info("there is no iam cert in {} "
                            "home pass.".format(cbs_node.ip))
                cmd = "cp -pf /opt/huawei/console/tomcat/webapps/" \
                      "cbs/WEB-INF/classes/config/iamKeyStore.jks /home"
                ret = self._ex_cmd_by_karbor_client(cbs_node, cmd)
                if not ret:
                    raise Exception("Failed to backup console {} "
                                    "iam cert.".format(cbs_node.ip))
                logger.info("Backup console {} iam cert "
                            "successfully.".format(cbs_node.ip))

    def replace_cert(self):
        console_cbs_nodes = self._get_console_cbs_nodes()
        for cbs_node in console_cbs_nodes:
            cmd = "rm -f /opt/huawei/console/tomcat/webapps/" \
                  "cbs/WEB-INF/classes/config/iamKeyStore.jks"
            ret = self._ex_cmd_by_karbor_client(cbs_node, cmd)
            if not ret:
                raise Exception("Failed to delete {} "
                                "iam cert.".format(cbs_node.ip))

            cmd = "cp -pf /home/iamKeyStore.jks /opt/huawei/console/" \
                  "tomcat/webapps/cbs/WEB-INF/classes/config/"
            ret = self._ex_cmd_by_karbor_client(cbs_node, cmd)
            if not ret:
                raise Exception("Failed to replace console {} "
                                "iam cert.".format(cbs_node.ip))
            logger.info("Replace console {} iam cert "
                        "successfully.".format(cbs_node.ip))

    def _get_console_cbs_nodes(self):
        ips = [self.console_param_dict.get("console_host0"),
               self.console_param_dict.get("console_host1")]
        cbs_pwd = self.db_params_dict.get("console_cbs_admin_pwd")
        root_pwd = self.db_params_dict.get("console_root_pwd")

        result = []
        for ip in ips:
            cbs_node = CBSNode(ip=ip, user="cbs_admin", user_pwd=cbs_pwd,
                               root_pwd=root_pwd, extra="")
            result.append(cbs_node)
        return result

    def _ex_cmd_by_karbor_client(self, cbs_node, cmd):
        ssh_client = \
            self.ssh_client.get_sshclient_user_su_root_console(cbs_node)
        result = \
            self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
        ret = self.ssh_client.is_ssh_cmd_executed(result)
        self.ssh_client.ssh_close(ssh_client)
        return ret

    def set_config_dic(self):
        console_param_dict = dict()
        for key, value in self.params_tools.get_console_vm_ips().items():
            console_param_dict[key] = value
        console_param_dict['console_version'] = \
            self.upgrade_path.console_pkg_version
        console_param_dict['cpu_arch'] = \
            'x86' if self.project_scene.is_x86_scene else 'arm'
        return console_param_dict
