# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.
import utils.common.log as logger
from utils.common.fic_base import TestCase
from utils.business.project_condition_utils import get_project_condition_boolean
from plugins.DistributedStorage.logic.deploy_operate import DeployOperate
from plugins.DistributedStorage.utils.common.deploy_constant import DeployConstant


class CreateClient(TestCase):
    def __init__(self, project_id, pod_id, fs_args, **kwargs):
        super(CreateClient, self).__init__(project_id, pod_id)
        self.fs_args = fs_args
        self.bmc_info_lists = self.db.get_install_os_list_info(self.pod_id)
        self.fsa_list = self.fs_args.get('fsa_list')
        self.opr = DeployOperate(self.fs_args)
        self.more_args = kwargs
        self.update_pwd = self.fs_args['dm_update_pwd']
        self.iscsi_condition = get_project_condition_boolean(self.project_id, 'TenantStorFB_iSCSI')

    def procedure(self):
        logger.info("Start to create client.")
        self.login_dm()
        self.check_cluster()
        self.check_pool()
        if not self.fsa_list:
            err_msg = "fsa list is none"
            logger.error(err_msg)
            return
        logger.info("Check whether VBS is installed.")
        fsa_ip_list = [fsa.get('om_ip') for fsa in self.fsa_list]
        self.get_fsa_ip_list(fsa_ip_list)
        self.create_client(fsa_ip_list)
        self.config_iscsi_port()
        logger.info('Create VBS client success')
        self.opr.logout()

    def get_fsa_ip_list(self, fsa_ip_list):
        rsp_obj = self.opr.query_cluster_servers()
        rsp_code, rsp_result, rsp_data = rsp_obj.get_rsp_data()
        error_code = rsp_result.get("code")
        if rsp_code != 0 or error_code != 0:
            error_des = rsp_result.get('description')
            error_sgt = rsp_result.get('suggestion')
            err_msg = "Failed to query the node to install vbs, Detail:[%s]%s.Suggestion:%s" % (
                error_code, error_des, error_sgt)
            logger.error(err_msg)
            raise Exception(err_msg)
        for server in rsp_data:
            ip = server.get('management_ip')
            if ip in fsa_ip_list and "vbs" in server.get('usage'):
                fsa_ip_list.remove(server.get('management_ip'))
                logger.info("VBS has been installed on node[%s]." % ip)

    def login_dm(self):
        """
        登陆存储
        :return:
        """
        status_code, error_code, error_des = self.opr.login(DeployConstant.DM_LOGIN_USER, self.update_pwd)
        if status_code != 200 or error_code != 0:
            err_msg = "Failed to login deploy manager, Detail:[status:%s,code:%s]%s" % (status_code, error_code,
                                                                                        error_des)
            logger.error(err_msg)
            raise Exception(err_msg)

    def check_cluster(self):
        """
        检查集群
        :return:
        """
        logger.info('Check cluster...')
        res_cluster = self.opr.query_manage_cluster()
        cluster = res_cluster.get_query_data()
        if not cluster.get('clusterName'):
            logger.error('Check cluster fail...')
            raise Exception("Check cluster fail...")
        else:
            logger.error('Check cluster successfully...')

    def check_pool(self):
        """
        检查存储池
        :return:
        """
        logger.info('Check pool...')
        res_pool = self.opr.query_storage_pool()
        pool = res_pool.get_query_data()
        if len(pool.get('storagePools')) == 0:
            logger.error('Check pool fail...')
            raise Exception("Check pool fail...")
        else:
            logger.info('Check pool successfully...')

    def create_client(self, fsa_ip_list):
        """
        串讲客户端
        :param fsa_ip_list:
        :return:
        """
        if fsa_ip_list:
            logger.info('create VBS client.')
            res_client = self.opr.create_client(fsa_ip_list)
            status_code, result, error_code, error_des = res_client.get_create_client_code()
            if status_code != 200 or result != 0:
                self.get_fsa_ip_list(fsa_ip_list)
                err_msg = "Failed to Create VBS client, Detail:[status:%s,result:%s,error:%s, ip:%s]%s" % \
                          (status_code, result, error_code, fsa_ip_list, error_des)
                logger.error(err_msg)
                raise Exception(err_msg)

    def config_iscsi_switch(self, fsa_manage_ip):
        result = self.opr.config_iscsi_switch(fsa_manage_ip)
        succ_ip = result.get_iscsi_result()
        if not succ_ip:
            err_msg = "Failed to config iscsi switch, fsa ip=%s, result=%s" % (fsa_manage_ip,
                                                                               result.res.json())
            logger.error(err_msg)
            raise Exception(err_msg)

    def config_iscsi_port(self):
        """
        配置iscsi网口
        :return:
        """
        if not self.iscsi_condition:
            return
        iscsi_configed_count = 0
        for fsa in self.fsa_list:
            fsa_manage_ip = fsa.get('om_ip')
            result = self.opr.query_network_serive(fsa_manage_ip)
            fsa_storage_ip = result.get_storage_ip()
            if not fsa_storage_ip:
                logger.error("fsa_ip= %s, storage ip is null. result=%s" % (fsa_manage_ip, result.res.json()))
                continue
            self.config_iscsi_switch(fsa_manage_ip)
            result = self.opr.add_iscsi_portal(fsa_manage_ip, fsa_storage_ip)
            succ_ip = result.get_iscsi_result()
            if not succ_ip:
                error_code = result.res.json().get('detail')[0].get('errorCode')
                if error_code == 33760258:
                    iscsi_configed_count += 1
                    logger.info("The specified iSCSI IP address already exists on node[%s]." % fsa_manage_ip)
                    continue
                err_msg = "Failed to add iscsi portal, fsa ip=%s, result=%s" % (fsa_manage_ip, result.res.json())
                logger.error(err_msg)
                raise Exception(err_msg)

            iscsi_configed_count += 1
        if 0 == iscsi_configed_count:
            err_msg = "Failed to add iscsi portal for fsa node, count = 0"
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info('Config iscsi service success, fsa count=%d' % iscsi_configed_count)
