# coding=utf-8
# Copyright (c) 2021 Huawei Technologies Co., Ltd.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from manila import exception
from manila.i18n import _
from oslo_log import log
from oslo_serialization import jsonutils

from . import constants, send_request

LOG = log.getLogger(__name__)


class RestHelper:
    """Helper class for Huawei OceanStorPacific storage system."""

    def __init__(self, root):
        self.send_request = send_request.SendRequest(root)
        self.call = self.send_request.call

    def log_in_pacific(self):
        self.call()

    def query_pool_by_id(self, pool_id):
        """This interface is used to query storage pools in a batch."""

        url = "data_service/storagepool?storagePoolId={0}".format(pool_id)
        result = self.call(url, None, "GET")

        if result['result'] == 0 and result["storagePools"]:
            LOG.debug("Query storage pool success.(pool_id: {0}) ".format(pool_id))
        else:
            err_msg = "Query storage pool failed.(pool_id: {0})".format(pool_id)
            raise exception.InvalidShare(reason=err_msg)

        return result['storagePools'][0]

    def query_account_by_name(self, account_name):
        """This interface is used to query an account."""

        url = "account/accounts"
        query_para = {
            'name': account_name
        }
        data = jsonutils.dumps(query_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0 and result["data"]:
            LOG.info(_("Query account name success.(account_name: {0})".format(account_name)))
        elif result['result']['code'] == constants.ACCOUNT_NOT_EXIST and not result['data']:
            LOG.info(_("Query account name does not exist.(account_name: {0})".format(account_name)))
        else:
            err_msg = _("Query account name({0}) failed".format(account_name))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def create_account(self, account_name):
        """This interface is used to create an account."""

        url = "account/accounts"
        account_para = {
            'name': account_name
        }
        data = jsonutils.dumps(account_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Create account success.(account_name: {0})".format(account_name)))
        else:
            err_msg = _("Create account failed.(account_name: {0})".format(account_name))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def delete_account(self, account_id):
        """This interface is used to delete an account."""

        url = "account/accounts"
        account_para = {
            'id': account_id
        }
        data = jsonutils.dumps(account_para)
        result = self.call(url, data, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete account success.(account_id: {0})".format(account_id)))
        else:
            err_msg = (_("Delete account failed.(account_id: {0})".format(account_id)))
            raise exception.InvalidShare(reason=err_msg)

    def query_access_zone_count(self, account_id):

        url = "eds_dns_service/zone_count?account_id={0}".format(account_id)
        result = self.call(url, None, "GET")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Query account access zone success.(account_id: {0})".format(account_id)))
        else:
            err_msg = _("Query account access zone failed.(account_id: {0})".format(account_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def query_namespaces_count(self, account_id):
        """This interface is used to query the number of configured namespaces."""

        url = "converged_service/namespaces_count"
        query_para = {
            'filter': {'account_id': account_id}
        }
        data = jsonutils.dumps(query_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Query namespace quantity of account success.(account_id :{0})".format(account_id)))
        else:
            err_msg = _("Query namespace quantity of account failed.(account_id :{0})".format(account_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def query_namespace_by_name(self, namespace_name):
        """Query the configurations of a namespace based on its name"""

        url = "converged_service/namespaces"
        query_para = {
            'name': namespace_name
        }
        data = jsonutils.dumps(query_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Query namespace success.(namespace_name: {0})".format(namespace_name)))
        elif result['result']['code'] == constants.NAMESPACE_NOT_EXIST and not result['data']:
            LOG.info(_("Query namespace does not exist.(namespace_name: {0})".format(namespace_name)))
        else:
            err_msg = _("Query namespace({0}) failed".format(namespace_name))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def create_namespace(self, namespace_name, storage_pool_id, account_id, forbidden_dpc, atime_mode):
        """This interface is used to create a namespace."""

        url = "converged_service/namespaces"
        namespace_para = {
            'name': namespace_name,
            'storage_pool_id': storage_pool_id,
            'account_id': account_id,
            'audit_log_rule': 0,
            'forbidden_dpc': forbidden_dpc,
            'atime_update_mode': atime_mode
        }
        data = jsonutils.dumps(namespace_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Create namespace success.(namespace_name {0})".format(namespace_name)))
        else:
            err_msg = _("Create namespace failed.(namespace_name {0})".format(namespace_name))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def delete_namespace(self, namespace_name):
        """This interface is used to delete a namespace based on its name."""

        url = "converged_service/namespaces"
        namespace_para = {
            'name': namespace_name
        }
        data = jsonutils.dumps(namespace_para)
        result = self.call(url, data, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete namespace success.(namespace_name: {0})".format(namespace_name)))
        elif result['result']['code'] == constants.NAMESPACE_NOT_EXIST:
            LOG.info(_("Delete namespace does not exist.(namespace_name: {0})".format(namespace_name)))
        else:
            err_msg = (_("Delete namespace({0}) failed.".format(namespace_name)))
            raise exception.InvalidShare(reason=err_msg)

    def query_quota_by_parent(self, namespace_id):
        """This interface is used to query namespace quotas in batches."""

        url = "converged_service/quota"
        query_para = {
            "parent_type": constants.QUOTA_PARENT_TYPE_NAMESPACE,
            "parent_id": namespace_id,
            "space_unit_type": constants.QUOTA_UNIT_TYPE_GB,
            "range": "{\"offset\": 0, \"limit\": 10}"
        }
        data = jsonutils.dumps(query_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Query quota success.(namespace_id: {0})".format(namespace_id)))
        else:
            err_msg = _("Query quota  failed.(namespace_id: {0})".format(namespace_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data'][0]

    def creat_quota(self, namespace_id, quota_size):
        """This interface is used to create a namespace quota."""

        url = "converged_service/quota"
        quota_para = {
            "parent_type": constants.QUOTA_PARENT_TYPE_NAMESPACE,
            "parent_id": namespace_id,
            "quota_type": constants.QUOTA_TYPE_DIRECTORY,
            "space_hard_quota": quota_size,
            "space_unit_type": constants.QUOTA_UNIT_TYPE_GB,
            "directory_quota_target": constants.QUOTA_TARGET_NAMESPACE
        }
        data = jsonutils.dumps(quota_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Create quote success. (quota_size: {0}GB)".format(quota_size)))
        else:
            err_msg = _("Create quote failed.")
            raise exception.InvalidShare(reason=err_msg)

    def change_quota_size(self, quota_id, new_size):
        """This interface is used to modify a namespace quota."""

        url = "converged_service/quota"
        quota_para = {
            "id": quota_id,
            "space_hard_quota": new_size,
            "space_unit_type": constants.QUOTA_UNIT_TYPE_GB
        }
        data = jsonutils.dumps(quota_para)
        result = self.call(url, data, "PUT")

        if result['result']['code'] == 0:
            LOG.info(_("Change quota success.(quota_size: {0}GB)".format(new_size)))
        else:
            err_msg = _("Change quota failed")
            raise exception.InvalidShare(reason=err_msg)

    def create_qos(self, qos_name, account_id, qos_config):
        """This interface is used to create a converged QoS policy."""

        url = "dros_service/converged_qos_policy"
        qos_para = {
            'name': qos_name,
            'qos_mode': constants.QOS_MODE_PACKAGE,
            'qos_scale': constants.QOS_SCALE_NAMESPACE,
            'account_id': account_id,
            'package_size': 10,
            'max_band_width': qos_config['max_band_width'],
            'basic_band_width': qos_config['basic_band_width'],
            'bps_density': qos_config['bps_density'],
            'max_iops': qos_config['max_iops']
        }
        data = jsonutils.dumps(qos_para)
        result = self.call(url, data, "POST")

        if result["result"]["code"] == 0 and result['data']:
            LOG.info(_("Create qos success.(qos_name: {0})".format(qos_name)))
        else:
            err_msg = _("Create qos failed.(qos_name: {0})".format(qos_name))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def add_qos_association(self, namespace_name, qos_policy_id, account_id):
        """This interface is used to add a converged QoS policy association."""

        url = "dros_service/converged_qos_association"
        qos_asso_para = {
            'qos_scale': constants.QOS_SCALE_NAMESPACE,
            'object_name': namespace_name,
            'qos_policy_id': qos_policy_id,
            'account_id': account_id
        }
        data = jsonutils.dumps(qos_asso_para)
        result = self.call(url, data, "POST")

        if result["result"]["code"] == 0:
            LOG.info(_("Add a QoS policy association success."))
        else:
            err_msg = _("Add a QoS policy association failed.")
            raise exception.InvalidShare(reason=err_msg)

    def delete_qos(self, qos_name):
        """This interface is used to delete a converged QoS policy."""

        url = "dros_service/converged_qos_policy"
        qos_para = {
            'name': qos_name,
            'qos_scale': 0
        }
        data = jsonutils.dumps(qos_para)
        result = self.call(url, data, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete the qos success.(qos_name: {0})".format(qos_name)))
        elif result['result']['code'] == constants.QOS_NOT_EXIST:
            LOG.info(_("Delete the qos does not exist.(qos_name: {0})".format(qos_name)))
        else:
            err_msg = "Delete the qos failed.(qos_name: {0})".format(qos_name)
            raise exception.InvalidShare(reason=err_msg)

    def add_tier_policy(self, tier_name, namespace_id, strategy, mtime_flag, mtime):
        """This interface is used to create the migration policy."""

        url = "tier_service/tier_migrate_policies"
        policy_para = {
            'name': tier_name if mtime_flag else tier_name + "_tohot",
            'fs_id': namespace_id,
            'strategy': strategy if mtime_flag else 0,
            'migration_type': constants.PERIODIC_MIGRATION_POLICY,
            'atime_operator': constants.MATCH_RULE_GT if mtime_flag else constants.MATCH_RULE_LT,
            'atime': mtime if mtime_flag else 4,
            'atime_unit': constants.DTIME_UNIT if mtime_flag else constants.HTIME_UNIT
        }

        if mtime_flag:
            policy_para['mtime_operator'] = constants.MATCH_RULE_GT
            policy_para['mtime'] = mtime
            policy_para['mtime_unit'] = constants.DTIME_UNIT

        data = jsonutils.dumps(policy_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0:
            LOG.info(_("Create tier migrate policy success.(tier_name: {0})".format(policy_para['name'])))
        else:
            err_msg = _("Create tier migrate policy failed.(tier_name: {0})".format(policy_para['name']))
            raise exception.InvalidShare(reason=err_msg)

    def add_tier_migrate_schedule(self, namespace_id):
        """This interface is used to modify execution interval of SmartTier migration policy."""

        url = "tier_service/tier_migrate_schedule"
        tier_para = {
            'fs_id': namespace_id,
            'path_name': '/',
            'migration_schedule_type': 0,
            'migration_day_operate': 0,
            'migration_start_hour': '0,4,8,12,16,20',
            'migration_start_minute': 0,
        }

        data = jsonutils.dumps(tier_para)
        result = self.call(url, data, "PUT")

        if result['result']['code'] == 0:
            LOG.info(_("Set tier migrate schedule success."))
        else:
            err_msg = _("Set tier migrate schedule failed.")
            raise exception.InvalidShare(reason=err_msg)

    def create_nfs_share(self, namespace_name, account_id):
        """This interface is used to create an NFS share."""

        url = "nas_protocol/nfs_share"
        nfs_para = {
            'share_path': '/' + namespace_name,
            'account_id': account_id
        }
        data = jsonutils.dumps(nfs_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Create NFS share success.(namespace_name: {0})".format(namespace_name)))
        else:
            err_msg = _("Create NFS share failed.(namespace_name: {0})".format(namespace_name))
            raise exception.InvalidShare(reason=err_msg)

    def create_cifs_share(self, namespace_name, account_id):
        """This interface is used to create a CIFS share."""

        url = "file_service/cifs_share"
        cifs_param = {
            "name": namespace_name,
            "share_path": '/' + namespace_name,
            "account_id": account_id,
        }

        data = jsonutils.dumps(cifs_param)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Create CIFS share success.(namespace_name: {0})".format(namespace_name)))
        else:
            err_msg = _("Create CIFS share failed.(namespace_name: {0})".format(namespace_name))
            raise exception.InvalidShare(reason=err_msg)

    def query_nfs_share_information(self, account_id):
        """This interface is used to batch query NFS share information."""

        url = "nas_protocol/nfs_share_list"
        nfs_para = {
            'account_id': account_id
        }
        data = jsonutils.dumps(nfs_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Query NFS share success.(account_id: {0})".format(account_id)))
        else:
            err_msg = _("Query NFS share failed.(account_id: {0})".format(account_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def query_cifs_share_information(self, account_id):
        """This interface is used to batch query basic information about CIFS shares."""

        url = "file_service/cifs_share_list"

        cifs_para = {
            'account_id': account_id
        }

        data = jsonutils.dumps(cifs_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0 and result['data']:
            LOG.info(_("Query CIFS share success.(account_id: {0})".format(account_id)))
        else:
            err_msg = _("Query CIFS share failed.(account_id: {0})".format(account_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def delete_nfs_share(self, nfs_share_id, account_id):
        """This interface is used to delete an NFS share."""

        url = "nas_protocol/nfs_share"
        nfs_para = {
            'id': nfs_share_id,
            'account_id': account_id
        }
        data = jsonutils.dumps(nfs_para)
        result = self.call(url, data, "DELETE")
        if result['result']['code'] == 0:
            LOG.info(_("Delete the NFS share success.(nfs_share_id: {0})".format(nfs_share_id)))
        else:
            err_msg = "Delete the NFS share failed.(nfs_share_id: {0})".format(nfs_share_id)
            raise exception.InvalidShare(reason=err_msg)

    def delete_cifs_share(self, cifs_share_id, account_id):
        """This interface is used to delete a CIFS share."""

        url = "file_service/cifs_share"
        cifs_para = {
            "id": cifs_share_id,
            "account_id": account_id
        }
        data = jsonutils.dumps(cifs_para)
        result = self.call(url, data, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete the CIFS share success.(cifs_share_id: {0})".format(cifs_share_id)))
        else:
            err_msg = "Delete the CIFS share failed.(cifs_share_id: {0})".format(cifs_share_id)
            raise exception.InvalidShare(reason=err_msg)

    def query_users_by_id(self, account_id):
        """This interface is used to query basic information about a UNIX user."""

        url = "nas_protocol/unix_user?account_id={0}".format(account_id)
        result = self.call(url, None, "GET")

        if result['result']['code'] == 0:
            LOG.info(_("Query users success.(account_id: {0})".format(account_id)))
        else:
            err_msg = _("Query users failed.(account_id: {0})".format(account_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def query_user_groups_by_id(self, account_id):
        """This interface is used to query basic information about a UNIX user group."""

        url = "nas_protocol/unix_group?account_id={0}".format(account_id)
        result = self.call(url, None, "GET")

        if result['result']['code'] == 0:
            LOG.info(_("Query users groups success.(account_id: {0})".format(account_id)))
        else:
            err_msg = _("Query users groups failed.(account_id: {0})".format(account_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def delete_unix_user(self, user_name, account_id):
        """This interface is used to delete a UNIX user."""

        url = "nas_protocol/unix_user?name={0}&account_id={1}".format(user_name, account_id)
        result = self.call(url, None, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete the user success.(user_name: {0})".format(user_name)))
        else:
            err_msg = _("Delete the user failed.(user_name: {0})".format(user_name))
            raise exception.InvalidShare(reason=err_msg)

    def delete_unix_user_group(self, group_name, account_id):
        """This interface is used to delete a UNIX user group."""

        url = "nas_protocol/unix_group?name={0}&account_id={1}".format(group_name, account_id)
        result = self.call(url, None, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete the user group success.(group_name: {0})".format(group_name)))
        else:
            err_msg = _("Delete the user group failed.(group_name: {0})".format(group_name))
            raise exception.InvalidShare(reason=err_msg)

    def allow_access_for_nfs(self, share_id, access_to, access_level, account_id):
        """This interface is used to add an NFS share client."""

        url = "nas_protocol/nfs_share_auth_client"
        access_para = {
            'access_name': access_to,
            'share_id': share_id,
            'access_value': 0 if access_level == 'ro' else 1,
            'sync': 1,
            'all_squash': 1,
            'root_squash': 0,
            'account_id': account_id,
        }

        data = jsonutils.dumps(access_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0:
            LOG.info(_("Add an NFS share client success.(access_to: {0})".format(access_to)))
        elif result['result']['code'] == constants.NFS_SHARE_CLIENT_EXIST:
            LOG.info(_("Add an NFS share client already exist.(access_to: {0})".format(access_to)))
        else:
            err_msg = _("Add an NFS shared client for share failed.(access_to: {0})".format(share_id))
            raise exception.InvalidShare(reason=err_msg)

    def allow_access_for_cifs(self, share_id, access_to, access_level, account_id):
        """This interface is used to add a CIFS share user or user group."""

        url = "file_service/cifs_share_auth_client"
        query_para = {
            "share_id": share_id,
            "name": access_to,
            "domain_type": 2,
            "permission": 0 if access_level == 'ro' else 1,
            "account_id": account_id
        }
        data = jsonutils.dumps(query_para)
        result = self.call(url, data, "POST")

        if result['result']['code'] == 0:
            LOG.info(_("Add an CIFS share user success.(access_to: {0})".format(access_to)))
        elif result['result']['code'] == constants.CIFS_SHARE_CLIENT_EXIST:
            LOG.info(_("Add an CIFS share user({0}) already exist.(access_to: {0})".format(access_to)))
        else:
            err_msg = _("Add an CIFS shared client for share failed.(access_to: {0})".format(share_id))
            raise exception.InvalidShare(reason=err_msg)

    def query_nfs_share_clients_information(self, share_id, account_id=None):
        """This interface is used to batch query NFS share client information."""

        url = "nas_protocol/nfs_share_auth_client_list"
        filter_para = {
            'filter': '[{{"share_id": "{0}"}}]'.format(share_id),
            'account_id': account_id
        }

        data = jsonutils.dumps(filter_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0:
            LOG.info(_("Query NFS share clients success.(nfs_share_id: {0})".format(share_id)))
        else:
            err_msg = _("Query NFS share clients failed.(nfs_share_id: {0})".format(share_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def query_cifs_share_user_information(self, share_id, account_id=None):
        """This interface is used to query CIFS share users or user groups in batches."""

        url = "file_service/cifs_share_auth_client_list"
        filter_para = {
            'filter': '[{{"share_id": "{0}"}}]'.format(share_id),
            'account_id': account_id
        }

        data = jsonutils.dumps(filter_para)
        result = self.call(url, data, "GET")

        if result['result']['code'] == 0:
            LOG.info(_("Query CIFS share user success.(cifs_share_id: {0})".format(share_id)))
        else:
            err_msg = _("Query CIFS share user failed.(cifs_share_id: {0})".format(share_id))
            raise exception.InvalidShare(reason=err_msg)

        return result['data']

    def deny_access_for_nfs(self, client_id, account_id):
        """This interface is used to delete an NFS share client."""

        url = "nas_protocol/nfs_share_auth_client"

        nfs_para = {
            'id': client_id,
            'account_id': account_id
        }
        data = jsonutils.dumps(nfs_para)
        result = self.call(url, data, "DELETE")
        if result['result']['code'] == 0:
            LOG.info(_("Delete the NFS client success.(client_id: {0})".format(client_id)))
        else:
            err_msg = "Delete the NFS client failed.(client_id: {0})".format(client_id)
            raise exception.InvalidShare(reason=err_msg)

    def deny_access_for_cifs(self, user_id, account_id):
        """This interface is used to delete a CIFS share user or user group."""

        url = "file_service/cifs_share_auth_client"
        query_para = {
            "id": user_id,
            "account_id": account_id
        }
        data = jsonutils.dumps(query_para)
        result = self.call(url, data, "DELETE")

        if result['result']['code'] == 0:
            LOG.info(_("Delete the CIFS client success.(user_id: {0})".format(user_id)))
        else:
            err_msg = "Delete the CIFS client failed.(user_id: {0})".format(user_id)
            raise exception.InvalidShare(reason=err_msg)
