#!/bin/bash

CUR_DIR=$(dirname $(readlink -f $0))

if [ ! -f /opt/dfv/oam/oam-u/ha/ha.conf ];then
  echo "RestConvert must install at FSM"
  exit 0
fi
source ${CUR_DIR}/bin/commonUtil.sh

LOG_NAME="install.log"
RESTCONVERT_SCRIPT_LOG_PATH="/var/log/restconvert/scriptLog"
LOG_FILE="${RESTCONVERT_SCRIPT_LOG_PATH}/${LOG_NAME}"

INSTALL_PATH=/opt/dfv/oam/rest
NEED_DISK_SPACE=512000
initLog "${RESTCONVERT_SCRIPT_LOG_PATH}" "${LOG_NAME}"

chown -Rh root:root ${RESTCONVERT_SCRIPT_LOG_PATH}

scriptName=$(basename $0)

CLI_PATH=${CUR_DIR}/config/cli

#当前暂时用oam用户属组
function addRestUser()
{
    addUser_Group ${OAM_SYS_USER} ${USER_OAM_ID} ${OAM_SYS_GROUP} ${GROUP_OAM_ID}
    if [ $? -ne 0 ]; then
        log_error "addUser_Group failed!"
        return 1
    fi

    usermod -a -G "${COMMON_GROUP}" oam 2>/dev/null
    ret=$?
    log_warn "usermod result is ${ret}."

    chage ${OAM_SYS_USER} -m -1 -M -1 -W -1 -E -1
    if [ $? -ne 0 ]; then
        log_warn "change ${OAM_SYS_USER} passwd policy failed!"
    fi

    chmod 700 /home/${OAM_SYS_USER}

    log_info "addRestUser success!"
}

#校验HA的IP，获得本节点的IP
function checkHAIp()
{
    isValidIP "${SERVICE_FLOAT_IP}"
    if [ $? -ne 0 ];then
        log_error "SERVICE_FLOAT_IP ${SERVICE_FLOAT_IP} is invalid ip."
        return 1
    fi

    # 校验主备IP地址是否合法
    local active_flag=1
    local standby_flag=1
    isValidIP "${ACTIVE_IP}"
    if [ $? -ne 0 ];then
        log_error "ACTIVE_IP ${ACTIVE_IP} is invalid ip."
        return 1
    fi
    ip addr show | fgrep -w "${ACTIVE_IP}" >/dev/null 2>&1
    active_flag=$?

    # 部署单个管理节点时，备节点IP地址为"--"或主节点IP，浮动IP地址等于管理节点固定IP地址
    if [[ x"${STANDBY_IP}" == x"--" ]] || [[ x"${STANDBY_IP}" == x"${ACTIVE_IP}" ]]; then
        if [ "${ACTIVE_IP}" != "${SERVICE_FLOAT_IP}" ]; then
            log_error "single installation active ip must equal to float ip."
            return 1
        fi
    else
        isValidIP "${STANDBY_IP}"
        if [ $? -ne 0 ];then
            log_error "STANDBY_IP ${STANDBY_IP} is invalid ip."
            return 1
        fi
        ip addr show | fgrep -w "${STANDBY_IP}" 2>>${LOG_FILE}
        standby_flag=$?
    fi

    if [[ ${active_flag} -eq 0 && ${standby_flag} -eq 0 ]]; then
        log_error "active_ip should not be equal with standby_ip."
        return 1
    elif [[ ${active_flag} -eq 0 && ${standby_flag} -ne 0 ]]; then
        log_info "current node is active"
    elif [[ ${active_flag} -ne 0 && ${standby_flag} -eq 0 ]]; then
        log_info "current node is standby"
    else
        log_error "one of active_ip and standby_ip should be localhost ip."
        return 1
    fi

    return 0
}

#将软件包文件复制到进程指定的安装目录
function installPackage()
{
  if [ -d ${INSTALL_PATH} ];then
    rm -rf ${INSTALL_PATH}
  fi
  log_info "create default install path"
  mkdir -p ${INSTALL_PATH}
  if [ $? -ne 0 ];then
    print_error "create defalut install path failed"
    return 1
  fi
  checkInstallPathSpace
  if [ $? -ne 0 ];then
    print_error "check install path space failed"
    return 1
  fi
  copyPackage
  if [ $? -ne 0 ];then
    print_error "copy package to default path failed"
    return 1
  fi
  return 0
}

#校验路径剩余空间是否足够
function checkInstallPathSpace()
{
  checkDiskSpace ${INSTALL_PATH} ${NEED_DISK_SPACE}
  return $?
}

function copyPackage()
{
  deleteInvalidJre
  if [ $? -ne 0 ]; then
       print_error "delete invalid jre failed"
       return 1
  fi
  deleteInvalidKmcLib
  if [ $? -ne 0 ]; then
       print_error "delete invalid kmclib failed"
       return 1
  fi
  setDefaultJarName
  if [ $? -ne 0 ]; then
       print_error "rename RestConvert jar failed"
       return 1
  fi

  local restName=$(basename ${CUR_DIR})
  dos2unix ${CUR_DIR}/*.sh >/dev/null 2>&1
  dos2unix ${CUR_DIR}/bin/*.sh >/dev/null 2>&1
  cp -rf ${CUR_DIR} ${INSTALL_PATH}
  mv -f ${INSTALL_PATH}/${restName} ${INSTALL_PATH}/RestConvert
  if [ $? -ne 0 ] && [ "X${restName}" != "XRestConvert" ]; then
       scriptExelog "copy package from ${CUR_DIR} failed. [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
       return 1
  fi
  mvKmcKey
  setInstallFilePermission

  setJre ${INSTALL_PATH}/RestConvert/jre
  if [ $? -ne 0 ]; then
       scriptExelog "set jre permission failed. [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
       return 1
  fi
  return 0
}

function mvKmcKey()
{
  mv ${INSTALL_PATH}/RestConvert/kmclib/primary_ks.key ${INSTALL_PATH}/RestConvert/config >/dev/null 2>&1
  mv ${INSTALL_PATH}/RestConvert/kmclib/standby_ks.key ${INSTALL_PATH}/RestConvert/config >/dev/null 2>&1
}

#将jar包改为默认名称
function setDefaultJarName()
{
  if [ ! -f "${CUR_DIR}"/RestConvert.jar ];then
    mv ${CUR_DIR}/RestConvert*.jar ${CUR_DIR}/RestConvert.jar
    return $?
  fi
  return 0
}

#调整安装文件的权限和属组
function setInstallFilePermission()
{
  is_link ${INSTALL_PATH}
  if [ $? -ne 0 ]; then
    scriptExelog "install path is link,failed. [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
    return 1
  fi

  chown -Rh ${OAM_SYS_USER}:${OAM_SYS_GROUP} ${INSTALL_PATH}

  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 750 ${INSTALL_PATH}"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod -R 500 ${INSTALL_PATH}/RestConvert"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 750 ${INSTALL_PATH}/RestConvert"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 750 ${INSTALL_PATH}/RestConvert/resources"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 750 ${INSTALL_PATH}/RestConvert/config"

  su - ${OAM_SYS_USER} -s "/bin/bash" -c "find ${INSTALL_PATH}/RestConvert/* -name "*.sh" -print0 | xargs -0 chmod 550"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "find ${INSTALL_PATH}/RestConvert/* -name "*.xml" -print0 | xargs -0 chmod 640"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "find ${INSTALL_PATH}/RestConvert/* -name "*.properties" -print0 | xargs -0 chmod 640"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "find ${INSTALL_PATH}/RestConvert/* -name "*.txt" -print0 | xargs -0 chmod 640"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "find ${INSTALL_PATH}/RestConvert/config -name "*.key" -print0 | xargs -0 chmod 640"

  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 640 ${INSTALL_PATH}/RestConvert/resources/application.properties"
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 640 ${INSTALL_PATH}/RestConvert/resources/restconvert.db"
}

#根据实际环境，删除多余的jre
function deleteInvalidJre()
{
  local type=`uname -m`
  if [[ x"${type}" == x"x86_64" ]];then
    rm -rf ${CUR_DIR}/jre/aarch64
    mv -f ${CUR_DIR}/jre/x86-64/* ${CUR_DIR}/jre >/dev/null 2>&1
    rm -rf ${CUR_DIR}/jre/x86-64
    return $?
  else
    rm -rf ${CUR_DIR}/jre/x86-64
    mv -f ${CUR_DIR}/jre/aarch64/* ${CUR_DIR}/jre >/dev/null 2>&1
    rm -rf ${CUR_DIR}/jre/aarch64
    return $?
  fi
}

#根据实际环境，删除多余的kmc库
function deleteInvalidKmcLib()
{
  local type=`uname -m`
  if [[ x"${type}" == x"x86_64" ]];then
    rm -rf ${CUR_DIR}/kmclib/aarch64
    mv -f ${CUR_DIR}/kmclib/x86-64/* ${CUR_DIR}/kmclib >/dev/null 2>&1
    rm -rf ${CUR_DIR}/kmclib/x86-64
    return $?
  else
    rm -rf ${CUR_DIR}/kmclib/x86-64
    mv -f ${CUR_DIR}/kmclib/aarch64/* ${CUR_DIR}/kmclib >/dev/null 2>&1
    rm -rf ${CUR_DIR}/kmclib/aarch64
    return $?
  fi
}

function changeLogPermission()
{
  is_link /var/log/restconvert
  if [ $? -ne 0 ]; then
       scriptExelog "log path is link,failed. [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
       return 1
  fi
  chown -Rh ${OAM_SYS_USER}:${OAM_SYS_GROUP} /var/log/restconvert
  su - ${OAM_SYS_USER} -s "/bin/bash" -c "chmod 750 /var/log/restconvert"
  return 0
}

#主机或者单机需要在安装后检查进程状态
function testProcess()
{
  getCurrentRole
  if [ $? -ne 2 ];then
    log_info "current role is not slave,need to check process status"
    testRestCovert
    return $?
  fi
  return 0
}

#日志收集注册
function registerLogCollection()
{
  if [ -f /opt/dfv/oam/public/script/collect_log/FusionStorage_oam-u/restConvert.sh ];then
    log_info "collect log script has exist"
    return 0
  fi
  if [ -f ${INSTALL_PATH}/RestConvert/bin/restConvert.sh ];then
    local targetPath=/opt/dfv/oam/public/script/collect_log/FusionStorage_oam-u
    mkdir -p ${targetPath}
    chmod 550 ${targetPath}
    chown -h root:oam ${targetPath}
    cp -f ${INSTALL_PATH}/RestConvert/bin/restConvert.sh ${targetPath}
    chmod 550 ${targetPath}/restConvert.sh
    chown -h root:oam ${targetPath}/restConvert.sh
    sh /opt/dfv/oam/oam-u/nma/scripts/non_root_scripts/refreshAllScripts.sh
    if [ $? -ne 0 ];then
      print_error "exec refreshAllScripts fail"
      return 1
    fi
    log_info "register collect log script success"
    return 0
  fi
  log_error "defalut install path not exist restConvert.sh"
  return 1
}

function addLogMonitorCrontab()
{
    cat "${CRONTAB_FILE_PATH}" | grep "restScriptLogMonitor.sh" >> /dev/null 2>&1
    if [ $? -eq 0 ];then
        delete_crontab_file "restScriptLogMonitor.sh"
        if [ $? -ne 0 ];then
            scriptExelog "Delete rest script log monitor in crontab failed [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
            return 1
        fi
    fi

    add_crontab_file "1 * * * * su - rest -s '/bin/bash' -c  'sh ${INSTALL_PATH}/RestConvert/bin/restScriptLogMonitor.sh >> /dev/null 2>&1'"
    if [ $? -ne 0 ];then
        scriptExelog "Add rest script log monitor to crontab failed [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
        return 1
    fi

    service crond reload >> ${LOG_FILE} 2>&1
    if [ $? -ne 0 ];then
        scriptExelog "service crond reload failed [Line:${LINENO}]" ${FUNCNAME} ${scriptName}
        return 1
    fi
    return 0
}

# 将CLI放到指定目录
function installRestCli()
{
  CLI_LIST=$(ls ${CLI_PATH})
  for cli in $CLI_LIST; do
    if [ -d ${CLI_INSTALL_PATH}/${cli} ];then
      log_warn "${cli} has exist in ${CLI_INSTALL_PATH}"
      continue
    fi
    cp -rf  ${CLI_PATH}/${cli} ${CLI_INSTALL_PATH}
    chown -R root:omm ${CLI_INSTALL_PATH}/${cli}
    chmod 550 ${CLI_INSTALL_PATH}/${cli}
    find ${CLI_INSTALL_PATH}/${cli} -name "*.xml" -print0 | xargs -0 chmod 440
    log_info "install cmd ${cli} success"
  done
}

function main()
{
  isRestCovertRunning
  if [ $? -eq 0 ];then
      print_error "restConvert is running,can not install again"
      return 1
  fi
  addRestUser
  if [ $? -ne 0 ];then
      print_error "add rest user failed!"
      return 1
  fi
  changeLogPermission
  if [ $? -ne 0 ];then
      print_error "change log permission failed!"
      return 1
  fi
  checkHAIp
  if [ $? -ne 0 ];then
      print_error "checkHAIp failed."
      return 1
  fi
  installPackage
  if [ $? -ne 0 ];then
      print_error "installPackage failed."
      return 1
  fi
  installRestCli
  if [ $? -ne 0 ];then
      print_error "installRestCli failed."
      return 1
  fi
  registerLogCollection
  if [ $? -ne 0 ];then
      print_error "registerLogCollection failed."
      return 1
  fi
  registerHAPlugin
  if [ $? -ne 0 ];then
      print_error "register HA failed."
      return 1
  fi
  addLogMonitorCrontab
  if [ $? -ne 0 ];then
      print_error "add crontab failed."
      return 1
  fi
  testProcess
  if [ $? -ne 0 ];then
      print_error "check process status failed."
      return 1
  fi
  return 0
}

main
ret=$?
[[ ${ret} -eq 0 ]] && print_info "restConvert installed successfully!" || print_error "restConvert installed failed!"
exit ${ret}
