# -*- coding:utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.

import traceback

import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import ParamUtil
import platforms.project.ProjectUtils as ProjectUtils
from utils.client.FSPAuthClient import FSPAuthClient
from utils.Driver.CloudDC.OpenStack.get_host_info import GetHostInfo
from utils.common.exception import FCDException
from plugins.DistributedStorage.scripts.utils.common.DeployConstant import DeployConstant


class Params(object):

    def __init__(self, project_id, pod_id, service_name):
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = service_name
        self.project_condition = ProjectUtils.get_project_conditions(self.project_id)
        self.db = BaseOps()

        # 所有参数 包含LLD中参数
        self.params_dict = {}
        # 复制节点信息
        self.replication_node_info = []
        # params.cfg配置参数
        self.local_params = {}

        self.param_util = ParamUtil()

    def get_admin_passwd_params(self):
        """
        部署工程、扩az新建部署复制场景：
            部署生产存储，LLD无参数，生产端、灾备端参数都取默认值，通过接口param_util.get_param_value获取参数结果：
                product_admin_password = None   disaster_admin_password = None
            部署灾备存储，LLD填生产端参数，生产端参数从LLD表获取，灾备参数取默认值，通过接口param_util.get_param_value获取参数结果：
                product_admin_password = pwd    disaster_admin_password = None

        新增业务存储--扩复制集群、安装可选云服务扩复制场景：
            部署生产存储，LLD填生产端参数，生产端参数从LLD表获取，灾备参数取默认值，通过接口param_util.get_param_value获取参数结果：
                product_admin_password = pwd   disaster_admin_password = None
            部署灾备存储，LLD填生产端和灾备端参数，参数都从LLD表获取，通过接口param_util.get_param_value获取参数结果：
                product_admin_password = pwd   disaster_admin_password = pwd
        """
        product_admin_password = self.param_util.get_param_value(
            self.pod_id, "FusionStorageBlockReplication", 'produce_storage_fsm_admin_password')
        disaster_admin_password = self.param_util.get_param_value(
            self.pod_id, "FusionStorageBlockReplication", 'disaster_storage_fsm_admin_password')
        local_admin_password = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMPortalPassword")
        if ProjectUtils.get_project_condition_boolean(
                self.project_id, '(ExpansionServiceStorage&!TenantStorFB80)|ExpansionAdCloudService'):
            local_admin_password = disaster_admin_password if disaster_admin_password else product_admin_password
        if not product_admin_password:
            product_admin_password = self.db.get_value_from_cloudparam(
                self.pod_id, "DistributedStorage", "FSMPortalPassword")
        if not disaster_admin_password:
            disaster_admin_password = self.db.get_value_from_cloudparam(
                self.pod_id, "DistributedStorage", "FSMPortalPassword")
        self.params_dict['local_admin_password'] = local_admin_password
        self.params_dict['produce_admin_password'] = product_admin_password
        self.params_dict['disaster_admin_password'] = disaster_admin_password

    # 复制独立部署 追加部署 生产场景
    def get_drstorage_tfb_pd_params(self):
        param_is_first = ["produce_storage_fsm_ip", "produce_storage_fsm_dsware_password",
                          "produce_storage_fsm_root_password", "produce_storage_fsm_cmd_password"]
        param_key = ["local_storage_fsm_ip", "local_storage_fsm_dsware_password",
                     "local_storage_fsm_root_password", "local_cmd_password"]
        for index, param_name in enumerate(param_is_first):
            if param_name == "produce_storage_fsm_cmd_password":
                self.params_dict[param_key[index]] = BaseOps().get_value_from_cloudparam(
                    self.pod_id, self.service_name, "cmdadmin_password")
            else:
                self.params_dict[param_key[index]] = self.param_util.get_param_value(
                    self.pod_id, self.service_name, param_name)

    # 复制独立部署 追加部署 灾备场景
    def get_not_drstorage_tfb_pd_params(self):
        param_not_first = ["disaster_tolerant_storage_fsm_ip", "disaster_tolerant_storage_fsm_dsware_password",
                           "disaster_tolerant_storage_fsm_root_password",
                           "disaster_tolerant_storage_fsm_cmd_password",
                           "storage_arbitration_manage_ip", "storage_arbitration_root_password",
                           "produce_storage_fsm_ip", "produce_storage_fsm_dsware_password",
                           "produce_storage_fsm_cmd_password"]

        param_key = ["local_storage_fsm_ip", "local_storage_fsm_dsware_password",
                     "local_storage_fsm_root_password", "local_cmd_password",
                     "storage_arbitration_ip", "storage_arbitration_root_password",
                     "remote_storage_fsm_ip", "remote_storage_fsm_dsware_password",
                     "remote_cmd_password"]
        # 如果选择的是CSDR，就不涉及到仲裁服务器相关
        if ("CSHAStorage_TFB" or "CSHA") not in self.project_condition:
            param_not_first.remove("storage_arbitration_manage_ip")
            param_not_first.remove("storage_arbitration_root_password")
            param_key.remove("storage_arbitration_ip")
            param_key.remove("storage_arbitration_root_password")
        for index, param_name in enumerate(param_not_first):
            if param_name not in ("disaster_tolerant_storage_fsm_cmd_password", "produce_storage_fsm_cmd_password"):
                self.params_dict[param_key[index]] = self.param_util.get_param_value(
                    self.pod_id, self.service_name, param_name)
            else:
                self.params_dict[param_key[index]] = BaseOps().get_value_from_cloudparam(
                    self.pod_id, self.service_name, "cmdadmin_password")

    # 复制独立部署/复制存储融合部署  全新建 生产场景
    def get_drstorage_tfb_pd_new_params(self):
        param_key = ["local_storage_fsm_ip", "local_storage_fsm_dsware_password",
                     "local_storage_fsm_root_password", "local_cmd_password"]

        # FSM浮动ip和cmd密码
        self.get_cmd_pwd_float_ip(param_key)
        # 在配置文件中获取dsware和root密码
        self.params_dict[param_key[1]] = BaseOps().get_value_from_cloudparam(
            self.pod_id, self.service_name, "dsware_password")
        self.params_dict[param_key[2]] = BaseOps().get_value_from_cloudparam(
            self.pod_id, self.service_name, "root_password")

    # 复制独立部署/复制存储融合部署 全新建 灾备场景
    def get_not_drstorage_tfb_pd_new_params(self):
        param_not_first = ["disaster_tolerant_storage_fsm_ip", "disaster_tolerant_storage_fsm_dsware_password",
                           "disaster_tolerant_storage_fsm_root_password",
                           "disaster_tolerant_storage_fsm_cmd_password",
                           "storage_arbitration_manage_ip", "storage_arbitration_root_password",
                           "produce_storage_fsm_ip", "produce_storage_fsm_dsware_password",
                           "produce_storage_fsm_cmd_password"]

        param_key = ["local_storage_fsm_ip", "local_storage_fsm_dsware_password",
                     "local_storage_fsm_root_password", "local_cmd_password",
                     "storage_arbitration_ip", "storage_arbitration_root_password",
                     "remote_storage_fsm_ip", "remote_storage_fsm_dsware_password",
                     "remote_cmd_password"]
        # 如果选择的是CSDR，就不涉及到仲裁服务器相关
        if ("CSHAStorage_TFB" or "CSHA") not in self.project_condition:
            param_not_first.remove("storage_arbitration_manage_ip")
            param_not_first.remove("storage_arbitration_root_password")
            param_key.remove("storage_arbitration_ip")
            param_key.remove("storage_arbitration_root_password")
        # FSM浮动ip和cmd密码
        self.get_cmd_pwd_float_ip(param_key)
        # 在配置文件中获取dsware和root密码
        self.params_dict[param_key[1]] = BaseOps().get_value_from_cloudparam(
            self.pod_id, self.service_name, "dsware_password")
        self.params_dict[param_key[2]] = BaseOps().get_value_from_cloudparam(
            self.pod_id, self.service_name, "root_password")
        # 直接在1.2参数中得到一部分参数
        for index, param_name in enumerate(param_not_first[4:]):
            if param_name == "produce_storage_fsm_cmd_password":
                # cmd密码从配置文件中获取
                self.params_dict[param_key[index + 4]] = BaseOps().get_value_from_cloudparam(
                    self.pod_id, self.service_name, "cmdadmin_password")
            else:
                self.params_dict[param_key[index + 4]] = self.param_util.get_param_value(self.pod_id, self.service_name,
                                                                                         param_name)

    def init_common_params(self):
        """
        获取1.2基本参数中信息
        """
        """
        复制独立部署 存储计算独立/存储计算融合 追加容灾场景
        """
        # 生产场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                'DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB|TenantStorFBHCI|TenantStorFBHCI80)&'
                'DRStorage_TFB_Sep&(ExpansionAdCloudService|ExpansionServiceStorage)'):
            self.get_drstorage_tfb_pd_params()
        # 灾备场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                '!DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB|TenantStorFBHCI|TenantStorFBHCI80)'
                '&DRStorage_TFB_Sep&(ExpansionAdCloudService|ExpansionServiceStorage)'):
            self.get_not_drstorage_tfb_pd_params()

        """
        复制独立部署 存储、计算独立/存储、计算融合 全新建场景
        """
        # 生产场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                'DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB|TenantStorFBHCI|TenantStorFBHCI80)&'
                'DRStorage_TFB_Sep&!(ExpansionAdCloudService|ExpansionServiceStorage)'):
            self.get_drstorage_tfb_pd_new_params()

        # 灾备场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                '!DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB|TenantStorFBHCI|TenantStorFBHCI80)&'
                'DRStorage_TFB_Sep&!(ExpansionAdCloudService|ExpansionServiceStorage)'):
            self.get_not_drstorage_tfb_pd_new_params()

        """
        复制&存储融合部署 计算独立部署 全新建场景
        """
        # 生产场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                'DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB)&!DRStorage_TFB_Sep&!ExpansionAdCloudService&'
                '(!TenantStorFBHCI|!TenantStorFBHCI80)'):
            self.get_drstorage_tfb_pd_new_params()

        # 灾备场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                '!DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB)&!DRStorage_TFB_Sep&!ExpansionAdCloudService&'
                '(!TenantStorFBHCI|!TenantStorFBHCI80)'):
            self.get_not_drstorage_tfb_pd_new_params()

        """
        复制&存储&计算融合部署 全新建场景
        """
        # 生产场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                '!DRStorage_TFB_Sep&DRStorage_TFB_PD&(TenantStorFBHCI|TenantStorFBHCI80)&!ExpansionAdCloudService'):
            self.get_drstorage_tfb_pd_new_params()

        # 灾备场景
        if ProjectUtils.get_project_condition_boolean(
                self.project_id,
                '!DRStorage_TFB_Sep&!DRStorage_TFB_PD&(TenantStorFBHCI|TenantStorFBHCI80)&'
                '!ExpansionAdCloudService'):
            self.get_not_drstorage_tfb_pd_new_params()

        # 获取生产和灾备的FSM admin的密码，如果没有key，则为缺省值。
        self.get_admin_passwd_params()

    def init_replication_cluster_nodes_info(self):
        """
        获取复制节点的信息1.4参数中获取
        """
        try:
            # 复制独立部署全新建场景/追加场景  /  复制存储融合全新建  不用区分是否生产或灾备
            if ProjectUtils.get_project_condition_boolean(
                    self.project_id,
                    '((CSHAStorage_TFB|CSDRStorage_TFB|TenantStorFBHCI|TenantStorFBHCI80)&DRStorage_TFB_Sep)|'
                    '((CSHAStorage_TFB|CSDRStorage_TFB)&!DRStorage_TFB_Sep&!ExpansionAdCloudService)'):

                # 1.4取参数时pod_id 不同
                nodes = self.db.get_install_os_list_info(self.pod_id, "rep")
                controller_cluster_ip, replication_cluster, ip_pool = [], [], []
                for node in nodes:
                    nodes_dict = {}
                    nodes_dict.setdefault("replication_manage_ip", node['manageIp'].split('/')[0])
                    string = node["creuser"]
                    user_list = string.split(",")
                    root_pwd = user_list[1]
                    common_usr = user_list[2]
                    common_pwd = user_list[3]
                    nodes_dict.setdefault("bmc_name", node["bmc_name"])
                    nodes_dict.setdefault("bmc_passwd", node["bmc_passwd"])
                    nodes_dict.setdefault("host_name", node["hostname"])
                    nodes_dict.setdefault("replication_manage_ip_root_password", root_pwd)
                    nodes_dict.setdefault("replication_manage_ip_common_username", common_usr)
                    nodes_dict.setdefault("replication_manage_ip_common_password", common_pwd)
                    nodes_dict.setdefault("arbitration_network", node["arbitration_plane_ip"])
                    nodes_dict.setdefault("replication_network", node["replication_plane_ip"])
                    ctrl = "1" if "rep_ctl" in node["deploy_component"] else ''
                    nodes_dict.setdefault("is_replication_controller_cluster", ctrl)
                    if "rep_ctl" in node["deploy_component"]:
                        nodes_dict.setdefault("replication_cluster_meta_info", node["replication_cluster_meta_info"])
                        nodes_dict.setdefault("replication_cluster_meta_type", node["replication_cluster_meta_type"])
                    self.replication_node_info.append(nodes_dict)
                    replication_cluster.append(node['manageIp'].split('/')[0])
                    ip_pool.append(node['replication_plane_ip'].split(',')[0])
                    ip_pool.append(node['replication_plane_ip'].split(',')[1])

                    if "rep_ctl" in node["deploy_component"]:
                        controller_cluster_ip.append(node['manageIp'].split('/')[0])
                # 将控制集群ip封装到ParamsDict
                self.params_dict.setdefault("replication_controller_cluster", controller_cluster_ip)
                # 封装复制集群,地址池ip到ParamsDict
                self.params_dict.setdefault("replication_cluster", replication_cluster)
                self.params_dict.setdefault("ip_pool", ip_pool)
                # 封装PSK_password，PSK_Label到ParamsDict
                self.params_dict["PSK_password"] = BaseOps().get_value_from_cloudparam(
                    self.pod_id, self.service_name, "PSK_password")
                self.params_dict["PSK_Label"] = BaseOps().get_value_from_cloudparam(
                    self.pod_id, self.service_name, "PSK_label")

                # 控制集群 复制集群校验
                self._check_rep_cluster_node_nums()
        except Exception as msg:
            err_msg = "catch exception:%s" % traceback.format_exc()
            logger.error(err_msg)
            raise Exception(msg) from msg

    def get_rep_cluster_info(self):
        """
        超融合 新建场景  复制节点信息在1.1设备信息参数中获取
        :return:
        """
        nodes = self.db.get_bmc_info_by_pod_id(self.pod_id)
        if not nodes:
            nodes = self.db.get_bmc_info_by_pod_id(self.pod_id)

        webui_client = FSPAuthClient.get_cps_web_client(self.db, self.project_id, self.pod_id)
        get_host_info = GetHostInfo(webui_client)
        controller_cluster_ip, replication_cluster = [], []
        for node in nodes:
            if 'rep' in node["ref_component"]:
                nodes_dict = {}
                bmc_ip = node['bmc_ip']
                # 通过bmc_ip 获得管理ip
                manage_info = get_host_info.get_host_info([bmc_ip])
                manage_ip = manage_info[1][0]
                nodes_dict.setdefault("replication_manage_ip", manage_ip)
                # 用户、密码
                root_pwd = self.db.get_value_from_cloudparam(
                    self.pod_id, "FusionStorageBlockReplication",
                    "FSRootpassword"
                )
                common_usr = "fsp"
                common_pwd = self.db.get_value_from_cloudparam(
                    self.pod_id, "FusionStorageBlockReplication",
                    "FSFsppassword"
                )
                nodes_dict.setdefault("replication_manage_ip_root_password", root_pwd)
                nodes_dict.setdefault("replication_manage_ip_common_username", common_usr)
                nodes_dict.setdefault("replication_manage_ip_common_password", common_pwd)
                nodes_dict.setdefault("bmc_name", node["bmc_name"])
                nodes_dict.setdefault("bmc_passwd", node["bmc_passwd"])
                ctrl = "1" if "rep_ctl" in node["ref_component"] else ''
                nodes_dict.setdefault("is_replication_controller_cluster", ctrl)
                if "rep_ctl" in node["ref_component"]:
                    nodes_dict.setdefault("replication_cluster_meta_info", node["replication_cluster_meta_info"])
                self.replication_node_info.append(nodes_dict)
                replication_cluster.append(manage_ip)
                if "rep_ctl" in node["ref_component"]:
                    controller_cluster_ip.append(manage_ip)
        # 将控制集群ip封装到ParamsDict
        self.params_dict.setdefault("replication_controller_cluster", controller_cluster_ip)
        # 封装复制集群,地址池ip到ParamsDict
        self.params_dict.setdefault("replication_cluster", replication_cluster)
        # 封装PSK_password，PSK_Label到ParamsDict
        self.params_dict["PSK_password"] = BaseOps().get_value_from_cloudparam(
            self.pod_id, self.service_name, "PSK_password")
        self.params_dict["PSK_Label"] = BaseOps().get_value_from_cloudparam(self.pod_id, self.service_name, "PSK_label")
        # 控制集群 复制集群校验
        self._check_rep_cluster_node_nums()

    def get_params_dict(self):
        """
        获取LLD参数接口 参数key见各方法中的列表
        :return:
        """
        if not self.params_dict:
            self.init_common_params()
            if ProjectUtils.get_project_condition_boolean(
                    self.project_id, '!DRStorage_TFB_Sep&(TenantStorFBHCI|TenantStorFBHCI80)&!ExpansionAdCloudService'):
                self.get_rep_cluster_info()
            else:
                self.init_replication_cluster_nodes_info()
            # 将复制节点信息以json格式放入ParamsDict
            self.params_dict.setdefault("replication_cluster_nodes_info", self.replication_node_info)
        return self.params_dict

    def get_cmd_pwd_float_ip(self, param_key):
        """
        全新建（扩容云服务、扩pod、扩AZ）场景 获取float_ip和cmd_password
        :param param_key:
        :return:
        """
        if ProjectUtils.get_project_condition_boolean(self.project_id, 'ExpansionAZ_KVM'):
            self.params_dict[param_key[0]] = self.param_util.get_param_value(
                self.pod_id, self.service_name, "expansion_new_fsm_float_ip")
            self.params_dict[param_key[3]] = BaseOps().get_value_from_cloudparam(
                self.pod_id, self.service_name, "cmdadmin_password")
        elif ProjectUtils.get_project_condition_boolean(self.project_id, 'ExpansionServiceStorage'):
            self.params_dict[param_key[0]] = self.param_util.get_param_value(
                self.pod_id, self.service_name, "expansion_fusionstorage_float_ip")
            self.params_dict[param_key[3]] = BaseOps().get_value_from_cloudparam(
                self.pod_id, self.service_name, "cmdadmin_password")
        else:
            self.params_dict[param_key[0]] = BaseOps().get_value_from_cloudparam(
                self.pod_id, "DistributedStorageBusiness", "FSMOMFloatingIP02")
            self.params_dict[param_key[3]] = self.db.get_value_from_cloudparam(
                self.pod_id, "DistributedStorage", "FSMPortalPassword")

    def get_rep_fsa_params(self):
        """
        获取复制安装FSA参数接口
        :return:
        """
        fsa_parmas = {}

        lld_dict = self.get_params_dict()
        fsa_parmas['float_ip'] = lld_dict.get("local_storage_fsm_ip")
        vbs_list = []
        fsa_list = []
        osd_list = []
        ip_list = lld_dict.get("replication_cluster_nodes_info")
        for info in ip_list:
            vbs = {}
            fsa_node = {}
            osd_node = {}
            vbs['hostname'] = info.get('host_name')
            vbs['model'] = info.get('equipment_model')
            vbs['om_ip'] = info.get('replication_manage_ip')
            vbs['root_pwd'] = info.get('replication_manage_ip_root_password')
            vbs['user'] = info.get('replication_manage_ip_common_username')
            vbs['passwd'] = info.get('replication_manage_ip_common_password')
            vbs['rep_ctl'] = info.get('is_replication_controller_cluster')
            vbs['bmc_name'] = info.get('bmc_name')
            vbs['bmc_passwd'] = info.get('bmc_passwd')
            fsa_node['om_ip'] = info.get('replication_manage_ip')
            osd_node['om_ip'] = info.get('replication_manage_ip')
            osd_node['root_pwd'] = info.get('replication_manage_ip_root_password')
            vbs_list.append(vbs)
            fsa_list.append(fsa_node)
            osd_list.append(osd_node)

        fsa_parmas['vbs_list'] = vbs_list
        fsa_parmas['fsa_list'] = fsa_list
        fsa_parmas['osd_list'] = osd_list
        fsa_parmas['dm_init_pwd'] = self.db.get_value_from_cloudparam(
            self.pod_id, "DistributedStorage", "DMPortalInitPassword")
        fsa_parmas['dm_update_pwd'] = lld_dict.get('local_admin_password')

        return fsa_parmas

    def get_storage_net_params(self):
        net_infos = dict()
        net_infos["storage_frontend_net"] = self._get_frontend_network_info()
        net_infos["storage_backend_net"] = self._get_backend_network_info()
        net_infos["iScsi_service_net"] = self._get_iscsi_network_info()
        net_infos["rep1_net"] = self._get_rep1_network_info()
        net_infos["rep2_net"] = self._get_rep2_network_info()
        net_infos["arb_net"] = self._get_iscsi_network_info()
        return net_infos

    def _get_arb_network_info(self):
        network_info = self._get_network_from_lld(
            subnet="fusionstorage_rep_range2",
            netmask="fusionstorage_arb_netmask",
            gateway="fusionstorage_rep_gateway2",
            vlan="fusionstorage_arb_vlan"
        )
        if network_info:
            network_info["net_type"] = "rep.1"
            network_info["port_name"] = "rep1"
        return network_info

    def _get_rep2_network_info(self):
        network_info = self._get_network_from_lld(
            subnet="fusionstorage_rep_range2",
            netmask="fusionstorage_rep_netmask2",
            gateway="fusionstorage_rep_gateway2"
        )
        if network_info:
            network_info["net_type"] = "rep.1"
            network_info["port_name"] = "rep1"
        return network_info

    def _get_rep1_network_info(self):
        network_info = self._get_network_from_lld(
            subnet="fusionstorage_rep_range1",
            netmask="fusionstorage_rep_netmask1",
            gateway="fusionstorage_rep_gateway1"
        )
        if network_info:
            network_info["net_type"] = "rep.0"
            network_info["port_name"] = "rep0"
        return network_info

    def _get_iscsi_network_info(self):
        network_info = self._get_network_from_lld(
            subnet="iscsi_service_range",
            netmask="iscsi_service_netmask",
            gateway="iscsi_service_gateway",
            vlan="iscsi_service_vlan",
            bond_mode="iscsi_service_plane_mode"
        )
        if network_info:
            network_info["net_type"] = "iscsi"
            network_info["port_name"] = "iscsi_data0"
        return network_info

    def _get_backend_network_info(self):
        network_info = self._get_network_from_lld(
            subnet="fusionstorage_inner_range",
            netmask="fusionstorage_inner_netmask",
            gateway="fusionstorage_inner_gateway",
            vlan="fusionstorage_inner_vlan",
            bond_mode="fusionstorage_inner_plane_mode"
        )
        if network_info:
            network_info["net_type"] = "storage_backend"
            network_info["port_name"] = DeployConstant.BACKEND_STORAGE_NET_INTF
        return network_info

    def _get_frontend_network_info(self):
        network_info = self._get_network_from_lld(
            subnet="fusionstorage_service_range",
            netmask="fusionstorage_service_netmask",
            gateway="fusionstorage_service_gateway",
            vlan="fusionstorage_service_vlan",
            bond_mode="fusionstorage_service_mode"
        )
        if network_info:
            network_info["net_type"] = "storage_frontend"
            network_info["port_name"] = DeployConstant.MGR_STORAGE_NET_INTF
        return network_info

    def _get_network_from_lld(self, **kwargs):
        subnet = kwargs.get("subnet")
        netmask = kwargs.get("netmask")
        gateway = kwargs.get("gateway")
        vlan = kwargs.get("vlan")
        bond_mode = kwargs.get("bond_mode")
        storage_network_info = dict()
        net_range = self.db.get_user_input_cloud_param_by_key(self.project_id, subnet)
        net_mask = self.db.get_user_input_cloud_param_by_key(self.project_id, netmask)
        net_gateway = self.db.get_user_input_cloud_param_by_key(self.project_id, gateway)
        net_vlan = None
        if vlan:
            net_vlan = self.db.get_user_input_cloud_param_by_key(self.project_id, vlan)
        net_bond_mode = None
        if bond_mode:
            net_bond_mode = self.db.get_user_input_cloud_param_by_key(self.project_id, bond_mode)
        if not net_mask or not net_gateway:
            err_msg = "Failed to get storage network[netmask(%s):%s, " \
                      "gateway(%s):%s, subnet(%s):%s]" % \
                      (netmask, net_mask, gateway, net_gateway,
                       subnet, net_range)
            logger.error(err_msg)
        else:
            storage_network_info['gateway'] = net_gateway
            storage_network_info['netmask'] = net_mask
            storage_network_info['range'] = net_range
            storage_network_info['vlan'] = net_vlan
            storage_network_info['bond_mode'] = net_bond_mode
        return storage_network_info

    def _check_rep_cluster_node_nums(self):
        ctl_cluster = self.params_dict.get("replication_controller_cluster")
        if not ctl_cluster or len(ctl_cluster) not in [3, 5, 7]:
            raise FCDException(627106, "replication_controller_cluster_ip:%s is invalid or empty." % ctl_cluster)
        rep_cluster = self.params_dict.get("replication_cluster")
        if not rep_cluster or len(rep_cluster) > 64:
            raise FCDException(627109, "Service cluster :%s length is invalid or empty." % rep_cluster)