# -*- coding: utf-8 -*-
import traceback

import utils.common.log as logger
from utils.common.message import Message
from utils.common.exception import HCCIException
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from plugins.DistributedStorage.common.base import TestCase
from plugins.DistributedStorage.common.upgrade_operate import UpgradeOperate


class UpdateCMDB(TestCase):
    def __init__(self, project_id, pod_id, regionid_list, fs_args, **kwargs):
        super(UpdateCMDB, self).__init__(project_id, pod_id)
        self.more_args = kwargs
        self.opr = UpgradeOperate(fs_args)
        self.user_name = fs_args["user_name"]
        self.password = fs_args["password"]
        self.service_name = 'FusionStorageBlock'
        self.region_id = regionid_list[0]
        self.float_ip = fs_args.get('float_ip')
        self.master_ip = fs_args.get('master_ip')
        self.slaver_ip = fs_args.get('slaver_ip')

    def procedure(self):
        logger.info('Start update CMDB.')
        try:
            self.main()
        except HCCIException as e:
            logger.error('update CMDB failed:{}'.format(e))
            logger.error(traceback.format_exc())
            raise e
        except Exception as e:
            logger.error('update CMDB failed:{}'.format(e))
            logger.error(traceback.format_exc())
            raise HCCIException(620008, str(e)) from e
        return Message(200)

    def main(self):
        status_code, error_code, error_des = self.opr.try_login(
            self.user_name, self.password)
        if status_code != 200 or error_code != 0:
            err_msg = "Failed to login, Detail:[status:%s,code:%s]%s" \
                      % (status_code, error_code, error_des)
            logger.error(err_msg)
            raise Exception(err_msg)
        product_version = self._get_product_version()
        product_esn = self._get_product_esn()
        name = self.service_name + "_" + product_esn
        cmdb_info = self._get_cmdb_info(name)
        if cmdb_info is not None and len(cmdb_info) > 0:
            logger.info("get cmdb info")
            set_cmdb_info = {
                "indexName": self.service_name,
                "version": product_version,
                "name": name
            }
        else:
            logger.info("general cmdb info")
            set_cmdb_info = {
                "indexName": self.service_name,
                "version": product_version,
                "name": name,
                "extendInfos": [
                    {"key": "fsm_float_ip", "value": self.float_ip},
                    {"key": "fsm_primary_ip", "value": self.master_ip},
                    {"key": "fsm_slave_ip", "value": self.slaver_ip}
                ]
            }
        mo_cmdb = ManageOneCmdbUtil(self.project_id)
        logger.info("set cmdb %s" % set_cmdb_info)
        mo_cmdb.set_cloud_service_info(self.region_id, set_cmdb_info)
        logger.info("End to set cloud info into CMDB.")

    def _get_cmdb_info(self, name):
        mo_cmdb = ManageOneCmdbUtil(self.project_id)
        cmdb_info = mo_cmdb.get_cloud_service_info(self.region_id, name=name)
        logger.info("get cmdb info %s" % cmdb_info)
        return cmdb_info

    def _get_product_version(self):
        logger.info('get cluster product version.')
        ret_result, ret_data = self.opr.get_product()
        if ret_result["code"] != 0:
            err_msg = "get cluster product failed, " \
                      "Detail:[result:%s, data:%s]" \
                      % (ret_result, ret_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        product_version = ret_data["version"]
        return product_version

    def _get_product_esn(self):
        ret_result, ret_data = self.opr.get_esn()
        if ret_result["code"] != 0:
            err_msg = "get cluster product failed, " \
                      "Detail:[result:%s, data:%s]" \
                      % (ret_result, ret_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        product_esn = ret_data["esn"]
        return product_esn
