# -*- coding: utf-8 -*-
from IPy import IP
from utils.common.fic_base import StepBaseInterface
from utils.common.fic_base import TestCase
from utils.common.message import Message
from utils.common.exception import HCCIException
import utils.common.log as logger


class NodeCheckIntf(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(NodeCheckIntf, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.implement = NodeCheck(project_id, pod_id)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        try:
            self.implement.ip_check()
        except HCCIException as e1:
            return Message(500, e1)
        except Exception as e2:
            return Message(500, HCCIException(627212, str(e2)))
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass


class NodeCheck(TestCase):

    @staticmethod
    def ip_ping(ip):
        import subprocess
        try:
            IP(ip)
        except Exception as e:
            logger.info("%s is not a ip address. Detail:%s" % (ip, str(e)))
            return False
        if IP(ip).version() == 6:
            ping_act = "ping6"
        else:
            ping_act = "ping"
        p = subprocess.Popen([ping_act, "-c", "3", ip], shell=False, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        try:
            (std_output, err_output) = p.communicate(timeout=10)
        except subprocess.TimeoutExpired as e:
            logger.info("{}".format(e))
            return False
        std_output = std_output.decode('utf-8') if isinstance(std_output, bytes) else std_output
        err_output = err_output.decode('utf-8') if isinstance(err_output, bytes) else err_output
        logger.info("Run ping cmd detail: node:%s, standard output:%s, error output:%s" % (ip, std_output, err_output))
        if std_output.find('ttl=') >= 0:
            return True
        return False

    def ip_check(self):
        """
        检查工具分配的存储节点管理IP地址是否可PING通
        """
        all_nodes_data = self.db.get_install_os_list_info(self.pod_id) + \
            self.db.get_install_os_list_info(self.pod_id, "rep")
        ip_dict = dict([(node.get('bmc_ip'), node.get('manageIp')) for node in all_nodes_data])
        ip_used_list = list()
        bmc_ip_list = list()
        for bmc_ip, om_ip in ip_dict.items():
            if self.ip_ping(om_ip):
                bmc_ip_list.append(bmc_ip)
                ip_used_list.append("(BMC IP:%s, Manage IP:%s)" % (bmc_ip, om_ip))
        if len(ip_used_list) > 0:
            logger.info("The management IP addresses of the nodes can be pinged. nodes:%s" % ip_used_list)
            raise HCCIException(627212, ','.join(bmc_ip_list), ','.join(ip_used_list))
