# -*- coding: utf-8 -*-
import os
import utils.common.log as logger
from plugins.DistributedStorage.logic.InstallOperate import InstallOperate
from plugins.ManageOne.scripts.utils.get_solution import get_solution
from utils.business.OpenStack.cps_pkg_uploader import CpsPkgUploader
from utils.client.FSPAuthClient import FSPAuthClient
from utils.constant.FCDConstant import get_compute_dpdk_hosts
from utils.common.fic_base import TestCase
from utils.common.OpenStackNodeManager import OpenStackNodeManager
from utils.Driver.CloudDC.OpenStack.get_host_info import GetHostInfo
from utils.PasswordManager.PasswordManager import FSPasswordManager
from utils.common.ssh_util import Ssh as sshCmd


class InstallRole(TestCase):
    def __init__(self, project_id, pod_id, fs_args, condition=None, metadata=None, **kwargs):
        super(InstallRole, self).__init__(project_id, pod_id)
        self.operate = InstallOperate(self.project_id, self.pod_id, fs_args)
        self.cps_web_client = FSPAuthClient.get_cps_web_client(self.db, self.project_id, self.pod_id)
        self.cps_host_client = GetHostInfo(self.cps_web_client)
        self.condition = condition
        self.metadata = metadata
        self.more_args = kwargs
        self.progress_deploy = get_solution()
        self.pkg_file_path = fs_args.get("uvp_install_path")
        # 要预留资源的计算节点列表
        self.vbs_bmc_ip_list = self.get_vbs_ip_list()
        self.scale_cpu = 4
        self.scale_memory = 7

    def get_vbs_ip_list(self):
        x86_arm_compute_role_list = get_compute_dpdk_hosts()
        # 筛选适合的计算节点
        tsfb_compute_bmc_ip_list = OpenStackNodeManager.get_bmc_ip_by_role_list(
            self.db, self.pod_id, x86_arm_compute_role_list)
        return tsfb_compute_bmc_ip_list

    def procedure(self):
        if len(self.vbs_bmc_ip_list) == 0:
            logger.info("Openstack compute node not exist, skip to install role.")
            return
        logger.info("Upload UVP package to cps-web.")
        cps_web_client = FSPAuthClient.get_cps_rest_client(self.db, self.project_id, self.pod_id)
        pkg_path = os.path.dirname(self.pkg_file_path)
        pkg_file = os.path.basename(self.pkg_file_path)
        upload_obj = CpsPkgUploader(cps_web_client)
        upload_obj.pkg_upload(pkg_file, pkg_path)
        logger.info("Succeed to upload UVP package[%s/%s] to cps-web." % (pkg_path, pkg_file))

        logger.info("Install fusionstorage-block role on bmc compute node:%s" % self.vbs_bmc_ip_list)
        compute_id_list, compute_ip_list = self.cps_host_client.get_host_info(self.vbs_bmc_ip_list)
        if not compute_id_list:
            err_msg = "The node list is empty. There is no node to install role"
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info('Start config and install role for node%s' % compute_ip_list)
        self.operate.config_and_deploy_role(compute_id_list, self.scale_cpu, self.scale_memory)

        logger.info('Checking FusionStorage-Agent rpm package')
        host_list = self.get_host_info_by_bmc(self.pod_id, self.cps_host_client, self.vbs_bmc_ip_list)
        self.operate.check_fusionstorage_agent_rpm(host_list)

        logger.info("Stop fusionstorage-blockXXX role nodes%s" % str(compute_ip_list))
        self.operate.stop_fusionstorage_block_template(cps_web_client, compute_id_list)

    def cleanup(self):
        compute_id_list, compute_ip_list = self.cps_host_client.get_host_info(self.vbs_bmc_ip_list)
        cps_client = FSPAuthClient.get_cps_rest_client(self.db, self.project_id, self.pod_id)
        logger.info("Stop fusionstorage-blockXXX role nodes%s" % str(compute_ip_list))
        self.operate.stop_fusionstorage_block_template(cps_client, compute_id_list)
        logger.info("Cleaning up fusionstorage-blockXXX role nodes%s" % str(compute_ip_list))
        self.operate.remove_fusionstorage_block_role(cps_client, compute_id_list)
        logger.info("Checking fusionstorage-blockXXX role nodes%s" % str(compute_ip_list))
        self.operate.check_fusionstorage_block_on_host(cps_client, compute_id_list)

    @staticmethod
    def get_host_info_by_bmc(pod_id, host_client, role_bmc_list):
        role_list = list()
        all_cps_hosts = host_client.get_hosts_info()
        for host in all_cps_hosts:
            if host["metadata"].get('ipmiip') is None:
                continue
            for role_bmc in role_bmc_list:
                if role_bmc != host['metadata']['ipmiip']:
                    continue
                node = dict()
                node['bmc_ip'] = role_bmc
                node['om_ip'] = host['omip']
                node['hostname'] = host['id']
                node['user'] = 'fsp'
                node['passwd'] = FSPasswordManager(pod_id).get_fsp_password()
                node['root_pwd'] = FSPasswordManager(pod_id).get_root_password()
                role_list.append(node)
        return role_list

    def unregister_cloud_agent(self):
        host_list = self.get_host_info_by_bmc(self.pod_id, self.cps_host_client, self.vbs_bmc_ip_list)
        for host in host_list:
            om_ip = host.get('om_ip')
            user = host.get('user')
            password = host.get('passwd')
            root_pwd = host.get('root_pwd')
            logger.info("unregister cloud_agent on node[%s]" % om_ip)
            ssh_client = InstallOperate.create_ssh_root_client(om_ip, user, password, root_pwd)
            self.unregister_component(ssh_client, 'cloud_agent')

    def modify_agent_control_file(self):
        """
        修改FusionStorage-AgentControl文件,确保预留资源场景组件不告警
        :return:
        """
        if not self.vbs_bmc_ip_list:
            return
        host_list = self.get_host_info_by_bmc(self.pod_id, self.cps_host_client, self.vbs_bmc_ip_list)
        for host in host_list:
            om_ip = host.get('om_ip')
            user = host.get('user')
            password = host.get('passwd')
            root_pwd = host.get('root_pwd')
            logger.info("modify agent control file on node[%s]" % om_ip)
            ssh_client = InstallOperate.create_ssh_root_client(om_ip, user, password, root_pwd)
            cmd = "line=`grep -an \"STATUS)\" /usr/bin/FusionStorage-AgentControl | " \
                  "awk -va=4 -F \"-|:\" '{print $1+a}'`;sed -i \"${line}i return 0\" " \
                  "/usr/bin/FusionStorage-AgentControl;echo last=$?"
            cmd_res = sshCmd.ssh_exec_command_return(ssh_client, cmd, 120, 0)
            if "last=0" in ';'.join(cmd_res):
                logger.info("modify agent control file on node[%s] success" % om_ip)
            else:
                error_msg = "modify agent control file on node[%s] failed, detail:%s" % (om_ip, cmd_res)
                logger.error(error_msg)
                raise Exception(error_msg)

    @staticmethod
    def unregister_component(ssh_client, component):
        cmd_unreg = "cps-monUnRegisterIdentify -P %s;echo unreg_status=$?" % component
        cmd_res = sshCmd.ssh_exec_command_return(ssh_client, cmd_unreg, 120, 0)
        logger.info("Run cmd[%s], result: %s" % (cmd_unreg, str(cmd_res)))
        if ssh_client:
            sshCmd.ssh_close(ssh_client)
