# coding=utf-8
import traceback

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.fic_base import StepBaseInterface
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.param_util import ParamUtil
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil


class CheckBmcConflictImpl(object):
    def __init__(self, project_id, pod_id):
        self.pod_id = pod_id
        self.project_id = project_id
        self.db = BaseOps()
        self.param_util = ParamUtil()
        self.mo_cmdb = ManageOneCmdbUtil(self.project_id, self.pod_id)

    def procedure(self):
        conflict_bmc_ip = []
        lld_bmc_ip_list = self.__get_lld_bmc_ip()
        deployed_bmc_ip_list = self.__get_exist_host_bmc_ip()
        logger.info("lld_bmc_ip_list: %s,deployed_bmc_ip_list: %s" % (lld_bmc_ip_list, deployed_bmc_ip_list))
        for bmc_ip in lld_bmc_ip_list:
            if bmc_ip in deployed_bmc_ip_list and bmc_ip not in conflict_bmc_ip:
                conflict_bmc_ip.append(bmc_ip)
                logger.error("lld bmc_ip:%s already exists,please check" % bmc_ip)
        logger.info("conflict_bmc_ip:%s " % conflict_bmc_ip)
        if conflict_bmc_ip:
            logger.error("conflict_bmc_ip:%s " % conflict_bmc_ip)
            raise HCCIException(365115, bmc_ip=conflict_bmc_ip)

    def __get_exist_host_bmc_ip(self):
        service_name = 'DistributedStorageBusiness'
        region_code = self.param_util.get_param_value(self.pod_id, service_name, 'region0_id')
        cmdb_infos = self.mo_cmdb.get_cloud_service_info(region_code, index_name="FusionStorageBlock")
        deployed_bmc_ip_list = []
        for infos in cmdb_infos:
            extend_infos = infos.get("extendInfos")
            for info in extend_infos:
                if info.get("key") == "deploy_bmc_node":
                    bmc_ips_list = info.get("value").split(";")
                    deployed_bmc_ip_list.extend(bmc_ips_list)
        return deployed_bmc_ip_list

    def __get_lld_bmc_ip(self):
        nodes = self.db.get_bmc_info_by_pod_id(self.pod_id)
        nodes_fsm_deploy = self.db.get_install_os_list_info(self.pod_id)
        nodes.extend(nodes_fsm_deploy)
        nodes_list = [node.get("bmc_ip") for node in nodes]
        return nodes_list


class CheckBmcConflict(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(CheckBmcConflict, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.implement = CheckBmcConflictImpl(project_id, pod_id)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def execute(self, project_id, pod_id):
        try:
            self.implement.procedure()
        except HCCIException as e1:
            logger.error("BMC IP conflict error:{}".format(traceback.format_exc()))
            return Message(500, e1)
        except Exception as e2:
            logger.error("BMC IP conflict error:{}".format(traceback.format_exc()))
            return Message(500, HCCIException('113999', str(e2)))
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)
