#!/usr/bin/bash

current_dir=$(
  cd "$(dirname "$0")" || exit
  pwd
)
TOP_DIR=$(readlink -f "${current_dir}/../../")
LOG_DIR="/var/log/fsc_cli"
LOG_PATH=$LOG_DIR"/fsc_cli.log"
RESERVED_PORTS=(11521 11522 11523 11524 11525)

function LOG() {
  local level=$1
  local content=$2
  if [ -L ${LOG_PATH} ]; then
    return 1
  fi
  if [ ! -d ${LOG_DIR} ]; then
    mkdir -p ${LOG_DIR}
    chmod 750 ${LOG_DIR}
  fi
  [ -L ${LOG_PATH} ] || echo "[$(date +"%Y-%m-%d %H:%M:%S")][$level]" "${content}" >> ${LOG_PATH}
}

##############################################################
## @Usage check_mount_unshare path
## @Return
## @Description 检查一个路径是否存在ushare加mount注入的风险，会去掉路径最后的"/",路径中的"//"会替换为"/"
##############################################################
function check_mount_unshare(){
    local checkPath="$1"
    if echo "$checkPath" | grep -q -E '\/$'
    then
        checkPath=${checkPath%?}
    fi
    checkPath=${checkPath//\/\//\/}
    # 通用方法，如果文件不存在则默认放行
    if [[ ! -e ${checkPath} ]]
    then
        return 0
    fi

    if [[ $(readlink -f ${checkPath}) != ${checkPath} ]]
    then
        LOG "${checkPath} can be attack by unshare and mount"
        return 1
    fi
    return 0
}

function init() {
  if [ -z $jre_dir ]; then
    JRE_PATH="/usr/share/dsware/jre1.8.0_272/"
    for folder in $(find /usr/share/dsware/ -name "jre*" -type d); do
        JRE_PATH=${folder}
    done
    KEY_TOOL="${JRE_PATH}/bin/keytool"
  else
    KEY_TOOL="${jre_dir}/bin/keytool"
  fi

  primary_key="/opt/dsware/agent/conf/kmc/conf/primary_ks.key"
  standby_key="/opt/dsware/agent/conf/kmc/bkp/standby_ks.key"
  if [ ! -f $primary_key ]; then
    if [ ! -d "/opt/dsware/agent/conf/kmc/conf/" ]; then
        mkdir -p "/opt/dsware/agent/conf/kmc/conf/"
    fi
    \cp -arf "${TOP_DIR}/lib/primary_ks.key" $primary_key
  fi
  if [ ! -f $standby_key ]; then
    if [ ! -d "/opt/dsware/agent/conf/kmc/bkp/" ]; then
        mkdir -p "/opt/dsware/agent/conf/kmc/bkp/"
    fi
    \cp -arf "${TOP_DIR}/lib/standby_ks.key" $standby_key
  fi
  
  tmp_cert_dir="${TOP_DIR}/conf/${float_ip}_$RANDOM/"
  split_output_path="${tmp_cert_dir}/split_output/"
  tmp_CA_jks=${tmp_cert_dir}"client_trust.keystore"
  tmp_cert_jks=${tmp_cert_dir}"client_self.keystore"

  if [[ ! -d ${tmp_cert_dir} ]]; then
    mkdir -p ${tmp_cert_dir}
    chmod 750 ${tmp_cert_dir}
  fi

  bak_time=$(date "+%Y%m%d%H%M%S")
}

function check_cert_format() {
  openssl x509 -in "${ca_cert}" -text -noout 1>/dev/null 2>&1
  ret1=$?
  openssl x509 -in "${server_cert}" -text -noout 1>/dev/null 2>&1
  ret2=$?
  openssl rsa -in "${private_key}" -text -noout -passin stdin <<EOF
${cert_password}
EOF
  ret3=$?

  ret=$((ret1 + ret2 + ret3))
  if [[ ${ret} -ne 0 ]]; then
    LOG "ERROR" "${ca_cert} verify format failed.[Line:${LINENO}]"
    return 1
  fi

  openssl verify -CAfile "${ca_cert}" "${server_cert}" 1>/dev/null 2>&1
  local ret=$?
  if [ ${ret} -ne 0 ]; then
    LOG "ERROR" "failed to verify the CA certificate. [Line:${LINENO}]"
  fi

  local pub_key1
  pub_key1=$(openssl rsa -pubout -in "${private_key}" -passin stdin <<EOF
${cert_password}
EOF
)
  local pub_key2
  pub_key2=$(openssl x509 -pubkey -noout -in "${server_cert}")
  if [ "X${pub_key1}" != "X${pub_key2}" ]; then
    LOG "ERROR" "The private key does not match the certificate. [Line:${LINENO}]"
    return 1
  fi
  LOG "INFO" "verify the certificate success. [Line:${LINENO}]"
  return 0
}

function split_cert_chain() {
  local number=0
  local flag=0
  rm -rf ${split_output_path}
  mkdir -p ${split_output_path}
  while read -r line || [ -n "${line}" ]; do
    if [[ "$line" == *-----BEGIN* ]]; then
      ((++number)) || true
      flag=1
    fi
    if [ ${flag} -eq 1 ]; then
      [ -L "${split_output_path}${number}" ] || echo "$line" >> "${split_output_path}${number}"
    fi
    if [[ "$line" == *-----END* ]]; then
      flag=0
    fi
  done <"${ca_cert}"
  chmod 750 ${split_output_path}
  chmod 440 ${split_output_path}/*
  LOG "INFO" "split ca certificate success. [Line:${LINENO}]"

}

function gen_encrypt_jks() {
  cd ${tmp_cert_dir} || exit
  export LD_LIBRARY_PATH=""
  LOG "INFO" "begin gen dswareApi client_trust.keystore.[${SCRIPTNAME}:${LINENO}]"

  if [ ! -f ${KEY_TOOL} ]; then
    LOG "ERROR" "${KEY_TOOL} not exist.[${SCRIPTNAME}:${LINENO}]"
    return 1
  fi
  # 1.generate ca jks.
  for file_path in ${split_output_path}/*; do
    [[ -e "$file_path" ]] || break
    check_mount_unshare ${file_path}
    if [[ $? -eq 1 ]]; then
      LOG "ERROR" "var ${file_path} is forbbidon contain .. or link"
      rm -rf "${current_path:?}"/*
      return 1
    fi
    file_name=$(basename ${file_path})
    ${KEY_TOOL} -delete -alias ca_${file_name} -keystore ${tmp_CA_jks} <<EOF
${cert_password}
${cert_password}
EOF
    ${KEY_TOOL} -importcert -keystore ${tmp_CA_jks} -file ${file_path} -alias ca_${file_name} -noprompt <<EOF
${cert_password}
${cert_password}
EOF
    ret=$?
    if [[ $ret -ne 0 ]]; then
      LOG "ERROR" "gen encrypt dswareApi ca jks by password failed.[${SCRIPTNAME}:${LINENO}]"
      return 5
    fi
  done

  LOG "INFO" "end gen dswareApi client_trust.keystore.[${SCRIPTNAME}:${LINENO}]"

  # 2.generate cert jks.
  # combine private key and public cert into p12 format file.
  LOG "INFO" "begin gen client_self.keystore.[${SCRIPTNAME}:${LINENO}]"
  cert_p12="${tmp_cert_dir}/cert.p12"
  private_file=$(basename ${private_key})
  server_file=$(basename ${server_cert})
  openssl pkcs12 -export -inkey "${private_key}" -in "${server_cert}" -out ${cert_p12} -passin stdin -password stdin <<EOF
${cert_password}
${cert_password}
EOF

  # convert p12 file to jks format.
  chmod 440 ${cert_p12:?}
  check_mount_unshare ${cert_p12}
  if [[ $? -eq 1 ]]; then
    LOG "ERROR" "var ${cert_p12} is forbbidon contain .. or link"
    rm -rf "${current_path:?}"/*
    return 1
  fi
  ${KEY_TOOL} -importkeystore -srckeystore ${cert_p12} -srcstoretype pkcs12 -deststoretype pkcs12 -destkeystore ${tmp_cert_jks} -noprompt <<EOF
${cert_password}
${cert_password}
${cert_password}
EOF
  ret=$?
  if [[ $ret -ne 0 ]]; then
    LOG "ERROR" "gen dswareApi encrypt cert jks by password failed.[${SCRIPTNAME}:${LINENO}]"
    return 5
  fi
  chmod 550 ${tmp_cert_jks:?}
  LOG "INFO" "end gen dswareApi client_self.keystore.[${SCRIPTNAME}:${LINENO}]"
}

function replaceJks() {
  if [ -n "${float_ip}" ]; then
    DAWARE_API_DIR="${TOP_DIR}/conf/cert/${float_ip}/"
  else
    DAWARE_API_DIR="${TOP_DIR}/conf/"
  fi
  if [ ! -d ${DAWARE_API_DIR} ]; then
    mkdir -p ${DAWARE_API_DIR}
  fi

  if [ -n "${float_ip}" ]; then
    current_CA_jks="${TOP_DIR}/conf/cert/${float_ip}/client_trust.keystore"
    current_cert_jks="${TOP_DIR}/conf/cert/${float_ip}/client_self.keystore"
  else
    current_CA_jks="${TOP_DIR}/conf/client_trust.keystore"
    current_cert_jks="${TOP_DIR}/conf/client_self.keystore"
  fi

  if [ -f ${current_CA_jks} ]; then
    mv -f ${current_CA_jks} ${current_CA_jks}.bak
  fi
  if [ -f ${current_cert_jks} ]; then
    mv -f ${current_cert_jks} ${current_cert_jks}.bak
  fi

  mv -f ${tmp_CA_jks} ${current_CA_jks}
  ret=$?
  if [ $ret != 0 ]; then
    LOG "ERROR" "mv ${tmp_CA_jks} --> ${current_CA_jks} failed.[${SCRIPTNAME}:${LINENO}]"
    return 1
  fi
  mv -f ${tmp_cert_jks} ${current_cert_jks}
  ret=$?
  if [ $ret != 0 ]; then
    LOG "ERROR" "mv ${tmp_cert_jks} --> ${current_cert_jks} failed.[${SCRIPTNAME}:${LINENO}]"
    return 1
  fi
  chmod 600 ${current_CA_jks}
  chmod 600 ${current_cert_jks}

  LOG "INFO" "replace certificate: ${current_CA_jks} ${current_cert_jks} success.[${SCRIPTNAME}:${LINENO}]"
  return 0
}

function replaceConfig() {
  if [ -n "${float_ip}" ]; then
    DAWARE_API_DIR="conf/cert/${float_ip}/"
  else
    DAWARE_API_DIR="conf/"
  fi
  if [ ! -d ${TOP_DIR}/${DAWARE_API_DIR} ]; then
    mkdir -p ${TOP_DIR}/${DAWARE_API_DIR}
  fi
  DAWARE_API_PATH="${DAWARE_API_DIR}/dsware-api.properties"
  if [ -f ${TOP_DIR}/${DAWARE_API_PATH} ]; then
    mv -f ${TOP_DIR}/${DAWARE_API_PATH} ${TOP_DIR}/${DAWARE_API_PATH}.bak
  fi

  if [ -z ${LD_LIBRARY_PATH} ]; then
    export LD_LIBRARY_PATH=${TOP_DIR}/lib
  else
    export LD_LIBRARY_PATH=${TOP_DIR}/lib:$LD_LIBRARY_PATH
  fi
  export LD_LIBRARY_PATH=${TOP_DIR}/lib:$LD_LIBRARY_PATH
  python "${current_dir}/update_cfg.py" ${TOP_DIR} ${DAWARE_API_PATH} <<EOF
${cert_password}
EOF
  ret=$?
  if [ $ret != 0 ];then
    LOG "ERROR" "upgrade cfg ${DAWARE_API_PATH} failed.[${SCRIPTNAME}:${LINENO}]"
    return 1
  fi
  chmod 600 ${TOP_DIR}/${DAWARE_API_PATH}
  return ${ret}
}

function readInIfile()
{
  iniFile=$1
  section=$2
  option=$3

  if [ "${section}" != "" ] && [ "${option}" = "" ];then
      iniValue=$(awk "/\[${section}\]/{a=1}a==1"  ${iniFile}|sed -e'1d' -e '/^$/d'  -e 's/[ \t]*$//g' -e 's/^[ \t]*//g' -e 's/[ ]/@G@/g' -e '/\[/,$d' )
      LOG  "info" "iniOptions size:-${#iniOptions[@]}- elements:-${iniOptions[*]}-.[${SCRIPTNAME}:${LINENO}]"
  elif [ "${section}" != "" ] && [ "${option}" != "" ];then
      iniValue=`awk -F '=' "/\[${section}\]/{a=1}a==1" ${iniFile}|sed -e '1d' -e '/^$/d' -e '/^\[.*\]/,$d' -e "/^${option}.*=.*/!d" -e "s/^${option}.*= *//"`
      LOG  "info" "iniValue value:-${iniValue}-.[${SCRIPTNAME}:${LINENO}]"
  fi
  echo ${iniValue}
}

function writeInifile()
{
  iniFile=$1
  section=$2
  option=$3
  value=$4

  allSections=$(awk -F '[][]' '/\[.*]/{print $2}' ${iniFile})
  iniSections=(${allSections// /})
  sectionFlag="0"
  for temp in "${iniSections[@]}";do
      if [ "${temp}" = "${section}" ];then
          sectionFlag="1"
          break
      fi
  done

  if [ "$sectionFlag" = "0" ];then
      [ -L ${iniFile} ] || echo "[${section}]" >> ${iniFile}
  fi
  awk "/\[${section}\]/{a=1}a==1" ${iniFile}|sed -e '1d' -e '/^$/d'  -e 's/[ \t]*$//g' -e 's/^[ \t]*//g' -e '/\[/,$d'|grep "${option}.\?=">/dev/null
  if [ "$?" = "0" ];then
      sectionNum=$(sed -n -e "/\[${section}\]/=" ${iniFile})
      sed -i "${sectionNum},/^\[.*\]/s/\(${option}.\?=\).*/\1 ${value}/g" ${iniFile}
      LOG  "info" "update [$iniFile][$section][$option][$value] success.[${SCRIPTNAME}:${LINENO}]"
  else
      sed -i "/^\[${section}\]/a\\${option} = ${value}" ${iniFile}
      LOG  "info" "add [$iniFile][$section][$option][$value] success.[${SCRIPTNAME}:${LINENO}]"
  fi
}

function gen_listen_port_cfg() {
  iniFile="${TOP_DIR}/storage_port.ini"
  section="default"
  cfg_lock_file="/tmp/dsware_lock_file"

  exec 211>$cfg_lock_file
  flock -n 211 || {
    LOG  "error" "Another user is doing the same thing, please wait.."
    flock -w 10 211 || {
      LOG  "error" "wait timeout.[${SCRIPTNAME}:${LINENO}]"
      return 1
    }
  }
  for reserver_port in "${RESERVED_PORTS[@]}"; do
    if [ -f $iniFile ]; then
      exist_port=$(readInIfile ${iniFile} ${section} ${float_ip})
      LOG "info" "exist port is $exist_port, reserver_port $reserver_port. [${SCRIPTNAME}:${LINENO}]"
      if [ -n "${exist_port}" ]; then
        LOG "info" "exist_port $exist_port is free. [${SCRIPTNAME}:${LINENO}]"
        free_port=$exist_port
        break
      else
        all_ported=$(readInIfile ${iniFile} ${section})
        LOG "info" "all port is $all_ported, reserver_port $reserver_port. [${SCRIPTNAME}:${LINENO}]"
        if $(echo "$all_ported" | grep -q "$reserver_port"); then
          LOG  "info" "port $reserver_port existed, continue.[${SCRIPTNAME}:${LINENO}]"
          continue
        fi
      fi
    fi
    LOG "info" "reserver_port $reserver_port is free. [${SCRIPTNAME}:${LINENO}]"
    free_port=$reserver_port
    break
  done
  if [ "$free_port" == "" ]; then
    LOG  "error" "no free port.[${SCRIPTNAME}:${LINENO}]"
    return 1
  fi
  if [ -n "${float_ip}" ]; then
    writeInifile ${iniFile} ${section} ${float_ip} ${free_port}
    chmod 644 ${iniFile}
  fi
}

function restartService() {
  sleep 3
  old_pid=$(ps -ewwf | grep -F "FSCTools --op startServer" | grep ${free_port}  | grep -v grep | awk '{print $2}')
  LOG  "warning" "kill old pid ${old_pid} .[${SCRIPTNAME}:${LINENO}]"
  LOG  "ps -ewwf | grep -F 'FSCTools --op startServer' | grep ${free_port}  | grep -v grep | awk '{print $2}'"

  ps -ewwf | grep -F "FSCTools --op startServer" | grep ${free_port}  | grep -v grep | awk '{print $2}' | xargs kill -9 >/dev/null 2>&1
  ret=$?
  LOG  "warning" "kill old pid ret $ret .[${SCRIPTNAME}:${LINENO}]"

  return 0
}

function clear_tmp() {
  rm -rf ${tmp_cert_dir:?}
}

function main() {
  init

  echo "please input cert password:"
  stty -echo 2>/dev/null
  read -r "cert_password"
  stty echo 2>/dev/null

  check_cert_format
  if [ $? != 0 ]; then
    return 1
  fi
  split_cert_chain
  if [ $? != 0 ]; then
    return 1
  fi

  gen_encrypt_jks
  if [ $? != 0 ]; then
    return 1
  fi

  replaceJks
  if [ $? != 0 ]; then
    return 1
  fi

  gen_listen_port_cfg
  if [ $? != 0 ]; then
    return 1
  fi

  replaceConfig
  if [ $? != 0 ]; then
    return 1
  fi

  restartService
  if [ $? != 0 ]; then
    return 1
  fi

}

function helper() {
  echo "usage: $0
    -s the path of server certificate file
    -c the path of ca certificate file
    -k the path of private key file
    -f the float IP of storage
    -j the dir for jre
    -h the info of help"
}

while getopts "s:c:k:f:j:h" arg; do
  case $arg in
  s)
    server_cert=$OPTARG
    ;;
  c)
    ca_cert=$OPTARG
    ;;
  k)
    private_key=$OPTARG
    ;;
  f)
    float_ip=$OPTARG
    ;;
  j)
    jre_dir=$OPTARG
    ;;
  h)
    helper
    exit 0
    ;;
  ?)
    helper
    exit 1
    ;;
  esac
done

if [ -z $server_cert ] || [ -z $ca_cert ] || [ -z $private_key ] || [ -z $float_ip ]; then
  LOG "ERROR" "param is wrong.[${SCRIPTNAME}:${LINENO}]"
  helper
  exit 1
fi

main
ret=$?
clear_tmp
exit $ret
