# -*- coding: utf-8 -*-
from IPy import IP
from utils.common.fic_base import StepBaseInterface
from utils.common.fic_base import TestCase
from utils.common.message import Message
from utils.common.exception import HCCIException
import utils.common.log as logger


class NodeCheckIntf(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(NodeCheckIntf, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.implement = NodeCheck(project_id, pod_id)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:
        """
        pass

    def execute(self, project_id, pod_id):
        try:
            self.implement.procedure()
        except HCCIException as e1:
            return Message(500, e1)
        except Exception as e2:
            return Message(500, HCCIException(627212, str(e2)))
        return Message(200)

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:
        """
        pass


class NodeCheck(TestCase):

    @staticmethod
    def ip_ping(ip):
        import subprocess
        try:
            IP(ip)
        except Exception as e:
            logger.info("%s is not a ip address. Detail:%s" % (ip, str(e)))
            return False
        if IP(ip).version() == 6:
            ping_act = "ping6"
        else:
            ping_act = "ping"
        p = subprocess.Popen([ping_act, "-c", "3", ip], shell=False, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        try:
            (std_output, err_output) = p.communicate(timeout=5)
        except subprocess.TimeoutExpired as e:
            logger.warn("Details: {}".format(e))
            return False
        std_output = std_output.decode('utf-8') if isinstance(std_output, bytes) else std_output
        err_output = err_output.decode('utf-8') if isinstance(err_output, bytes) else err_output
        logger.info("Run ping cmd detail: node:%s, standard output:%s, error output:%s" % (ip, std_output, err_output))
        if std_output.find('ttl=') >= 0:
            return True
        return False

    def procedure(self):
        """检查工具分配的存储节点IP地址是否被占用

        IP：manageIp、storageIp、replication_plane_ip、iscsi_business_plane_ip、storageInnerIp、arbitration_plane_ip
        """
        all_nodes_data = self.db.get_install_os_list_info(self.pod_id) + \
            self.db.get_install_os_list_info(self.pod_id, "rep")
        conflict_info = dict()
        conflict_node_set = set()
        for node_info in all_nodes_data:
            manage_ip = node_info.get("manageIp")
            storage_ip = node_info.get("storageIp")
            node_conflict_ip = {}
            if self.ip_ping(manage_ip):
                node_conflict_ip["manageIp"] = manage_ip
            if self.ip_ping(storage_ip):
                node_conflict_ip["storageIp"] = storage_ip

            storage_inner_ip = node_info.get("storageInnerIp")
            iscsi_business_plane_ip = node_info.get("iscsi_business_plane_ip")
            arbitration_plane_ip = node_info.get("arbitration_plane_ip")
            if storage_inner_ip and self.ip_ping(storage_inner_ip):
                node_conflict_ip["storageInnerIp"] = storage_inner_ip
            if iscsi_business_plane_ip and self.ip_ping(iscsi_business_plane_ip):
                node_conflict_ip["iscsi_business_plane_ip"] = iscsi_business_plane_ip
            if arbitration_plane_ip and self.ip_ping(arbitration_plane_ip):
                node_conflict_ip["arbitration_plane_ip"] = arbitration_plane_ip

            replication_plane_ip = node_info.get("replication_plane_ip")
            if replication_plane_ip:
                rep_ip1, rep_ip2 = replication_plane_ip.split(",")
                if self.ip_ping(rep_ip1):
                    node_conflict_ip["replication_plane_ip1"] = rep_ip1
                if self.ip_ping(rep_ip2):
                    node_conflict_ip["replication_plane_ip2"] = rep_ip2

            if node_conflict_ip:
                conflict_node_set.add(node_info.get("bmc_ip"))
                conflict_info[node_info.get("bmc_ip")] = node_conflict_ip

        if len(conflict_info) > 0:
            logger.error("Nodes with IP address conflicts: {}".format(conflict_info))
            raise HCCIException(627212, conflict_node_set, conflict_info)
