#!/bin/bash

CUR_DIR=$(dirname $(readlink -f $0))
source ${CUR_DIR}/bin/commonUtil.sh

LOG_NAME="upgrade.log"
RESTCONVERT_SCRIPT_LOG_PATH="/var/log/restconvert/scriptLog"
LOG_FILE="${RESTCONVERT_SCRIPT_LOG_PATH}/${LOG_NAME}"

initLog "${RESTCONVERT_SCRIPT_LOG_PATH}" "${LOG_NAME}"

UPGRADE_BACKUP_PATH="/home/rest/backUpProcessFile"

INSTALL_PATH=/opt/dfv/oam/rest

appPropertyFile=${INSTALL_PATH}/RestConvert/resources/application.properties

dbFile=${INSTALL_PATH}/RestConvert/resources/restconvert.db

UNINSTALL_SCRIPT_PATH="/opt/dfv/oam/rest/RestConvert/uninstall.sh"

CURRENT_VERSION_FILE="${INSTALL_PATH}/RestConvert/rest-version"

function prepareBeforeUpgrade()
{
  checkVersion
  if [ $? -ne 0 ];then
    log_error "check version failed"
    return 1
  fi

  if [ ! -f "${appPropertyFile}" ];then
    log_warn "application properties file not exist"
    return 1
  fi
  if [ ! -f "${dbFile}" ];then
    log_warn "db file not exist"
    return 1
  fi
  log_info "prepare before upgrade success"
  return 0
}

function checkVersion()
{
  local newMinorVersion=$(grep 'version=' ${CUR_DIR}/rest-version | awk -F '=' '{print $2}')
  local oldMinorVersion=$(grep 'version=' ${CURRENT_VERSION_FILE} | awk -F '=' '{print $2}')
  if [ "${newMinorVersion}" -eq "${oldMinorVersion}" ];then
    log_error "version num must not be same,new is ${newMinorVersion},current is ${oldMinorVersion}"
    return 1
  fi
  log_info "check version success,new version is ${newMinorVersion}"
  return 0
}

function backUpBeforeUpgrade()
{
  rm -rf ${UPGRADE_BACKUP_PATH}
  mkdir -p ${UPGRADE_BACKUP_PATH}
  chown -h rest:oam ${UPGRADE_BACKUP_PATH}
  su - rest -s "/bin/bash" -c "cp -rp ${INSTALL_PATH} ${UPGRADE_BACKUP_PATH}"
  if [ $? -ne 0 ];then
    log_error "copy rest process file failed"
    return 1
  fi
  return 0
}

function restoreBackUpFile()
{
  cp -f ${UPGRADE_BACKUP_PATH}/rest/RestConvert/resources/application.properties ${CUR_DIR}/resources/
  if [ $? -ne 0 ];then
    log_error "restore application properties file failed"
    return 1
  fi

  cp -f ${UPGRADE_BACKUP_PATH}/rest/RestConvert/resources/restconvert.db ${CUR_DIR}/resources/
  if [ $? -ne 0 ];then
    log_error "restore db File failed"
    return 1
  fi
}

function uninstallOldRest()
{
  sh ${UNINSTALL_SCRIPT_PATH} 1 >> /dev/null 2>&1
  return $?
}

function installNewRest()
{
  sh ${CUR_DIR}/install.sh >> /dev/null 2>&1
  return $?
}

function main()
{
  prepareBeforeUpgrade
  if [ $? -ne 0 ];then
      print_error "upgrade check failed!"
      return 1
  fi
  backUpBeforeUpgrade
  if [ $? -ne 0 ];then
      print_error "upgrade backup failed!"
      return 1
  fi
  restoreBackUpFile
  if [ $? -ne 0 ];then
      print_error "restore back up file failed!"
      return 1
  fi
  echo "start unInstall old RestConvert..."
  uninstallOldRest
  if [ $? -ne 0 ];then
      print_error "uninstall old rest process failed!"
      return 1
  fi
  echo "start install new RestConvert..."
  installNewRest
  if [ $? -ne 0 ];then
      print_error "install new rest process failed!"
      return 1
  fi
  return 0
}

main
ret=$?
[[ ${ret} -eq 0 ]] && print_info "restConvert upgrade successfully!" || print_error "restConvert upgrade failed!"
exit ${ret}