# coding=utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.fic_base import StepBaseInterface
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.project_condition_utils import get_project_condition_boolean
import utils.common.log as logger
from utils.business.vm_util import can_vm_pinged_to


class CheckIPReachable(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        super(CheckIPReachable, self).__init__(project_id, pod_id)
        self.project_id = project_id
        self.pod_id = pod_id
        self.db = BaseOps()

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def execute(self, project_id, pod_id):
        try:
            self.procedure(project_id)
            return Message(200)
        except HCCIException as e1:
            return Message(500, e1)
        except Exception as e2:
            return Message(500, HCCIException('626273', str(e2)))

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        return self.execute(project_id, pod_id)

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def procedure(self, project_id):
        params = dict()
        if get_project_condition_boolean(project_id, 'TenantStorFBReuse80&ExpansionAZ_KVM'):
            expansion_az_fsm_float_ip = self.db.get_user_input_cloud_param_by_key(
                project_id, "expansion_az_fsm_float_ip")
            params['expansion_az_fsm_float_ip'] = expansion_az_fsm_float_ip
        elif get_project_condition_boolean(
                self.project_id, 'ManageStorFB80&RegionConHA&ExpansionMgmtRes_ServiceNode'):
            dc001_expansion_fsm_float_ip = self.db.get_user_input_cloud_param_by_key(
                project_id, "dc001_expansion_fsm_float_ip")
            dc002_expansion_fsm_float_ip = self.db.get_user_input_cloud_param_by_key(
                project_id, "dc002_expansion_fsm_float_ip")
            params['dc001_expansion_fsm_float_ip'] = dc001_expansion_fsm_float_ip
            params['dc002_expansion_fsm_float_ip'] = dc002_expansion_fsm_float_ip
        elif get_project_condition_boolean(
                project_id, 'ManageStorFB80&(ExpansionScale_CloudServiceNode|ExpansionMgmtRes_ServiceNode)'):
            expansion_fsm_float_ip = self.db.get_user_input_cloud_param_by_key(project_id, "expansion_fsm_float_ip")
            params['expansion_fsm_float_ip'] = expansion_fsm_float_ip
        elif get_project_condition_boolean(
                project_id, 'TenantStorFB80&(ExpansionComputeRes_KVMNode|ExpansionNetworkRes_NetworkNode)'):
            reuse_fsm_float_ip = self.db.get_user_input_cloud_param_by_key(project_id, "reuse_fsm_float_ip")
            params['reuse_fsm_float_ip'] = reuse_fsm_float_ip

        not_ping_list = list()
        for key, value in params.items():
            logger.info("Start to ping %s[%s]" % (key, value))
            if not can_vm_pinged_to(value):
                not_ping_list.append("{}:{}".format(key, value))

        if len(not_ping_list) > 0:
            logger.info("Failed to ping ip address:%s" % not_ping_list)
            raise HCCIException(626308, not_ping_list)
