#!/bin/bash

# 返回值:0表示成功 1表示失败
# 温度值返回invalid表示无效，返回正确温度值为十进制整数

# DPU-P的ct7481的器件地址为0x98,iic协议传入的地址需要右移一位,0x0偏移为进风口温度,0x1为内存温度
dpu_p_ct7481_addr=0x4c
# DPU-S的ct7481的器件地址为0x98,iic协议传入的地址需要右移一位,0x0偏移为内存温度
dpu_s_ct7481_addr=0x4c
# DPU-S的LM75的器件地址为0x92,iic协议传入的地址需要右移一位,0x0偏移为进风口温度
dpu_s_lm75_addr=0x49

# 回传温度数组
declare -A temparray
cpu_result=''
# 温度写入临时文件
DPU_TEMPRATURE_FILE_PATH="/var/run/dpu_temprature.txt"
# 硬件形态,DPU-P/DPU-S
now_envir=$(dmidecode -t2 | grep "Product Name"|awk -F ': ' '{print $2}')

# 获取cpu温度
function get_cpu_temp()
{
    # 查看软连接个数
    cnt=$(ls /sys/class/thermal/ -l |grep -c "^l")
    if [ ${cnt} -eq 0 ]; then
        return
    fi
    for((i=0;i<${cnt};i++));
    do
        local cpu_temp=$(cat /sys/class/thermal/thermal_zone$i/temp)
        if [ $? -ne 0 ]; then
            cpu_temp='invalid'
        else
            # 合法性校验-40 - 125
            if [ ${cpu_temp} -gt -40000 -a ${cpu_temp} -lt 125000 ]; then
                cpu_temp=$((${cpu_temp}/1000))
            else
                cpu_temp='invalid'
            fi
        fi
        temparray[cpu$i]=${cpu_temp}
        cpu_result="${cpu_result}cpu$i=${temparray[cpu$i]},"
    done
    cpu_result=$(echo ${cpu_result%?})
}

function get_temp()
{
    get_ddr_temp
    get_inlet_temp
    # 核温跟硬件产品型号相关,只在DPU交付,ddr inlet中已经判断了硬件产品适配,如果不适配,脚本直接退出
    get_cpu_temp
}

function get_ddr_temp()
{
    case ${now_envir} in
        # DPU-P
        "STL6IEX2F001")
        get_ct7481_data '0x1' 'ddr'
        ;;
        # DPU-S
        "STL6IEC2F001")
        get_lm75_data '0x0' 'ddr'
        ;;
        "STL6SPCIA")
        get_lm75_data '0x0' 'ddr'
        ;;
        *)
        echo "Current environment is ${now_envir}, not dpu environment."
        exit 1
        ;;
    esac
}

function get_inlet_temp()
{
    case ${now_envir} in
        # DPU-P
        "STL6IEX2F001")
        get_ct7481_data '0x0' 'inlet'
        ;;
        # DPU-S
        "STL6IEC2F001")
        get_ct7481_data '0x0' 'inlet'
        ;;
        "STL6SPCIA")
        get_ct7481_data '0x0' 'inlet'
        ;;
        *)
        echo "Current environment is ${now_envir}, not dpu environment."
        exit 1
        ;;
    esac
}


# 操作ct7481温度传感器获取温度,$1表示偏移地址，$2表示获取的温度类型(内存,进风口)
function get_ct7481_data()
{
    local temp=$(bsptool -c readiic 0 ${dpu_s_ct7481_addr} $1 0x1 1)
    local isFailed=$(echo $temp | grep "failed" -c)
    if [ $isFailed -ge 1 ]; then
        temparray["$2"]='invalid'
    else
        if [ $((temp)) -gt 0 -a $((temp)) -lt 128 ]; then
            temparray["$2"]=$((temp)) 
        else
            temparray["$2"]='invalid'
        fi
    fi
}

# 操作LM75温度传感器获取温度,$1表示偏移地址，$2表示获取的温度类型(内存,进风口)
# 温度值为两字节数据，取高11位为温度数据，其中最高位为0表示正温度，1表示负温度
# 正温度为高11位数据值除以8，负温度为高11位数据的补码除以8
function get_lm75_data()
{
    # LM75芯片手册:在上电时，第一次温度读取会不正确,必须被舍弃,连续读取温度是正确的
    local temp=$(bsptool -c readiic 0 ${dpu_s_lm75_addr} $1 0x1 2)
    local temp=$(bsptool -c readiic 0 ${dpu_s_lm75_addr} $1 0x1 2)
    isFailed=$(echo $temp | grep "failed" -c)
    if [ $isFailed -ge 1 ]; then
        temparray["$2"]='invalid'
        return
    fi
    temp=(${temp})
    # 判断最高位是否为0，0为正温度，1为负温度
    Ispositive=$((((${temp[0]}>>7))&1))
    if [ ${Ispositive} -eq 0 ]; then
        # 温度为正数
        if [ $((temp[0])) -gt 0 -a $((temp[0])) -lt 128 ];then
            temparray["$2"]=$((temp[0]))
        else
            temparray["$2"]='invalid'
        fi
    else
        # 温度为负数
        # 高八位温度,temp[0]
        temp[0]=$(((1<<8)- 1 -$((temp[0]))))
        # 低三位温度,temp[1]
        temp[1]=$((($((1<<3))-$(($((temp[1]))>>5)))/8))
        inlet_temp=$(((${temp[0]}+${temp[1]})*(-1)))
        if [ ${inlet_temp} -gt -64 -a ${inlet_temp} -lt 0 ];then
            temparray["$2"]=${inlet_temp}
        else
            temparray["$2"]='invalid'
        fi
    fi
}

# 写入温度查询结果到临时文件
function set_dpu_temrature_file()
{
    if [ ! -e "${DPU_TEMPRATURE_FILE_PATH}" ]; then
        touch ${DPU_TEMPRATURE_FILE_PATH}
        chmod 640 ${DPU_TEMPRATURE_FILE_PATH}
    fi
    # 临时文件当前只有产品模块使用，约定以#作为结束符，防止未写完文件，产品模块就进行读取
    echo "$1#" > ${DPU_TEMPRATURE_FILE_PATH}
}

function return_temp()
{
    get_temp
    if [[ "${cpu_result}" = '' ]]; then
        echo "inlet=${temparray['inlet']},ddr=${temparray['ddr']}"
        set_dpu_temrature_file "inlet=${temparray['inlet']},ddr=${temparray['ddr']}"
    else
        echo "${cpu_result},inlet=${temparray['inlet']},ddr=${temparray['ddr']}"
        set_dpu_temrature_file "${cpu_result},inlet=${temparray['inlet']},ddr=${temparray['ddr']}"
    fi
}

return_temp
exit 0
