# ********************************************** ***************************** #
# File Name: log_config.sh
# Description: 修改rsyslog和logrotate的配置
# **************************************************************************** #
#! /bin/bash

# 变量定义
# rsyslog配置文件名
RSYSLOG_CONF_FILENAME="pangeaBspRsyslog.conf"
# rsyslog配置文件路径
SYSLOG_CONFIG_DIR="/etc/rsyslog.d"
# shell脚本日志打印的文件
LOG_FILE="/var/log/pangea_internal_upgrade/base_upgrade.log"
# 脚本日志打印前缀
LOG_PREFIX="[PANGEA BSP]"
# logrorate配置文件名
LOGRORATE_CONF_FILENAME="pangeaBspLogRorate.conf"
# logrotate配置文件路径
LOGRORATE_CONFIG_DIR="/etc/logrotate.d"

# **************************************************************************** #
# Function Name: force_check_pre_step
# Description: 检查上一步是否返回0，失败直接退出
# Parameter: $1 当前操作的描述
# **************************************************************************** #
function force_check_pre_step()
{
    if [ $? -ne 0 ]
    then
        print_log "[ERR] $1 failed."
        exit 1
    fi
    print_log "[INFO] $1 success."
    return 0
}

# **************************************************************************** #
# Function Name: exec_and_retry
# Description: 重试 $2 次 函数 $1
# Parameter: $1 回调函数 $2 重试次数 $3 当前操作的描述
# **************************************************************************** #
function exec_and_retry()
{
    local i=0
    local ret=0
    while [ $i -le $2 ]
    do
        let "i++"
        eval "$1"
        ret=$?
        # 如果执行成功就退出循环
        if [ $ret -eq 0 ]
        then
            print_log "[INFO] $3 success."
            return 0
        fi
        print_log "[WARN] $3 exec failed, ret: $ret, retry..."
    done
    print_log "[ERR] $3 tried $2 times failed, exit."
    return 0
}

# **************************************************************************** #
# Function Name: print_log
# Description: 此shell脚本打印日志
# Parameter: $1 要打印的字符串内容
# **************************************************************************** #
function print_log()
{
    # 打印屏显
    echo "$1"
    # 日志文件存在才打印到文件中
    if [ -e ${LOG_FILE} ]
    then
        # 打印到 脚本日志文件中
        echo "${LOG_PREFIX}"$1 >> ${LOG_FILE}
    fi
}

# **************************************************************************** #
# Function Name: set_syslog_config
# Description: 修改rsyslog配置
# **************************************************************************** #
function set_syslog_config()
{
    print_log "[INFO] start set syslog config."
    # 安装rpm时的配置文件路径
    local rsyslog_conf_filepath=$(readlink -f ${RSYSLOG_CONF_FILENAME})

    # bsp 属于内核态安装的模块，直接拷贝日志配置，reboot生效
    if [ -e $SYSLOG_CONFIG_DIR ]
    then
        # 将配置文件拷贝到/etc/rsyslog.d/
        cp -rf ${rsyslog_conf_filepath} ${SYSLOG_CONFIG_DIR}
        force_check_pre_step "cp ${rsyslog_conf_filepath} to ${SYSLOG_CONFIG_DIR}"
        # 修改权限
        chmod 644 "${SYSLOG_CONFIG_DIR}/${RSYSLOG_CONF_FILENAME}"
    else
        print_log "[ERR] rsyslog config dir: ${SYSLOG_CONFIG_DIR} not exist."
    fi
    return 0
}

# **************************************************************************** #
# Function Name: set_logrotate_config
# Description: 修改logrotate配置
# **************************************************************************** #
function set_logrotate_config()
{
    print_log "[INFO] start set logrotate config."
    # 安装rpm时的配置文件路径
    local logrotate_conf_filepath=$(readlink -f ${LOGRORATE_CONF_FILENAME})

    # bsp 属于内核态安装的模块，直接拷贝日志配置，reboot生效
    if [ -e $LOGRORATE_CONFIG_DIR ]
    then
        # 将配置文件拷贝到/etc/logrotate.d
        cp -rf ${logrotate_conf_filepath} ${LOGRORATE_CONFIG_DIR}
        force_check_pre_step "cp logrotate ${logrotate_conf_filepath} to ${LOGRORATE_CONFIG_DIR}"
        # 修改权限
        chmod 644 "${LOGRORATE_CONFIG_DIR}/${LOGRORATE_CONF_FILENAME}"
    else
        print_log "[ERR] logrotate config dir: ${LOGRORATE_CONFIG_DIR} not exist."
    fi
    return 0
}

# **************************************************************************** #
# Function Name: add_config
# Description: 增加rsyslog和logrotate配置
# **************************************************************************** #
function add_config()
{
    print_log "[INFO] start to add config."
    # 修改rsyslog配置
    set_syslog_config
    force_check_pre_step "set syslog config"
    # 修改logrotate配置
    set_logrotate_config
    force_check_pre_step "set logrotate config"
    print_log "[INFO] add config success."
    return 0
}

# **************************************************************************** #
# Description: 程序入口
# **************************************************************************** #
while getopts ":a" opt
do
    case "${opt}" in
        a)
            # 增加配置
            add_config
            exit 0
            ;;
        *)
            exit 1
            ;;
    esac
done
