#!/usr/bin/env bash
# install net rpm lib
# Copyright © Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

LOG_DIR="/var/log/pangea_internal_upgrade/net"
OPERATE_LOG="$LOG_DIR/operate_log"
PROGRESS_RATE_FILE="$LOG_DIR/net_progress"

STATUS_OK=0
STATUS_FAIL=1
STATUS_ENVIRONMENT_NOT_MATCH=2
STATUS_NO_FILE=3
STATUS_NO_PERMISSION=4

STATUS_CHECK_FAIL=2
STATUS_NEED_UPDATE=1
STATUS_NO_NEED_UPDATE=0

# operate_log - record log
function operate_log()
{
    [[ ! -d ${LOG_DIR} ]] && mkdir -p -m 0750 $LOG_DIR
    if [[ $2 == message ]]; then
        echo "$1" >> /dev/kmsg
    fi
    echo "[$(date '+%Y-%m-%d %H:%M:%S')][$1]" >> "$OPERATE_LOG"
}

# print - show information in shell, 
# There will be no echo unless print is called when using install.sh
function print()
{
    echo "$@" >&3
    operate_log "$@"
}

# pangea_upgrade_progress - update background upgrade progress file
# $1 progress info -- $1 should be success or contains keywords upgrading/failure
function pangea_upgrade_progress()
{
    echo $1 > $PROGRESS_RATE_FILE
    operate_log "$@"
}

# pangea_net_install - install rpm
# $1 - rpm name
function pangea_net_install()
{
    local rpm_name=$1
    local driver=""
    local retry_times=0

    operate_log "Install ${rpm_name} start..."
    if ! ls ${rpm_name}*.rpm >> $OPERATE_LOG 2>&1; then
        operate_log "Install failed, no ${rpm_name} in $(pwd) found!!!"
        return $STATUS_NO_FILE
    fi

    # rpm包安装失败最多重试3次
    while (( ${retry_times} < 3 )); do
        if rpm -ivh ${rpm_name}*.rpm >> $OPERATE_LOG 2>&1; then
            operate_log "Install pangea net ${rpm_name} succeed"
            return ${STATUS_OK}
        fi
        operate_log "Install pangea net ${rpm_name} rpm failed retry:${retry_times}."
        (( retry_times++ ))
    done;

    return ${STATUS_FAIL}
}

# pangea_net_uninstall - uninstall rpm
# $1 - rpm name
function pangea_net_uninstall()
{
    local rpm_name=$1

    operate_log "Uninstall ${rpm_name} start..."

    if rpm -q $rpm_name >> $OPERATE_LOG 2>&1;then
        rpm -e --nodeps $rpm_name >> $OPERATE_LOG 2>&1
        if [ $? -ne $STATUS_OK ];then
            operate_log "uninstall $rpm_name fail!"
            return $STATUS_FAIL
        else
            operate_log "uninstall $rpm_name success!"
            return $STATUS_OK
        fi
    else
        operate_log "No $rpm_name driver installed now."
        return $STATUS_OK
    fi
}

# 获取当前驱动版本
function pangea_net_is_need_update_rpm()
{
    local rpm_name=$1
    local rpm_path=$2

    operate_log "upgrade pangea net get $rpm_name version..."

    if ! rpm  -q $rpm_name; then
        operate_log "No $rpm_name installed now."
        return $STATUS_NEED_UPDATE
    fi

    if ! ls $rpm_name*.rpm; then
        operate_log "No $rpm_name in $(pwd)."
        return $STATUS_CHECK_FAIL
    fi

    local old_rpm_info=$(rpm -qi $rpm_name)
    local new_rpm_info=$(rpm -qi ${rpm_path}/${rpm_name}*.rpm)
    for commit_num_type in $(grep -Eo ".*commit_num" <<< ${new_rpm_info}); do
        local commit_num_old=$(awk -v commit_num_type=${commit_num_type} '$0 ~ commit_num_type {print $NF}' <<< ${old_rpm_info})
        local commit_num_new=$(awk -v commit_num_type=${commit_num_type} '$0 ~ commit_num_type {print $NF}' <<< ${new_rpm_info})
        operate_log "commit_num_type:${commit_num_type} old:${commit_num_old} new:${commit_num_new}"
        if [[ "${commit_num_old}" != "${commit_num_new}" ]]; then
            return $STATUS_NEED_UPDATE
        fi
    done

    return $STATUS_NO_NEED_UPDATE
}

function help()
{
    local opt="$1"

    print "invaild opt:$opt"
    print "usage:"
    print "bash install.sh <option>"
    print "option:"
    print "getver:   get install driver pkg version,"
    print "          if echo NOT_MATCH is mean this driver package can not install in this machine"
    print "          if echo -- is mean this driver package never installed on this machine"
    print "upgrade:          upgrade driver package"
    print "getupgraderate:   get upgrade progress, when echoed success is mean upgrade is complete"
}

# net_get_card_fw_version - get card firmwall version
# input: 
# $1 - bus number
# output(define at the calling script): 
# eth - ethernet name
# card_fw_version - card firmwall version
function net_get_card_fw_version()
{
    local bus_num="$1"
    card_type

    local bus_info=$(lspci -s ${bus_num})
    card_type=$(grep -wo -E -- "$CARD_TYPE_LIST" <<< "$bus_info")
    if [[ -z "$card_type" ]]; then
        operate_log "Invalid card type, bus_info: $bus_info"
        return $STATUS_FAIL
    fi

    new_rpm_fw_version=${FW_VERSION[$card_type]}

    local link=$(find /sys/class/net -maxdepth 1 -type l -exec readlink -e {} \; | grep -w ${bus_num})
    eth=$(basename $link)
    card_fw_version=$(ethtool -i $eth | sed -n 's/firmware-version: //p')
    return $STATUS_OK
}

# pangea_net_relog - 日志转储
# 驱动固件包部署完成后,重启日志转储服务和定时服务,使能驱动内核日志转储功能
function pangea_net_relog()
{
    if ! systemctl restart rsyslog.service; then
        pangea_upgrade_progress "failure:Failed restart rsyslog ret=$?"
        return $STATUS_FAIL
    fi

    if ! systemctl restart crond.service; then
        pangea_upgrade_progress "failure:Failed restart crond ret=$?"
        return $STATUS_FAIL
    fi

    return $STATUS_OK
}

# is_strict_kernel_version - 判断当前欧拉环境是否为小版本兼容版本
# 欧拉内核在v2r10及以后的版本是小版本兼容版本
# 小版本：5.10.0-60.18.0.50.h725.eulerosv2r11.aarch64,其中h725是小版本
# 大版本：5.10.0-60.18.0.50.h725.eulerosv2r11.aarch64,其中v2r11是大版本
# 小版本兼容: 当大版本相同相同时,即使小版本不同的镜像编译出来的内核模块也能互相加载
function is_strict_kernel_version()
{
    HOST_EULER_VERSION=($(uname -r | sed -r 's/.*eulerosv([0-9]+)r([0-9]+).*/\1 \2/'))
    [[ ${HOST_EULER_VERSION[0]} < 2 ]] && return 0
    (( ${HOST_EULER_VERSION[0]} == 2 && ${HOST_EULER_VERSION[1]} < 10 )) && return 0
    return 1
}

# 重做小镜像,重做后小镜像为/boot/initramfs-$(uname -r).img
# 该动作时间较长1min左右
function rebuild_initramfs()
{
    local ret;
    operate_log "begin rebuild initramfs"
    depmod "$(uname -r)" -a>> ${OPERATE_LOG} 2>&1
    ret=$?
    if [[ $ret != 0 ]]; then
        operate_log "depmod fail ret:$ret"
        return $ret;
    fi

    # dracut -f命令永远返回0,不会返错
    dracut -f >> ${OPERATE_LOG} 2>&1
    operate_log "end rebuild initramfs"
}

# 重做小镜像检查,检查驱动是否打入小镜像内
# 该动作时间较长30s左右
function initramfs_check()
{
    local driver_path_list="$1"
    local driver_path

    for driver_path in ${driver_path_list//|/ }; do
        if ! lsinitrd | grep -q "$driver_path"; then
            pangea_upgrade_progress "Install pangea net rpm failed, $driver_path not in initramfs."
            return $STATUS_FAIL
        fi
    done

    return $STATUS_OK
}

# 部署board包
function deploy_board_pkg()
{
    local board_pkg="$1"
    local check_ret
    local retry_times=0

    if [[ ! -f ${WORK_DIR}/board/${board_pkg}.zip ]]; then
        operate_log "no ${board_pkg} in ${WORK_DIR}/board"
        return $STATUS_NO_FILE
    fi

    while (( ${retry_times} < 3 )); do
        unzip -o ${WORK_DIR}/board/${board_pkg}.zip -d ${WORK_DIR}/${board_pkg}/ >> ${OPERATE_LOG} 2>&1
        check_ret=$?
        if [ ${check_ret} == 0 ]; then
            operate_log "deploy pkg ${WORK_DIR}/board/${board_pkg}.zip succeed"
            PKG_DIR[$board_pkg]="${WORK_DIR}/${board_pkg}"
            return $STATUS_OK
        fi
        operate_log "unzip ${WORK_DIR}/board/${board_pkg}.zip fail ret:$check_ret retry_times=${retry_times}"
        (( retry_times++ ))
    done;

    return $STATUS_FAIL;
}

# 部署driver包
function deploy_driver_pkg()
{
    local pkg="$1"
    local check_ret
    local retry_times=0

    if [[ ! -f ${ROOT_DIR}/pkg_pool/${pkg}.zip ]]; then
        operate_log "deploy fail no ${pkg}.zip in ${ROOT_DIR}/pkg_pool"
        return $STATUS_NO_FILE
    fi

    while (( ${retry_times} < 3 )); do
        unzip -o ${ROOT_DIR}/pkg_pool/${pkg}.zip -d ${WORK_DIR}/${pkg}/ >> ${OPERATE_LOG} 2>&1
        check_ret=$?
        if [ ${check_ret} == 0 ]; then
            operate_log "deploy pkg ${WORK_DIR}/pkg_pool/${pkg}.zip succeed"
            PKG_DIR[$pkg]="${WORK_DIR}/${pkg}"
            return $STATUS_OK
        fi
        operate_log "unzip ${ROOT_DIR}/pkg_pool/${pkg}.zip fail ret:$check_ret retry_times=${retry_times}"
        (( retry_times++ ))
    done;

    return $STATUS_FAIL;
}