# -*-coding:utf-8-*-
# python3.x
import subprocess
import os

from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.DBAdapter.DBConnector import BaseOps
import utils.common.log as logger
from utils.common.fic_common import GetConfig
from utils.common.ssh_util import Ssh
from utils.security.crypt import decrypt
from plugins.DistributedStorage.utils.common.DeployConstant import DeployConstant
from plugins.DistributedStorage.logic.DeployOperate import DeployOperate


class DistributedStorageArgs(object):
    def __init__(self, project_id, pod_id, storage=None, condition=None, metadata=None, **kwargs):
        self.project_id = project_id
        self.pod_id = pod_id
        self.storage = storage
        self.condition = condition
        self.metadata = metadata
        self.more_args = kwargs
        self.params = ParamUtil()
        self.db = BaseOps()

    def get_manage_storage_float_ip(self):
        """
        :function 获取管理存储浮动IP
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get ManageStorFB80 Parameters")
        manage_float_ip = self.params.get_value_from_cloudparam(
            self.pod_id, "DistributedStorageManage", "FSMOMFloatingIP")
        logger.info("Returned DistributeStorage manage storage floatIP[%s]" % manage_float_ip)
        return manage_float_ip

    def get_ha_manage_storage_float_ip(self):
        """
        :function 获取高可用备管理存储浮动IP
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get ManageStorFB80 Parameters")
        ha_manage_float_ip = self.params.get_value_from_cloudparam(
            self.pod_id, "DistributedStorageManage", "HAFSMOMFloatingIP")
        logger.info("Returned DistributeStorage manage storage floatIP[%s]" % ha_manage_float_ip)
        return ha_manage_float_ip

    def get_business_storage_float_ip(self):
        """
        :function 获取业务存储浮动IP
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get TenantStorFB80 Parameters")
        if get_project_condition_boolean(self.project_id, '(TenantStorFB80|TenantStorFBHCI80)&ProjectDeploy'):
            logger.info("Get (TenantStorFB80|TenantStorFBHCI80)&ProjectDeploy Parameters")
            business_float_ip = self.params.get_value_from_cloudparam(
                self.pod_id, "DistributedStorageBusiness", "FSMOMFloatingIP02")
        elif get_project_condition_boolean(self.project_id, 'TenantStorFB80&(ExpansionAZ_KVM|ExpansionAZ_BMS)'):
            logger.info("Get TenantStorFB80&(ExpansionAZ_KVM|ExpansionAZ_BMS) Parameters")
            business_float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "expansion_new_fsm_float_ip")
        elif get_project_condition_boolean(self.project_id, 'TenantStorFB80&ExpansionServiceStorage'):
            logger.info("Get TenantStorFB80&ExpansionServiceStorage Parameters")
            business_float_ip = self.db.get_user_input_cloud_param_by_key(
                self.project_id, "expansion_fusionstorage_float_ip")
        elif get_project_condition_boolean(self.project_id,
                                           '((TenantStorFBReuse80|TenantStorHCIFS800Reuse)&ExpansionAZ_KVM)|'
                                           'ExpansionServiceStorage&(TenantStorNewPool|TenantStorNewNode)'):
            logger.info("Get ((TenantStorFBReuse80|TenantStorHCIFS800Reuse)&ExpansionAZ_KVM)"
                        "|ExpansionServiceStorage&(TenantStorNewPool|TenantStorNewNode) Parameters")
            business_float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "expansion_az_fsm_float_ip")
        elif get_project_condition_boolean(self.project_id,
                                           '(ExpansionAZ_BMS&TenantStorFBReuse80)|'
                                           '(ExpansionComputeRes_KVMNode|ExpansionNetworkRes_NetworkNode)&'
                                           '(TenantStorFB80|TenantStorFBHCI80)|TenantStorFBPoolReuse80'):
            logger.info("Get (ExpansionAZ_BMS&TenantStorFBReuse80)|"
                        "(ExpansionComputeRes_KVMNode|ExpansionNetworkRes_NetworkNode)&"
                        "(TenantStorFB80|TenantStorFBHCI80)|TenantStorFBPoolReuse80 Parameters")
            business_float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_float_ip")
        else:
            logger.info("Get other Parameters")
            business_float_ip = None
        logger.info("Returned DistributeStorage business storage floatIP[%s]" % business_float_ip)
        return business_float_ip

    def get_init_fsm_portal_pwd(self):
        """
        :function 获取初始的界面密码
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get  FSM default portal Parameters")
        init_fsm_portal_pwd = self.params.get_value_from_cloudparam(
            self.pod_id, "DistributedStorage", "DMPortalInitPassword")
        if not init_fsm_portal_pwd:
            logger.info("Failed to get DistributeStorage init fsm portal psw")
        else:
            logger.info("Succeed to get DistributeStorage init fsm portal psw")
        return init_fsm_portal_pwd

    def get_default_fsm_portal_pwd(self):
        """
        :function 获取默认的界面密码
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get  FSM default portal Parameters")
        default_portal_pwd = self.params.get_value_from_cloudparam(
            self.pod_id, "DistributedStorage", "FSMPortalPassword")
        if not default_portal_pwd:
            logger.info("Failed to get DistributeStorage default portal psw")
        else:
            logger.info("Succeed to get DistributeStorage default portal psw")
        return default_portal_pwd

    def get_default_fsm_fsadmin_pwd(self):
        """
        :function 获取默认的fsadmin密码
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get  FSM default portal Parameters")
        fsm_fsadmin_pwd = self.params.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMfsdminPassword")
        if not fsm_fsadmin_pwd:
            logger.info("Failed to get DistributeStorage fsm fsadmin psw")
        else:
            logger.info("Succeed to get DistributeStorage fsm fsadmin psw")
        return fsm_fsadmin_pwd

    def get_default_fsm_root_pwd(self):
        """
        :function 获取默认的root密码
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get  FSM default portal Parameters")
        fsm_root_pwd = self.params.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMrootPassword")
        if not fsm_root_pwd:
            logger.info("Failed to get DistributeStorage default fsm root psw")
        else:
            logger.info("Succeed to get DistributeStorage default fsm root psw")
        return fsm_root_pwd

    def get_default_fsm_dsware_pwd(self):
        """
        :function 获取默认的dsware密码
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get  FSM default portal Parameters")
        fsm_dsware_pwd = self.params.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSdswarePassword")
        if not fsm_dsware_pwd:
            logger.info("Failed to get DistributeStorage default fsm dsware psw")
        else:
            logger.info("Succeed to get DistributeStorage default fsm dsware psw")
        return fsm_dsware_pwd

    def get_new_fsm_fsadmin_pwd(self):
        """
        :function 获取默认的new fsadmin密码
        :param: pod_id
        :return: 返回float_ip，如果不存在，返回空
        """
        logger.info("Get  FSM default portal Parameters")
        new_fsm_fsadmin_pwd = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_fsadmin_passwd")
        if not new_fsm_fsadmin_pwd:
            logger.info("Failed to get DistributeStorage default fsm new fsadmin psw")
        else:
            logger.info("Succeed to get DistributeStorage default fsm new fsadmin psw")
        return new_fsm_fsadmin_pwd

    def get_business_fsm_portal_pwd(self):
        """
        获取所有业务分离场景的业务存储fsm portal密码
           部署工程、扩az新建、IPSAN扩分布式存储，使用默认密码
           扩az复用，新增业务存储--新增存储节点/存储池，扩计算节点，扩计算az复用分布式存储，获取LLD表填写的密码
        """
        portal_pwd = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMPortalPassword")
        if get_project_condition_boolean(
                self.project_id,
                'ExpansionAZ_KVM&TenantStorFBPoolReuse80'
                '|TenantStorFB80&(ExpansionComputeRes_KVMNode|ExpansionNetworkRes_NetworkNode)'
                '|TenantStorFBReuse80&(ExpansionAZ_KVM|ExpansionAZ_BMS)'
                '|ExpansionServiceStorage&(TenantStorNewPool|TenantStorNewNode)'):
            portal_pwd = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_admin_passwd")
        if not portal_pwd:
            logger.error('Failed to get fsm portal info')
        return portal_pwd

    def get_rep_local_admin_pwd(self):
        """
        全新建场景，使用默认密码，追加生产、灾备端使用LLD表填写的密码
        """
        # 追加生产端：
        exp_product = get_project_condition_boolean(
            self.project_id,
            'DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB)&DRStorage_TFB_Sep'
            '&(ExpansionAdCloudService|ExpansionServiceStorage)'
        )
        # 追加灾备端：
        exp_disaster = get_project_condition_boolean(
            self.project_id,
            '!DRStorage_TFB_PD&(CSHAStorage_TFB|CSDRStorage_TFB)&DRStorage_TFB_Sep'
            '&(ExpansionAdCloudService|ExpansionServiceStorage)'
        )
        if exp_product:
            local_admin_password = self.params.get_param_value(
                self.pod_id, "FusionStorageBlockReplication", 'produce_storage_fsm_admin_password')
        elif exp_disaster:
            local_admin_password = self.params.get_param_value(
                self.pod_id, "FusionStorageBlockReplication", 'disaster_storage_fsm_admin_password')
        else:
            local_admin_password = self.db.get_value_from_cloudparam(
                self.pod_id, "DistributedStorage", "FSMPortalPassword")
        if not local_admin_password:
            logger.error('Failed to get local admin info')
        return local_admin_password

    def get_rep_net_info(self, rep_index="1"):
        """
        获取复制网络信息
        :param rep_index:
        :return:
        """
        fs_args = dict()
        fs_args["port_name"] = "rep1" if rep_index == "1" else "rep2"
        fs_args["net_info"] = self.get_net_info_from_lld('fusionstorage_rep', 'DistributedStorageBusiness',
                                                         index=rep_index)
        fs_args["net_plane"] = 'replication_plane'
        fs_args["nic_bond_mode"] = 'active_standby'
        fs_args["vlan_flag"] = False
        fs_args["port_index"] = 1 if rep_index == "1" else 2
        fs_args["node_role"] = "rep"
        return fs_args

    def get_net_info_from_lld(self, network_name, condition="DistributedStorageBusiness", index="1"):
        param_util = ParamUtil()
        _tag = param_util.get_param_value(self.pod_id, condition, network_name + "_tag" + index)
        _vlan = param_util.get_param_value(self.pod_id, condition, network_name + "_vlan" + index)
        _vlan_id = param_util.get_param_value(self.pod_id, condition, network_name + "_vlan_id" + index)
        if _tag:
            vlan = _tag
        elif _vlan_id:
            vlan = _vlan_id
        else:
            vlan = _vlan

        ip_range = param_util.get_param_value(self.pod_id, condition, network_name + "_range" + index)
        if not ip_range:
            return {}

        net_info_dic = {
            "ip_start": ip_range.split("-")[0],
            "gateway": param_util.get_param_value(self.pod_id, condition, network_name + "_gateway" + index),
            "netmask": param_util.get_param_value(self.pod_id, condition, network_name + "_netmask" + index),
            "name": network_name,
            "vlan": vlan
        }

        return net_info_dic


class DistributedStorageTool(object):
    def __init__(self, project_id, pod_id, db):
        self.project_id = project_id
        self.pod_id = pod_id
        self.db = db
        logger.init('StorageTool Util Init')

    def get_fsc_cli_cert_file(self, return_path, float_ip, storage_info=None):
        """
        获取APIG的API注册工具的服务名称和版本
        :param return_path:
        :param float_ip:
        :param storage_info: 默认为None,表示该场景为分布式存储部署工程；其他场景调用接口需要传参，参数格式：
        {"float_ip": "xxx.xxx.xxx.xxx", "portal_init_pwd": "******", "portal_pwd": "******",
        "fsadmin_pwd": "******", "dsware_pwd": "******", "root_pwd": "******"}
        :return: dict{'ca_file': ca证书文件路径,
                     'cert_file': 服务证书文件路径,
                     'key_file': 私钥文件路径,
                     'password': 加密口令}
        """
        if not os.path.exists(return_path):
            subprocess.getstatusoutput("mkdir -m 700 -p %s" % return_path)
        opr = DeployOperate(float_ip=float_ip)
        if storage_info is None:
            storage_cluster_list = self.get_storage_float_ip()
            for storage_cluster in storage_cluster_list:
                if storage_cluster.get("float_ip") == float_ip:
                    storage_info = storage_cluster
                    break
        if storage_info is None:
            err_msg = "Failed to get storage_info, please make sure params correct!"
            logger.error(err_msg)
            raise Exception(err_msg)
        status_code, error_code, error_des = opr.login(DeployConstant.DM_LOGIN_USER, storage_info.get("portal_pwd"))
        if status_code != 200 or error_code != 0:
            err_msg = "Failed to login deploy manager, Detail:[status:%s,code:%s]%s" % \
                      (status_code, error_code, error_des)
            logger.error(err_msg)
            raise Exception(err_msg)

        rsp_result, rsp_data = opr.get_product()
        if rsp_result.get("code") != 0:
            err_msg = "Failed to get product, Detail:[result:%s]%s" % (rsp_result, rsp_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        model = rsp_data.get("product_model")
        if model == "OceanStor Pacific":
            conf_obj = GetConfig(path='/plugins/DistributedStorage/utils/common/config.ini')
            password = decrypt(conf_obj.get_item('default_pwd', 'cert_pwd'))
            rsp_result, rsp_data = opr.get_cert(password)
            fsadmin_pwd = storage_info.get("fsadmin_pwd")
            rsp_cert_path = rsp_data.get("EXPORT_FSM_PEM_PATH")
            cert_file_dst_name = os.path.join(return_path, "cert_file.cert")
            self._up_cert(float_ip, fsadmin_pwd, rsp_cert_path, cert_file_dst_name)
            rsp_ca_path = rsp_data.get("EXPORT_CA_PEM_PATH")
            ca_file_dst_name = os.path.join(return_path, "ca_file.ca")
            self._up_cert(float_ip, fsadmin_pwd, rsp_ca_path, ca_file_dst_name)
            rsp_key_path = rsp_data.get("EXPORT_FSM_KEY_PATH")
            key_file_dst_name = os.path.join(return_path, "key_file.ca")
            self._up_cert(float_ip, fsadmin_pwd, rsp_key_path, key_file_dst_name)
            cert_info = {'ca_file': ca_file_dst_name,
                         'cert_file': cert_file_dst_name,
                         'key_file': key_file_dst_name,
                         'password': password}
            self.del_cert_file(opr)
            return cert_info
        else:
            raise Exception("Get fsc cli cert file Failed, Detail:does not support the current product model")

    @staticmethod
    def del_cert_file(opr):
        """
        :param opr:
        :return:
        """
        try:
            opr.del_cert()
        except Exception as e:
            logger.error("Del cert file failed, Detail: %s" % e)

    @staticmethod
    def _up_cert(float_ip, fsadmin_pwd, cert_file, dst_file):
        if Ssh.get_file(host=float_ip, user='fsadmin', passwd=fsadmin_pwd, source=cert_file,
                        destination=dst_file):
            logger.info("success")

    def get_storage_float_ip(self, az_mode=None, reuse_mode=None, dc=None):
        """
        :param az_mode: 默认 None 表示所有，manager 表示管理存储， business 表示业务存储, rep表示复制
        :param reuse_mode: 默认 None 表示所有，reuse 表示复用， new 表示新建
        :param dc: 默认 None 表示未用管理高可用， master表示管理高可用的主dc，standy 表示管理高可用的备dc
        :return: [{"float_ip": "xxx.xxx.xxx.xxx", "portal_init_pwd": "******", "portal_pwd": "******",
                   "fsadmin_pwd": "******", "dsware_pwd": "******", "root_pwd": "******"}]
        """
        storage_infos = []
        fsm_pwd_dic = self._get_fsm_all_pwd_info()
        if not az_mode or az_mode == "manager":
            logger.info('Get manager storage info')
            self._get_master_storage_info(dc, fsm_pwd_dic, reuse_mode, storage_infos)
            self._get_standby_storage_info(dc, fsm_pwd_dic, reuse_mode, storage_infos)

        if not az_mode or az_mode == "business":
            logger.info('Get business storage info')
            self._get_business_storage_info(fsm_pwd_dic, reuse_mode, storage_infos)

        self._get_rep_storage_infos(az_mode, fsm_pwd_dic, storage_infos)
        return storage_infos

    def _get_business_storage_info(self, fsm_pwd_dic, reuse_mode, storage_infos):
        if not reuse_mode or reuse_mode == "new":
            logger.info('Get business storage info, reuse mode:{}'.format(reuse_mode))
            float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "expansion_new_fsm_float_ip")
            if float_ip:
                storage_info = {"float_ip": float_ip, reuse_mode: "new"}
                storage_info.update(fsm_pwd_dic)
                storage_infos.append(storage_info)
            float_ip = self.db.get_user_input_cloud_param_by_key(
                self.project_id, "expansion_fusionstorage_float_ip")
            if float_ip:
                storage_info = {"float_ip": float_ip, reuse_mode: "new"}
                storage_info.update(fsm_pwd_dic)
                storage_infos.append(storage_info)
            float_ip = self.db.get_all_data_with_key_value(
                self.pod_id, "DistributedStorageBusiness").get('FSMOMFloatingIP02')
            if float_ip:
                storage_info = {"float_ip": float_ip}
                storage_info.update(fsm_pwd_dic)
                storage_infos.append(storage_info)
        if not reuse_mode or reuse_mode == "reuse":
            logger.info('Get business storage info, reuse mode:{}'.format(reuse_mode))
            float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "expansion_az_fsm_float_ip")
            if float_ip:
                portal_pwd = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_admin_passwd")
                storage_info = {"float_ip": float_ip, reuse_mode: "reuse"}
                storage_info.update(fsm_pwd_dic)
                storage_info["portal_pwd"] = portal_pwd
                storage_infos.append(storage_info)

            float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_float_ip")
            if float_ip:
                portal_pwd = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_admin_passwd")
                storage_info = {"float_ip": float_ip, reuse_mode: "reuse"}
                storage_info.update(fsm_pwd_dic)
                storage_info["portal_pwd"] = portal_pwd
                storage_infos.append(storage_info)

    def _get_master_storage_info(self, dc, fsm_pwd_dic, reuse_mode, storage_infos):
        logger.info('Get {} storage info, reuse mode:{}'.format(dc, reuse_mode))
        if not dc or dc == "master":
            if not reuse_mode or reuse_mode == "new":
                float_ip = self.db.get_all_data_with_key_value(
                    self.pod_id, "DistributedStorageManage").get('FSMOMFloatingIP')
                if float_ip:
                    storage_info = {"float_ip": float_ip, reuse_mode: "new"}
                    storage_info.update(fsm_pwd_dic)
                    storage_infos.append(storage_info)
            if not reuse_mode or reuse_mode == "reuse":
                self._get_dc001_storage_info(fsm_pwd_dic, reuse_mode, storage_infos)
                self._get_expansion_storage_info(fsm_pwd_dic, reuse_mode, storage_infos)

    def _get_expansion_storage_info(self, fsm_pwd_dic, reuse_mode, storage_infos):
        logger.info('Get expansion storage info, reuse mode:{}'.format(reuse_mode))
        float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "expansion_fsm_float_ip")
        if float_ip:
            portal_pwd = self.db.get_user_input_cloud_param_by_key(
                self.project_id, "expansion_fs8_fsm_admin_passwd")
            storage_info = {"float_ip": float_ip, reuse_mode: "reuse"}
            storage_info.update(fsm_pwd_dic)
            storage_info["portal_pwd"] = portal_pwd
            storage_infos.append(storage_info)

    def _get_dc001_storage_info(self, fsm_pwd_dic, reuse_mode, storage_infos):
        logger.info('Get dc001 storage info, reuse mode:{}'.format(reuse_mode))
        float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id,
                                                             "dc001_expansion_fsm_float_ip")
        if float_ip:
            storage_info = {"float_ip": float_ip, reuse_mode: "reuse"}
            storage_info.update(fsm_pwd_dic)
            storage_info["portal_pwd"] = self.db.get_user_input_cloud_param_by_key(
                self.project_id, "dc001_expansion_fsm_admin_passwd")
            storage_infos.append(storage_info)

    def _get_standby_storage_info(self, dc, fsm_pwd_dic, reuse_mode, storage_infos):
        if not dc or dc == "standby":
            if not reuse_mode or reuse_mode == "new":
                float_ip = self.db.get_all_data_with_key_value(
                    self.pod_id, "DistributedStorageManage").get('HAFSMOMFloatingIP')
                if float_ip:
                    storage_info = {"float_ip": float_ip, reuse_mode: "new"}
                    storage_info.update(fsm_pwd_dic)
                    storage_infos.append(storage_info)
            if not reuse_mode or reuse_mode == "reuse":
                float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id,
                                                                     "dc002_expansion_fsm_float_ip")
                if float_ip:
                    storage_info = {"float_ip": float_ip, reuse_mode: "reuse"}
                    storage_info.update(fsm_pwd_dic)
                    storage_info["portal_pwd"] = self.db.get_user_input_cloud_param_by_key(
                        self.project_id, "dc002_expansion_fsm_admin_passwd")
                    storage_infos.append(storage_info)

    def _get_rep_storage_infos(self, az_mode, fsm_pwd_dic, storage_infos):
        logger.info('Get rep storage info')
        if not az_mode or az_mode == 'rep':
            produce_float_ip = self.db.get_user_input_cloud_param_by_key(self.project_id, "produce_storage_fsm_ip")
            if produce_float_ip:
                produce_storage_fsm_fsadmin_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "produce_storage_fsm_fsadmin_password")
                produce_storage_fsm_dsware_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "produce_storage_fsm_dsware_password")
                produce_storage_fsm_root_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "produce_storage_fsm_root_password")
                produce_storage_fsm_admin_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "produce_storage_fsm_admin_password")
                storage_info = {"float_ip": produce_float_ip,
                                "portal_init_pwd": fsm_pwd_dic.get("portal_init_pwd"),
                                "portal_pwd": produce_storage_fsm_admin_password,
                                "dsware_pwd": produce_storage_fsm_dsware_password,
                                "fsadmin_pwd": produce_storage_fsm_fsadmin_password,
                                "root_pwd": produce_storage_fsm_root_password,
                                "produce_storage_fsm_dsware_password": produce_storage_fsm_dsware_password,
                                "produce_storage_fsm_root_password": produce_storage_fsm_root_password,
                                "produce_storage_fsm_admin_password": produce_storage_fsm_admin_password,
                                "mode": "produce"}
                storage_infos.append(storage_info)
            disaster_float_ip = self.db.get_user_input_cloud_param_by_key(
                self.project_id, "disaster_tolerant_storage_fsm_ip")
            if disaster_float_ip:
                disaster_fsm_fsadmin_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "disaster_storage_fsm_fsadmin_password")
                disaster_fsm_dsware_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "disaster_tolerant_storage_fsm_dsware_password")
                disaster_fsm_root_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "disaster_tolerant_storage_fsm_root_password")
                disaster_fsm_admin_password = self.db.get_user_input_cloud_param_by_key(
                    self.project_id, "disaster_storage_fsm_admin_password")
                storage_info = {"float_ip": disaster_float_ip,
                                "portal_init_pwd": fsm_pwd_dic.get("portal_init_pwd"),
                                "portal_pwd": disaster_fsm_admin_password,
                                "dsware_pwd": disaster_fsm_dsware_password,
                                "fsadmin_pwd": disaster_fsm_fsadmin_password,
                                "root_pwd": disaster_fsm_root_password,
                                "disaster_tolerant_storage_fsm_dsware_password": disaster_fsm_dsware_password,
                                "disaster_tolerant_storage_fsm_root_password": disaster_fsm_root_password,
                                "disaster_storage_fsm_admin_password": disaster_fsm_admin_password,
                                "mode": "disaster"}
                storage_infos.append(storage_info)

    def _get_fsm_all_pwd_info(self):
        logger.info('Get fsm info')
        portal_init_pwd = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "DMPortalInitPassword")
        portal_pwd = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMPortalPassword")
        fsadmin_pwd = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMfsdminPassword")
        dsware_pwd = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSdswarePassword")
        root_pwd = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorage", "FSMrootPassword")
        fsadmin_pwd_new = self.db.get_user_input_cloud_param_by_key(self.project_id, "reuse_fsm_fsadmin_passwd")
        fsadmin_pwd = fsadmin_pwd_new if fsadmin_pwd_new else fsadmin_pwd
        pwd_dic = {"portal_init_pwd": portal_init_pwd,
                   "portal_pwd": portal_pwd,
                   "fsadmin_pwd": fsadmin_pwd,
                   "dsware_pwd": dsware_pwd,
                   "root_pwd": root_pwd}
        return pwd_dic
