# -*- coding: utf-8 -*-
import traceback
from IPy import IP
from utils.common.fic_base import TestCase
import utils.common.log as logger
from utils.common.message import Message
from utils.common.exception import FCDException
from plugins.DistributedStorage.scripts.logic.DeployOperate import DeployOperate
from plugins.DistributedStorage.scripts.utils.common.DeployConstant \
    import DeployConstant


class ConfigStorageNetwork(TestCase):
    def __init__(self, project_id, pod_id, fs_args, condition=None,
                 metadata=None, **kwargs):
        super(ConfigStorageNetwork, self).__init__(project_id, pod_id)
        self.opr = DeployOperate({'float_ip': fs_args.get('float_ip')})

        self.update_pwd = fs_args['dm_update_pwd']
        self.frontend_network = fs_args.get(
            'frontend_storage_network_list')
        self.backend_network = fs_args.get('backend_storage_network_list')
        self.fsa_list = fs_args.get('fsa_list')
        self.osd_list = fs_args.get('osd_list')

        self.condition = condition
        self.metadata = metadata
        self.more_args = kwargs

    def add_to_network_list(self, start_ip, end_ip, netmask, gateway,
                            network_list):
        ip_seg = dict()
        if not network_list:
            ip_seg['begin_ip'] = start_ip
            ip_seg['end_ip'] = end_ip
            ip_seg['net_mask'] = netmask
            ip_seg['gateway'] = gateway
            network_list.append(ip_seg)
            return network_list

        is_exist = False
        for network_ip_info in network_list:
            net_start_ip = network_ip_info.get('begin_ip')
            net_end_ip = network_ip_info.get('end_ip')
            if start_ip == net_start_ip and net_end_ip == end_ip:
                is_exist = True
                break

            start_ip_merge = self.get_start_ip_merge(net_start_ip, start_ip)

            end_ip_merge = self.get_end_ip_merge(net_end_ip, end_ip)

            if not start_ip_merge or not end_ip_merge:
                continue
            network_ip_info['begin_ip'] = start_ip_merge
            network_ip_info['end_ip'] = end_ip_merge
            if not network_ip_info.get('net_mask'):
                network_ip_info['net_mask'] = netmask
            if not network_ip_info.get('gateway'):
                network_ip_info['gateway'] = gateway
            is_exist = True
            break

        if not is_exist:
            ip_seg['begin_ip'] = start_ip
            ip_seg['end_ip'] = end_ip
            ip_seg['net_mask'] = netmask
            ip_seg['gateway'] = gateway
            network_list.append(ip_seg)
        return network_list

    def get_start_ip_merge(self, net_start_ip, start_ip):
        start_ip_merge = ''
        net_start_ip_list = net_start_ip.split('.')
        start_ip_list = start_ip.split('.')
        if net_start_ip_list[0] == start_ip_list[0] and \
                net_start_ip_list[1] == start_ip_list[1] and \
                net_start_ip_list[2] == start_ip_list[2]:
            start_ip_list[3] = str(min(int(net_start_ip_list[3]),
                                       int(start_ip_list[3])))
            start_ip_merge = '.'.join(start_ip_list)
        return start_ip_merge

    def get_end_ip_merge(self, net_end_ip, end_ip):
        end_ip_merge = ''
        net_end_ip_list = net_end_ip.split('.')
        end_ip_list = end_ip.split('.')
        if net_end_ip_list[0] == end_ip_list[0] and \
                net_end_ip_list[1] == end_ip_list[1] and \
                net_end_ip_list[2] == end_ip_list[2]:
            end_ip_list[3] = str(max(int(net_end_ip_list[3]),
                                     int(end_ip_list[3])))
            end_ip_merge = '.'.join(end_ip_list)
        return end_ip_merge

    def get_network_info_list(self, net_info_list, new_network_info_list):
        storage_network_list = list()
        network_list = list()
        scenario = 'initialization'
        if net_info_list:
            for sub_network in net_info_list:
                net_ips = dict()
                sub_ip_segment = sub_network.get('ip_segment')
                sub_start_ip = sub_ip_segment.get('begin_ip')
                sub_end_ip = sub_ip_segment.get('end_ip')
                sub_net_mask = sub_network.get('subnet_prefix')
                sub_gateway = sub_network.get('default_gateway')
                if (not sub_start_ip) or (not sub_end_ip):
                    continue
                net_ips['begin_ip'] = sub_start_ip
                net_ips['end_ip'] = sub_end_ip
                net_ips['net_mask'] = sub_net_mask
                net_ips['gateway'] = sub_gateway
                scenario = 'extend'
                network_list.append(net_ips)

        if new_network_info_list:
            for new_network_info in new_network_info_list:
                storage_netmask = new_network_info.get("netmask")
                storage_gateway = new_network_info.get("gateway")
                storage_subnet = new_network_info.get("subnet")
                if not storage_netmask or not storage_gateway or not storage_subnet:
                    err_msg = "Failed to get storage network[netmask:%s, gateway:%s, subnet:%s]" \
                              % (storage_netmask, storage_gateway, storage_subnet)
                    logger.error(err_msg)
                    raise Exception(err_msg)
                storage_subnet_list = storage_subnet.split('-')
                storage_net_start = storage_subnet_list[0]
                storage_net = IP(storage_net_start).make_net(storage_netmask)
                ip_net = IP(storage_net)
                net_segment = ip_net.strNormal(3)
                net_segment_list = net_segment.split('-')
                storage_start_ip = net_segment_list[0]
                storage_end_ip = net_segment_list[1]
                network_list = self.add_to_network_list(storage_start_ip, storage_end_ip, storage_netmask,
                                                        storage_gateway, network_list)

        for each_net_info in network_list:
            stor_start_ip = each_net_info.get('begin_ip')
            stor_end_ip = each_net_info.get('end_ip')
            stor_net_mask = each_net_info.get('net_mask')
            stor_gateway = each_net_info.get('gateway')
            storage_net = {'port_name': '',
                           'ip_segment': {'begin_ip': stor_start_ip, 'end_ip': stor_end_ip},
                           'subnet_prefix': stor_net_mask,
                           'default_gateway': stor_gateway}
            storage_network_list.append(storage_net)

        return storage_network_list, scenario

    def procedure(self):
        try:
            logger.info("[FS8]Start to config storage network.")
            self.login_deploy_manager()

            logger.info("Query frontend storage network.")
            net_type = 'storage_frontend'
            storage_frontend_network_list, frontend_scenario = self.query_storage_network(net_type)
            logger.info("Config frontend storage network.")
            self.config_storage_network(storage_frontend_network_list, net_type, frontend_scenario)

            logger.info("Query backend storage network.")
            net_type = 'storage_backend'
            storage_backend_network_list, backend_scenario = self.query_storage_network(net_type)
            logger.info("Config backend storage network.")
            self.config_storage_network(storage_backend_network_list, net_type, backend_scenario)

            self.validity_frontend_net()
            self.validity_backend_net()

        except FCDException as e:
            logger.error(traceback.format_exc())
            raise e
        except Exception as e:
            logger.error(traceback.format_exc())
            raise FCDException(626064, str(e))
        finally:
            self.opr.logout()
        return Message()

    def login_deploy_manager(self):
        status_code, error_code, error_des = self.opr.login(
            DeployConstant.DM_LOGIN_USER, self.update_pwd)
        if status_code != 200 or error_code != 0:
            err_msg = "Failed to login deploy manager, Detail:" \
                      "[status:%s,code:%s]%s" % (status_code, error_code, error_des)
            logger.error(err_msg)
            raise Exception(err_msg)

    def query_storage_network(self, net_type='storage_frontend'):
        rsp_result, rsp_data = self.opr.get_storage_net_plane(net_type)
        error_code = rsp_result.get('code')
        if error_code != 0:
            err_msg = "Failed to query frontend storage network"
            logger.error(err_msg)
            raise Exception(err_msg)
        net_info_list = rsp_data.get("ip_list")
        if net_type == 'storage_frontend':
            storage_network_list, scenario = self.get_network_info_list(net_info_list, self.frontend_network)
            logger.info("Current frontend storage network scenario: %s." % scenario)
        else:
            storage_network_list, scenario = self.get_network_info_list(net_info_list, self.backend_network)
            logger.info("Current backend storage network scenario: %s." % scenario)
        return storage_network_list, scenario

    def config_storage_network(self, storage_network_list, net_type, scenario):
        res_result, res_data = self.opr.set_net_plane(storage_network_list, net_type, scenario)
        if res_result.get("code") != 0 or res_data is not None:
            err_msg = "Failed to config storage network [network:%s, nettype:%s], Detail:[%s]%s" \
                      % (storage_network_list, net_type, res_result, res_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info("Config storage network [nettype:%s] successful" % net_type)

    def validity_frontend_net(self):
        net_type = 'storage_frontend'
        if self.fsa_list:
            fsa_list = [fsa['om_ip'] for fsa in self.fsa_list]
            rsp_result, rsp_data = self.opr.validity_net(net_type, fsa_list)
            fail_ip_list = list(rsp_data.keys())
            if rsp_result.get("code") != 0 or len(fail_ip_list) > 0:
                err_msg = "Failed to config frontend storage network on " \
                          "host%s, Detail:[%s] %s" % (fail_ip_list, rsp_result, rsp_data)
                logger.error(err_msg)
                raise Exception(err_msg)
        else:
            logger.info("There is no node need to validity storage_frontend network!")

    def validity_backend_net(self):
        net_type = 'storage_backend'
        if self.osd_list:
            osd_list = [osd['om_ip'] for osd in self.osd_list]
            res_result, res_data = self.opr.validity_net(net_type, osd_list)
            fail_ip_list = list(res_data.keys())
            if res_result.get("code") != 0 or len(fail_ip_list) > 0:
                err_msg = "Failed to config backend storage network on host%s, Detail:%s" % (fail_ip_list, res_data)
                logger.error(err_msg)
                raise Exception(err_msg)
            logger.info("Config backend storage network finished.")
        else:
            logger.info("There is no node need to validity storage_backend network!")
