# -*- coding: utf-8 -*-
import traceback
from IPy import IP
from utils.common.fic_base import TestCase
import utils.common.log as logger
from utils.common.message import Message
from utils.common.exception import FCDException
from plugins.DistributedStorage.scripts.logic.DeployOperate import DeployOperate
from plugins.DistributedStorage.scripts.utils.common.DeployConstant import DeployConstant
from platforms.project.ProjectUtils import get_project_condition_boolean
from platforms.project.ProjectUtils import get_project_conditions


class ConfigStorageNetwork(TestCase):
    def __init__(self, project_id, pod_id, fs_args, condition=None, metadata=None, **kwargs):
        super(ConfigStorageNetwork, self).__init__(project_id, pod_id)
        self.fs_args = fs_args
        self.opr = DeployOperate(self.fs_args)
        self.condition = condition
        self.metadata = metadata
        self.more_args = kwargs
        self.network_info_list = list()
        self.storage_network_list = list()
        self.project_id = project_id
        self.pod_id = pod_id
        self.condition_dic = get_project_conditions(self.project_id)

    def add_to_network_list(self, start_ip, end_ip, netmask, gateway):
        ip_seg = dict()
        if not self.network_info_list:
            ip_seg['begin_ip'] = start_ip
            ip_seg['end_ip'] = end_ip
            ip_seg['net_mask'] = netmask
            ip_seg['gateway'] = gateway
            self.network_info_list.append(ip_seg)
            return

        is_exist = False
        for network_ip_info in self.network_info_list:
            net_start_ip = network_ip_info.get('begin_ip')
            net_end_ip = network_ip_info.get('end_ip')
            if start_ip == net_start_ip and net_end_ip == end_ip:
                is_exist = True
                break

            start_ip_merge = self.get_start_ip_merge(net_start_ip, start_ip)

            end_ip_merge = self.get_end_ip_merge(net_end_ip, end_ip)

            if not start_ip_merge or not end_ip_merge:
                continue
            network_ip_info['begin_ip'] = start_ip_merge
            network_ip_info['end_ip'] = end_ip_merge
            if not network_ip_info.get('net_mask'):
                network_ip_info['net_mask'] = netmask
            if not network_ip_info.get('gateway'):
                network_ip_info['gateway'] = gateway
            is_exist = True
            break

        if not is_exist:
            ip_seg['begin_ip'] = start_ip
            ip_seg['end_ip'] = end_ip
            ip_seg['net_mask'] = netmask
            ip_seg['gateway'] = gateway
            self.network_info_list.append(ip_seg)
        return

    def get_start_ip_merge(self, net_start_ip, start_ip):
        start_ip_merge = ''
        net_start_ip_list = net_start_ip.split('.')
        start_ip_list = start_ip.split('.')
        if net_start_ip_list[0] == start_ip_list[0] and net_start_ip_list[1] == start_ip_list[1] \
                and net_start_ip_list[2] == start_ip_list[2]:
            start_ip_list[3] = str(min(int(net_start_ip_list[3]), int(start_ip_list[3])))
            start_ip_merge = '.'.join(start_ip_list)
        return start_ip_merge

    def get_end_ip_merge(self, net_end_ip, end_ip):
        end_ip_merge = ''
        net_end_ip_list = net_end_ip.split('.')
        end_ip_list = end_ip.split('.')
        if net_end_ip_list[0] == end_ip_list[0] and net_end_ip_list[1] == end_ip_list[1] \
                and net_end_ip_list[2] == end_ip_list[2]:
            end_ip_list[3] = str(max(int(net_end_ip_list[3]), int(end_ip_list[3])))
            end_ip_merge = '.'.join(end_ip_list)
        return end_ip_merge

    def procedure(self):
        try:
            logger.info("Start to config storage network.")
            self.opr.login(DeployConstant.DM_LOGIN_USER, self.fs_args['dm_update_pwd'])

            logger.info("Query frontend storage network.")
            net_type = 'storage_frontend'
            net_info_list = self.query_storage_network(net_type)
            if net_info_list:
                self.get_network_info_list(net_info_list)

            if get_project_condition_boolean(self.project_id, '(TenantStorFB80|TenantStorFBReuse80)&'
                                                              '(ExpansionAZ_KVM|ExpansionAZ_BMS)|'
                                                              '(TenantStorFB80&ExpansionServiceStorage)'):
                self.get_compute_net_plane()

            if get_project_condition_boolean(self.project_id,
                                             '(TenantStorFB80|TenantStorFBReuse80|DRStorage_TFB_Sep)&'
                                             '(!TenantStorFB_Heterogeneous|TenantStor_ThreeTierNetwork)'):
                self.get_storage_net_plane("business_separate_storage_netmask",
                                           "business_separate_storage_gateway", "business_separate_storage_subnet")

            self.get_storage_network_list()

            if get_project_condition_boolean(self.project_id, '((ExpansionPOD_KVM|ExpansionPOD_BMS)&'
                                                              'ExpansionPOD_CloudService)|ExpansionAZ_KVM|'
                                                              'ExpansionServiceStorage|ExpansionAZ_BMS|'
                                                              'ExpansionMgmtRes_ServiceNode|'
                                                              'ExpansionScale_CloudServiceNode|'
                                                              'DRStorage_TFB_Sep&(CSHAStorage_TFB|CSDRStorage_TFB|'
                                                              'TenantStorFBHCI80)'):
                scenario = 'extend'
            else:
                scenario = 'initialization'
            logger.info("Current scenario: %s." % scenario)

            logger.info("Config frontend storage network.")
            net_type = 'storage_frontend'
            self.config_storage_network(net_type, scenario)

            logger.info("Config backend storage network.")
            net_type = 'storage_backend'
            self.config_storage_network(net_type, scenario)

            node_list = [vbs['om_ip'] for vbs in self.fs_args['vbs_list']]
            self.validity_config_net(net_type, node_list)

            logger.info("Config storage network finished.")
            self.opr.login_out(DeployConstant.DM_LOGIN_USER, self.fs_args['dm_update_pwd'])
        except FCDException as e:
            logger.error(traceback.format_exc())
            return Message(500, e)
        except Exception as e:
            return Message(500, FCDException(626064, str(e)))
        return Message()

    def query_storage_network(self, net_type):
        rsp_obj = self.opr.get_storage_frontend_net_plane(net_type)
        rsp_code, rsp_result, rsp_data = rsp_obj.get_rsp_data()
        error_code = rsp_result.get('code')
        if error_code != 0:
            err_msg = "Failed to query storage_frontend network"
            logger.error(err_msg)
            raise Exception(err_msg)
        net_info_list = rsp_data.get("ip_list")
        return net_info_list

    def config_storage_network(self, net_type, scenario):
        res_result, res_data = self.opr.set_net_plane(self.storage_network_list, net_type, scenario)
        if res_result.get("code") != 0 or res_data is not None:
            err_msg = "Failed to config storage network[network:%s, nettype:%s], Detail:[%s]%s" \
                      % (self.storage_network_list, net_type, res_result, res_data)
            logger.error(err_msg)
            raise Exception(err_msg)

    def get_network_info_list(self, net_info_list):
        for sub_network in net_info_list:
            net_ips = dict()
            sub_ip_segment = sub_network.get('ip_segment')
            sub_start_ip = sub_ip_segment.get('begin_ip')
            sub_end_ip = sub_ip_segment.get('end_ip')
            sub_net_mask = sub_network.get('subnet_prefix')
            sub_gateway = sub_network.get('default_gateway')
            if (not sub_start_ip) or (not sub_end_ip):
                continue
            net_ips['begin_ip'] = sub_start_ip
            net_ips['end_ip'] = sub_end_ip
            net_ips['net_mask'] = sub_net_mask
            net_ips['gateway'] = sub_gateway
            self.network_info_list.append(net_ips)

    def get_storage_network_list(self):
        for each_net_info in self.network_info_list:
            stor_start_ip = each_net_info.get('begin_ip')
            stor_end_ip = each_net_info.get('end_ip')
            stor_net_mask = each_net_info.get('net_mask')
            stor_gateway = each_net_info.get('gateway')
            storage_net = {'port_name': '',
                           'ip_segment': {'begin_ip': stor_start_ip, 'end_ip': stor_end_ip},
                           'subnet_prefix': stor_net_mask,
                           'default_gateway': stor_gateway}
            self.storage_network_list.append(storage_net)

    def get_compute_net_plane(self):
        exp_compute_net_planes = self.db.get_computing_node_network_planning(self.project_id)
        compute_net_plane = None
        for compute_net in exp_compute_net_planes:
            if compute_net.get('network_plane') == DeployConstant.MGR_STORAGE_NET_INTF:
                compute_net_plane = compute_net
                break
        if compute_net_plane:
            ip_net = IP(compute_net_plane.get('network_segment_plan'))
            net_segment = ip_net.strNormal(3)
            net_segment_list = net_segment.split('-')
            manage_storage_start_ip = net_segment_list[0]
            manage_storage_end_ip = net_segment_list[1]
            manage_storage_netmask = IP(compute_net_plane.get('network_segment_plan')).strNetmask()
            manage_storage_gateway = compute_net_plane.get('gateway')
            self.add_to_network_list(manage_storage_start_ip, manage_storage_end_ip, manage_storage_netmask,
                                     manage_storage_gateway)
        else:
            err_msg = "Failed to get compute storage network from 1.2Sheet"
            logger.error(err_msg)
            raise Exception(err_msg)

    def validity_config_net(self, net_type, node_list):
        rsp = self.opr.config_net_validity(net_type, node_list)
        rsp_code, rsp_data = rsp.get_net_validity_data()
        fail_ip_list = list(rsp_data.keys())
        if rsp_code != 0 or len(fail_ip_list) > 0:
            err_msg = "Failed to config storage network on host%s, Detail:%s" % (fail_ip_list, rsp_data)
            logger.error(err_msg)
            raise Exception(err_msg)

    def get_storage_net_plane(self, netmask, gateway, subnet):
        storage_netmask = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorageReplication", netmask)
        storage_gateway = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorageReplication", gateway)
        storage_subnet = self.db.get_value_from_cloudparam(self.pod_id, "DistributedStorageReplication", subnet)
        if not storage_netmask or not storage_gateway or not storage_subnet:
            err_msg = "Failed to get storage network[netmask:%s, gateway:%s, subnet:%s]" \
                      % (storage_netmask, storage_gateway, storage_subnet)
            logger.error(err_msg)
            raise Exception(err_msg)
        ip_net = IP(storage_subnet)
        net_segment = ip_net.strNormal(3)
        net_segment_list = net_segment.split('-')
        storage_start_ip = net_segment_list[0]
        storage_end_ip = net_segment_list[1]
        self.add_to_network_list(storage_start_ip, storage_end_ip, storage_netmask, storage_gateway)
        return
