/*
 * Copyright (c) Huawei Technologies Co., Ltd. 2020-2022. All rights reserved.
 */
function Promise(executor) {
    this.state = 'pending';
    this.value = undefined;
    this.reason = undefined;
    this.onResolvedCallbacks = [];
    this.onRejectedCallbacks = [];

    function resolve(value) {
        if (this.state === 'pending') {
            this.state = 'fulfilled';
            this.value = value;
            this.onResolvedCallbacks.forEach(function (fn) {
                fn();
            });
        }
    }

    function reject(reason) {
        if (this.state === 'pending') {
            this.state = 'rejected';
            this.reason = reason;
            this.onRejectedCallbacks.forEach(function (fn) {
                fn();
            });
        }
    }

    try {
        executor(resolve.bind(this), reject.bind(this));
    } catch (err) {
        reject(err);
    }

}


Promise.prototype = {
    then: function (onFulfilled, onRejected) {
        onFulfilled = typeof onFulfilled === 'function' ? onFulfilled : function (value) {
            return value
        };
        onRejected = typeof onRejected === 'function' ? onRejected : function (value) {
            throw value
        };
        var promise2 = new Promise(function (resolve, reject) {
            if (this.state === 'fulfilled') {
                setTimeout(function () {
                    try {
                        var x = onFulfilled(this.value);
                        resolvePromise(promise2, x, resolve, reject);
                    } catch (e) {
                        reject(e);
                    }
                }.bind(this), 0);
            }
            if (this.state === 'rejected') {
                setTimeout(function () {
                    try {
                        var x = onRejected(this.reason);
                        resolvePromise(promise2, x, resolve, reject);
                    } catch (e) {
                        reject(e);
                    }
                }.bind(this), 0);
            }
            if (this.state === 'pending') {
                this.onResolvedCallbacks.push(function () {
                    setTimeout(function () {
                        try {
                            var x = onFulfilled(this.value);
                            resolvePromise(promise2, x, resolve, reject);
                        } catch (e) {
                            reject(e);
                        }
                    }.bind(this), 0);
                }.bind(this));
                this.onRejectedCallbacks.push(function () {
                    setTimeout(function () {
                        try {
                            var x = onRejected(this.reason);
                            resolvePromise(promise2, x, resolve, reject);
                        } catch (e) {
                            reject(e);
                        }
                    }.bind(this), 0)
                }.bind(this));
            }
        }.bind(this));
        return promise2;
    }
}

function resolvePromise(promise2, x, resolve, reject) {
    if (x === promise2) {
        return reject(new TypeError('Chaining cycle detected for promise'));
    }
    var called;
    if (!(x != null && (typeof x === 'object' || typeof x === 'function'))) {
        resolve(x);
    } else {
        try {
            var then = x.then;
            if (typeof then !== 'function') {
                resolve(x);
            } else {
                then.call(x, function (y) {
                    if (!called) {
                        called = true;
                        resolvePromise(promise2, y, resolve, reject);
                    }
                }, function (err) {
                    if (!called) {
                        called = true;
                        reject(err);
                    }
                })
            }
        } catch (e) {
            if (!called) {
                called = true;
                reject(e);
            }
        }
    }
}

//resolve方法
Promise.resolve = function (val) {
    return new Promise(function (resolve, reject) {
        resolve(val)
    });
}
//reject方法
Promise.reject = function (val) {
    return new Promise(function (resolve, reject) {
        reject(val)
    });
}
//race方法
Promise.race = function (promises) {
    return new Promise(function (resolve, reject) {
        for (var i = 0; i < promises.length; i++) {
            promises[i].then(resolve, reject)
        }
    })
}
//all方法(获取所有的promise，都执行then，把结果放到数组，一起返回)
Promise.all = function (promises) {
    var arr = [];
    var i = 0;

    function processData(index, data, resolve) {
        arr[index] = data;
        i++;
        if (i === promises.length) {
            resolve(arr);
        }
    }

    return new Promise(function (resolve, reject) {
        for (var i = 0; i < promises.length; i++) {
            promises[i].then(function (data) {
                processData(i, data, resolve);
            }, reject);
        }
    });
}


