# -*- coding:utf-8 -*-
import traceback
import utils.common.log as logger
from utils.common.message import Message
from utils.common.exception import HCCIException
from utils.DBAdapter.DBConnector import BaseOps
from utils.common.fic_base import StepBaseInterface
from utils.business.project_condition_utils import get_project_condition_boolean
from plugins.DistributedStorage.utils.iterm.ParameterGain import ParamsGain
from plugins.DistributedStorage.implement.TC_StorageNode_Install_OS import InstallOS


class InstallOSIntf(StepBaseInterface):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.db = BaseOps()
        self.params = ParamsGain(project_id, pod_id, self.db)
        if get_project_condition_boolean(self.project_id, 'TenantStorNewPool|TenantStorNewNode'):
            fs_args = self.params.get_business_separate_expand_fusionstorage_args_while_new_pool_or_new_node()
        else:
            fs_args = self.params.get_business_separate_args()
        self.install_os = InstallOS(self.project_id, self.pod_id, fs_args)

    def pre_check(self, project_id, pod_id):
        """
        插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，工具框架不会直接调用此接口。
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message(200)

    def execute(self, project_id, pod_id):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.install_os.procedure()
        except HCCIException as e:
            logger.error(traceback.format_exc())
            return Message(500, e)
        except Exception as e:
            logger.error(traceback.format_exc())
            logger.error("install os failed:%s" % str(e))
            return Message(500, HCCIException(626025, str(e)))
        return Message()

    def rollback(self, project_id, pod_id):
        """
        标准调用接口：执行回滚
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        try:
            self.install_os.reset_status()
        except HCCIException as e:
            logger.error(traceback.format_exc())
            return Message(500, str(e))
        except Exception as e:
            logger.error(traceback.format_exc())
            logger.error("Rollback failed:%s" % str(e))
            return Message(500, HCCIException(626081, str(e)))
        return Message()

    def retry(self, project_id, pod_id):
        """
        标准调用接口：重试
        :return: Message类对象
        """
        try:
            self.install_os.procedure()
        except HCCIException as e:
            logger.error(traceback.format_exc())
            return Message(500, str(e))
        except Exception as e:
            logger.error(traceback.format_exc())
            return Message(500, HCCIException(626081, str(e)))
        return Message()

    def check(self, project_id, pod_id):
        """
        标准调用接口：重试
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        return Message()

    def _get_nic_mode(self, lld_key):
        nic_mode = self.db.get_user_input_cloud_param_by_key(self.project_id, lld_key)
        logger.info("Get nic mode:%s" % nic_mode)
        if nic_mode == 'active_standby':
            bond_mode = 1
        elif nic_mode == 'lacp':
            bond_mode = 4
        else:
            bond_mode = 1
        logger.info("bond mode:%s" % bond_mode)
        return bond_mode
