# -*- coding: utf-8 -*-
import traceback

import utils.common.log as logger
from plugins.DistributedStorage.common.UpgradeOperate import UpgradeOperate
from plugins.DistributedStorage.common.constants import DeployType, NodeRole
from plugins.DistributedStorage.common.storage_cmdb_util import StorageCMDBUtil
from plugins.DistributedStorage.common.base import TestCase
from utils.common.message import Message
from utils.common.exception import FCUException


class UpdateCMDB(TestCase):
    def __init__(self, project_id, pod_id, regionid_list, fs_args,
                 condition=None, metadata=None, **kwargs):
        super(UpdateCMDB, self).__init__(project_id, pod_id)
        self.condition = condition
        self.metadata = metadata
        self.more_args = kwargs
        self.opr = UpgradeOperate(fs_args)
        self.user_name = fs_args["user_name"]
        self.password = fs_args["password"]
        self.service_name = 'FusionStorageBlock'
        self.region_id = regionid_list[0]
        self.float_ip = fs_args.get('float_ip')
        self.master_ip = fs_args.get('master_ip')
        self.slaver_ip = fs_args.get('slaver_ip')
        self.fs_cmdb_util = StorageCMDBUtil(self.project_id, self.pod_id, self.region_id)
        self.plane_dict = {
            'management_internal': "External_om",
            'storage_frontend': 'storage_data0',
            'storage_backend': 'storage_inner',
            'replication': "replication",
            'quorum': "quorum"}

    def get_cloud_service_info(self):
        """
        生成V3接口注册cmdb的信息
        """
        name = self.get_name()
        version = self._get_product_version()
        deploy_node_infos, fsm_ip_list = self.get_node_info()
        osd_hostname_list = [deploy_node.get("name") for deploy_node in deploy_node_infos]
        vm_hostname_list = self.get_fsm_vm_hostname_list(name)
        cloud_service_info = {
            "indexName": self.service_name,
            "version": version,
            "name": name,
            "extendInfos": [
                {'key': "az_id", "value": self.get_az_id(name)},
                {"key": "fsm_float_ip", "value": self.float_ip},
                {"key": "fsm_primary_ip", "value": fsm_ip_list[0]},
                {"key": "fsm_slave_ip", "value": fsm_ip_list[1]},
            ],
            "deployNodeInfos": deploy_node_infos,
        }

        deploy_type = self.get_deploy_type()
        if deploy_type == DeployType.SEPARATE:
            cloud_service_info["extendInfos"].append(
                {"key": "deploy_bmc_node", "value": ";".join(self.get_all_bmc_ip(name))}
            )
            services = self.get_services_info(version, self.region_id, osd_hostname_list, vm_hostname_list)
        else:
            services = self.get_services_info(version, self.region_id, [], vm_hostname_list)
        cloud_service_info["extendInfos"].append({'key': "deployment_type", "value": deploy_type})
        cloud_service_info["services"] = services

        logger.info("get param cloud_service: %s" % str(cloud_service_info))
        return cloud_service_info

    def get_node_info(self):
        """
        获取和构造节点信息 按照cmdb的接口要求
        """
        fsm_list = []
        deploy_node_infos = []
        result = self.opr.get_net_service()
        for node in result:
            if NodeRole.MANAGEMENT_ROLE in node['role']:
                # 浮动ip部分
                fsm_list.append(str(node['management_internal_ip']))
                continue
            if NodeRole.STORAGE_ROLE not in node['role']:
                continue
            item = {"name": str(node.get('name'))}
            # 节点信息部分
            ip_addresses = self.get_ip_address(node)
            item.update({'ipAddresses': ip_addresses})
            logger.info("get node success. detail:%s" % str(item))
            deploy_node_infos.append(item)
        if not fsm_list:
            logger.error("FSM list length is zero.")
            raise Exception("FSM list length is zero.")
        elif len(fsm_list) > 1:
            fsm_primary_ip = fsm_list[0]
            fsm_slave_ip = fsm_list[1]
        else:
            fsm_primary_ip = fsm_list[0]
            fsm_slave_ip = fsm_list[0]
        return deploy_node_infos, [fsm_primary_ip, fsm_slave_ip]

    def get_ip_address(self, node):
        def extract_net_info(plane_dict, _ip_usage):
            net_info = {
                "netPlane": plane_dict[_ip_usage],
                "ip": str(ip_info['ip_address']),
                "gateway": str(ip_info['default_gateway']),
                "netMask": str(ip_info['subnet_prefix']),
            }
            ip_addresses.append(net_info)

        ip_addresses = []
        frontend_and_backend_converge = sorted(['storage_frontend', 'storage_backend'])
        for ip_info in node.get('ip_address_list'):
            ip_usage_list = ip_info.get('ip_usage')
            if not ip_usage_list:
                continue
            if sorted(ip_usage_list) == frontend_and_backend_converge:
                extract_net_info(self.plane_dict, _ip_usage="storage_frontend")
                continue
            for ip_usage in ip_usage_list:
                if ip_usage not in self.plane_dict:
                    continue
                extract_net_info(self.plane_dict, ip_usage)
        return ip_addresses

    def get_services_info(self, version, region_id, osd_host_names, vm_hostnames):
        """
        组装微服务参数
        """
        service_info = [
            {
                "indexName": self.service_name,
                "version": version,
                "regionCode": region_id,
                "extendInfos": [
                    {
                        "key": "Distributed-storage",
                        "value": "True"
                    }
                ],
                "components": [
                    {
                        "indexName": 'FSM',
                        "version": version,
                        "nodeNames": vm_hostnames
                    },
                    {
                        "indexName": 'FSA',
                        "version": version,
                        "nodeNames": osd_host_names
                    }
                ]
            }
        ]
        return service_info

    def get_deploy_type(self):
        ret = DeployType.SEPARATE
        for node_info in self.get_cluster_info():
            usage = node_info.get("usage", [])
            if "replicate_ControlCluster" in usage or "replicate_BusinessCluster" in usage:
                break
            node_role = node_info.get("role", [])
            if NodeRole.COMPUTE_ROLE in node_role and NodeRole.STORAGE_ROLE in node_role:
                ret = DeployType.CONVERGE
                break
        return ret

    def get_name(self):
        return self.service_name + "_" + self._get_product_esn()

    def get_all_bmc_ip(self, name):
        bmc_ip_list = self.fs_cmdb_util.get_deployed_bmc_ip_list_by_name(name)
        logger.info("bmc_ip_list:{}".format(bmc_ip_list))
        return sorted(set(bmc_ip_list))

    def get_az_id(self, name):
        az_id = self.fs_cmdb_util.get_az_id(name)
        logger.info("az_id:{}".format(az_id))
        return az_id

    def get_fsm_vm_hostname_list(self, name):
        fsm_vm_hostname_list = self.fs_cmdb_util.get_fsm_hostname_list(name)
        logger.info("fsm_vm_hostname_list:{}".format(fsm_vm_hostname_list))
        return fsm_vm_hostname_list

    def get_cluster_info(self):
        ret_result, ret_data = self.opr.get_servers()
        logger.info("result:{}, data:{}".format(ret_result, ret_data))
        if ret_result.get("code") != 0:
            err_msg = "get servers failed, Detail:[result:{}, data:{}]".format(ret_result, ret_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        return ret_data

    def procedure(self):
        logger.info('Start update CMDB.')
        try:
            status_code, error_code, error_des = self.opr.try_login(
                self.user_name, self.password)
            if status_code != 200 or error_code != 0:
                err_msg = "Failed to login, Detail:[status:%s,code:%s]%s" \
                          % (status_code, error_code, error_des)
                logger.error(err_msg)
                raise Exception(err_msg)

            set_cmdb_info = self.get_cloud_service_info()
            logger.info("set cmdb %s" % set_cmdb_info)
            self.fs_cmdb_util.set_cloud_service_info_v3(set_cmdb_info)
            logger.info("End to set cloud info into CMDB.")
        except FCUException as e:
            logger.error('update CMDB failed:{}'.format(e))
            logger.error(traceback.format_exc())
            raise e
        except Exception as e:
            logger.error('update CMDB failed:{}'.format(e))
            logger.error(traceback.format_exc())
            raise FCUException(620008, str(e))
        return Message(200)

    def _get_product_version(self):
        logger.info('get cluster product version.')
        ret_result, ret_data = self.opr.get_product()
        if ret_result["code"] != 0:
            err_msg = "get cluster product failed, " \
                      "Detail:[result:%s, data:%s]" \
                      % (ret_result, ret_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        product_version = ret_data["version"]
        return product_version

    def _get_product_esn(self):
        ret_result, ret_data = self.opr.get_esn()
        if ret_result["code"] != 0:
            err_msg = "get cluster product failed, " \
                      "Detail:[result:%s, data:%s]" \
                      % (ret_result, ret_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        product_esn = ret_data["esn"]
        return product_esn
