# -*- coding: utf-8 -*-
import traceback

import utils.common.log as logger
from utils.common.message import Message
from utils.common.fic_base import StepBaseInterface
from utils.common.exception import FCUException
from plugins.DistributedStorage.common.RestClient import StorageSSHClient
from plugins.DistributedStorage.common.ParameterGain import ParamsGain
from plugins.DistributedStorage.common.UpgradeOperate import UpgradeOperate
from plugins.DistributedStorage.common.constants import NodeRole


class PreCheckMgmtIP(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list, suit_id=None):
        super(PreCheckMgmtIP, self).__init__(project_id, pod_id, regionid_list)
        self.suit_id = suit_id
        self.fs_args = ParamsGain(project_id, pod_id, regionid_list).get_args(suit_id)
        self.opr = UpgradeOperate(self.fs_args)

    def execute(self, project_id, pod_id, regionid_list=None, suit_id=None):
        logger.info('Start pre upgrade mgmt IP check.')
        try:
            self.procedure()
        except FCUException as err:
            return Message(500, err)
        except Exception as err:
            logger.error('check mgmt IP failed. details:{}'.format(traceback.format_exc()))
            return Message(500, FCUException(621024, str(err)))
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None, suit_id=None):
        """
        标准调用接口：重试
        :return: Message对象
        """
        return self.execute(project_id, pod_id, regionid_list, suit_id)

    def procedure(self):
        """检查用户填写的FSM主备管理IP是否正确，检查主备管理IP的用户密码是否正确
        """
        self.opr.try_login(self.fs_args.get("user_name"), self.fs_args.get("password"))
        result = self.opr.get_net_service()
        logger.info(result)
        self.opr.logout()

        logger.info("step 1. Querying the Management IP Address from the Cluster.")
        fsm_list = []
        for node in result:
            if NodeRole.MANAGEMENT_ROLE in node.get('role', []):
                fsm_list.append(node.get('management_internal_ip'))
        if len(fsm_list) != 2:
            logger.error("Failed to obtain the FSM management IP address, fsm_list:{}".format(fsm_list))
            raise Exception("Failed to obtain the FSM management IP address")

        logger.info("step 2. Check whether the entered management IP address is correct.")
        master_node, slaver_node = self.fs_args.get("master_node"), self.fs_args.get("slaver_node")
        if master_node.ip not in fsm_list or slaver_node.ip not in fsm_list:
            logger.error(f"The FSM management IP is incorrect. The FSM IP in the cluster are:{fsm_list}. "
                         f"The FSM IP filled in by the user are: {[master_node.ip, slaver_node.ip]}")
            raise FCUException(621022,
                               self.fs_args.get("float_ip"),
                               fsm_list,
                               [master_node.ip, slaver_node.ip],
                               [f"FSM_{self.suit_id}_active_ip", f"FSM_{self.suit_id}_standby_ip"])

        logger.info("step 3. Check whether the user name of the management node is entered.")
        if not master_node.user_name:
            logger.error(f"the FSM_{self.suit_id}_active_user for {master_node.ip} is not filled in.")
            raise FCUException(621025, self.fs_args.get("float_ip"), master_node.ip, f"FSM_{self.suit_id}_active_user")
        if not slaver_node.user_name:
            logger.error(f"the FSM_{self.suit_id}_standby_user for {slaver_node.ip} is not filled in.")
            raise FCUException(621025, self.fs_args.get("float_ip"), slaver_node.ip, f"FSM_{self.suit_id}_standby_user")

        logger.info("step 4. Check whether the password of the management node is correct.")
        nodes_list = [(master_node, f"FSM_{self.suit_id}_active_pwd", f"FSM_{self.suit_id}_active_root_pwd"),
                      (slaver_node, f"FSM_{self.suit_id}_standby_pwd", f"FSM_{self.suit_id}_standby_root_pwd")]
        for node, pwd_key, root_pwd_key in nodes_list:
            try:
                ssh_client = StorageSSHClient(node.ip, node.user_name, node.passwd)
            except Exception as err:
                logger.error(err)
                raise FCUException(621023, node.user_name, self.fs_args.get("float_ip"), node.ip, pwd_key) from err
            try:
                ssh_client.switch_root(node.root_pwd)
            except Exception as err:
                logger.error(err)
                raise FCUException(621023, "root", self.fs_args.get("float_ip"), node.ip, root_pwd_key) from err
            del ssh_client

        logger.info("The check is passed.")
