# -*- coding: utf-8 -*-

import os
import re
import stat
import errno
import shutil
import tarfile
import time
import threading

import utils.common.log as logger
from utils.common.exception import FCUException
from plugins.DistributedStorage.common.UpgradeOperate import UpgradeOperate

FILE_LOCK = threading.Lock()


class SystemHandle:
    def __init__(self):
        """
        Distributed block storage打包、系统操作公共类，非对外接口
        """
        pass

    def unzip_file(self, file_path, local_dst_dir):
        limit_size = 8 * 1024 * 1024 * 1024
        file_size = os.path.getsize(file_path)
        if file_size > limit_size:
            err_msg = "file size %s is larger than 8G" % file_size
            logger.error(err_msg)
            raise FCUException(621003, err_msg)
        if os.path.exists(local_dst_dir):
            shutil.rmtree(local_dst_dir, onerror=self.error_remove_read_only)
        os.makedirs(local_dst_dir)
        logger.info("start unzip file")
        with tarfile.open(file_path) as tar_file:
            tar_file.extractall(members=self._deploymanage_pkg(tar_file), path=local_dst_dir)

    @staticmethod
    def error_remove_read_only(func, path, exc):
        excvalue = exc[1]
        if func in (os.rmdir, os.remove) and excvalue.errno == errno.EACCES:
            os.chmod(path, stat.S_IRWXU | stat.S_IRWXG | stat.S_IRWXO)
            # retry
            func(path)
        elif func in (os.rmdir, os.remove) and excvalue.errno == errno.ENOTEMPTY:
            func(path)
        else:
            raise FCUException(621003, "remove func is wrong")

    @staticmethod
    def _deploymanage_pkg(members):
        for tarinfo in members:
            logger.info(tarinfo.name)
            if os.path.split(tarinfo.name)[1].startswith("OceanStor-Pacific_deploymanager_"):
                yield tarinfo

    @staticmethod
    def check_pkg_version(fs_args):
        package_name = fs_args.get("package_name")
        if package_name.startswith("OceanStor-Pacific_8"):
            return 81
        else:
            err_msg = "pkg {} version is wrong".format(package_name)
            logger.error(err_msg)
            raise FCUException(621003, err_msg)

    @staticmethod
    def un_tar_pkg(ssh_client, remote_pkg_path, dst_dir):
        un_tar_cmd = "tar xfz {} -C {}; echo last_result=$?".format(remote_pkg_path, dst_dir)
        cmd_ret = ssh_client.send_cmd(un_tar_cmd, '#')
        cmd_ret = ''.join(cmd_ret)
        if "last_result=0" not in cmd_ret:
            logger.error("un tar %s failed, ret: %s" % (remote_pkg_path, cmd_ret))
            err_msg = "un tar %s failed" % remote_pkg_path
            raise FCUException(621007, err_msg)


class RestPublicMethod(object):
    def __init__(self, project_id=None, pod_id=None, fs_args=None, **kwargs):
        self.project_id = project_id
        self.pod_id = pod_id
        self.fs_args = fs_args
        self.opr = UpgradeOperate(fs_args)

    def check_storage_version(self):
        ret_result, ret_data = self.opr.get_version()
        if ret_result["code"] != 0:
            err_msg = "get version failed, Detail:[result:%s, data:%s]" % (ret_result, ret_data)
            logger.error(err_msg)
            raise FCUException(621003, err_msg)
        version = ret_data.get("version")
        logger.info("current version is %s" % version)
        if version >= "8.1":
            return 81
        else:
            return 80

    def get_server_list(self):
        osd_nodes_list = []
        vbs_nodes_list = []
        ret_result, ret_data = self.opr.get_servers()
        if ret_result["code"] != 0:
            err_msg = "get servers failed, Detail:[result:%s, data:%s]" % (ret_result, ret_data)
            logger.error(err_msg)
            raise Exception(err_msg)
        for item in ret_data:
            if "storage" in item["role"]:
                osd_nodes_list.append(item["management_ip"])
            if "compute" in item["role"]:
                vbs_nodes_list.append(item["management_ip"])
        return osd_nodes_list, vbs_nodes_list

    @staticmethod
    def get_failed_component(data_result):
        failed_component_map = dict()

        def _get_failed_component():
            component_info = sequence.get("componentInfoList")
            for component in component_info:
                if component.get("componentStatus") == "failed":
                    failed_component_list.append(component.get("name"))

        for node in data_result:
            if node["status"] != "failed":
                continue
            node_ip = node["hostIp"]
            sequence_infos = node["sequenceInfos"]
            failed_component_list = []
            for sequence in sequence_infos:
                _get_failed_component()
            failed_component_map[node_ip] = failed_component_list
        return failed_component_map


class SshPublicMethod:
    def __init__(self, project_id=None, pod_id=None, fs_args=None, *args, **kwargs):
        """
        ssh public interface
        """
        pass

    @staticmethod
    def check_status_opr(ssh_client):
        get_status_cmd = 'curl -sgkX GET -H "Content-type: application/json" --noproxy -m 5 --connect-timeout 5 ' \
                         'https://127.0.0.1:6098/api/v2/curl/deploy/tomcat_status'
        check_timeout = 600
        while check_timeout > 0:
            cmd_ret = ssh_client.send_cmd(get_status_cmd, "#", 10)
            cmd_ret = ''.join(cmd_ret)
            if "success" in cmd_ret:
                break
            check_timeout -= 10
            time.sleep(10)
        if check_timeout <= 0:
            raise FCUException(621003, "deploymanager status failed")

    @staticmethod
    def ssh_cmd_get_arch(ssh_client):
        arch_cmd = "uname -r"
        cmd_ret = ssh_client.send_cmd(arch_cmd, '#')
        return cmd_ret

    def get_deploymanager_pkg_name(self, ssh_client, local_pkg_dir):
        cmd_ret = self.ssh_cmd_get_arch(ssh_client)
        cmd_ret = ''.join(cmd_ret)
        if "x86_64" in cmd_ret:
            platform = "x86_64"
        elif "aarch" in cmd_ret:
            platform = "aarch64"
        else:
            raise FCUException(621003, "node arch %s is wrong." % cmd_ret)
        pkg_name_prefix = "OceanStor-Pacific_deploymanager_"
        pkg_name_like = pkg_name_prefix + platform
        file_list = os.listdir(local_pkg_dir)
        deploymanager_pkg_name = None
        for file_name in file_list:
            if len(re.findall(pkg_name_like, file_name)) > 0:
                deploymanager_pkg_name = file_name
                break
        if not deploymanager_pkg_name:
            raise FCUException(621003, "the deploymanager pkg {} is not exist".format(pkg_name_like))
        return deploymanager_pkg_name
