#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 定义硬盘类

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2023/07/24 18:00 created

"""
import time

from config.constants import CONFIGURATION
from lib.core.tools.mcscli import McsCli
from lib.exception.dfs_exception import DFSException
from lib.utils.log import SingleLog


class DriveFactoryMap(object):
    def __init__(self):
        self.drive_map = {
            "ata": AtaDrive,
            "scsi": ScsiDrive,
            "nvme": NvmeDrive
        }

    # 传递drive_info
    def get(self, drive_info):
        mn = drive_info.get("model_number")
        if mn in ("HWE62ST3960L005N", "HWE62ST31T9L005N"):
            drive_chunk_size = (65536, 4096)
        else:
            drive_chunk_size = (4096, 8192, 16384, 32768, 65536)
        drive_info["chunk_size"] = drive_chunk_size
        return self.drive_map.get(drive_info.get("protocol"))(drive_info)


class Drive(object):
    """
    Drive Entity Class
    """

    def __init__(self, info):
        self.info = info
        self.mcs = McsCli()
        self.log = SingleLog()
        # raid 卡信息
        self.raid = None
        self.upgrade_step = []
        self.upgrade_fw_list = []
        self.upgrade_firmware_list = []

    def add_upgrade_step(self, steps):
        self.upgrade_step = steps

    def add(self, raid_obj):
        """
            给dirve绑定raid卡对象
        Returns:

        """
        self.raid = raid_obj

    def smart(self):
        pass

    def internal(self):
        """
        Returns:
            log_path:内部日志存放的路径，路径由mcs确定,若失败则返回空字符串
        """
        target = self.info.get("target")
        drive_chunk_size = self.chunk_size()
        if target != "--":
            card_chunk_size = (4096, 8192, 16384, 32768, 65536)
            if target.split(":")[0] == "pmc":
                card_chunk_size = (4096, 8192, 16384, 32768)
            available_chunk_size = set(drive_chunk_size) & set(card_chunk_size)
        else:
            available_chunk_size = set(drive_chunk_size)
        if not available_chunk_size:
            self.log.error("Collect internal log: card chunkSize not match drive chunkSize.")
            return ""
        chunk_size = max(available_chunk_size)
        drive_letter = self.info.get("drive_letter", "")
        sn = self.info.get("serial_number", "")
        target = self.info.get("target", "")
        ret_status, log_path = self.mcs.disk_log(drive_letter=drive_letter, target=target, sn=sn, chunk_size=chunk_size)
        return ret_status, log_path

    def chunk_size(self):
        return self.info.get("chunk_size")

    def upgrade(self, a, drive_info, current_upgrade_order):
        pass

    def is_normal_before_upgrade(self, drive_letter):
        try:
            target = self.info["target"]
            flag, msg = self.mcs.get_smart_health_status(drive_letter, target)
            if not flag:
                self.log.warning(msg)
                return False
        except DFSException as e:
            self.log.warning(e.message)
            return False
        return True

    def is_normal_after_upgrade(self, fw):
        # 刷新状态跟初始状态对比
        is_change = False
        if self.raid is not None:
            is_change = self.raid.is_changed(self.info["serial_number"])
        if is_change:
            self.log.warning("Sn：{} raid status is changed after upgrade".format(self.info["serial_number"]))
            return False
        # 获取特殊盘配置文件
        power_config = CONFIGURATION.get("poweroff")
        if (self.model_number(), self.firmware_version()) in power_config:
            # 上下电
            return True

        # 循环三次 每次休眠5s 一共检查第0、5、10、15s 若最后一次检测异常则 超时
        params = {
            "drive_letter": self.drive_letter(),
            "through_opt": self.info.get("target", "")
        }
        for _ in range(3):
            single_drive_info = self.mcs.get_single_drive_info(params)
            if single_drive_info[0].get("firmware_version", "") == fw:
                return True
            self.log.info("Current_firmware_version: {}".format(single_drive_info[0].get("firmware_version", "")))
            self.log.info("Sleep 5s")
            time.sleep(5)
        single_drive_info = self.mcs.get_single_drive_info(params)
        if single_drive_info[0].get("firmware_version", "") == fw:
            return True
        self.log.info("Current_firmware_version: {}".format(single_drive_info[0].get("firmware_version", "")))
        # 超过15s 超时报错
        self.log.warning("Query drive info sn:{} timeout after upgrade".format(self.serial_number()))
        return False

    def farm_log(self):
        """
        Returns:
        """
        if self.info.get("vendor") == "SEAGATE":
            return self.mcs.get_seagate_farm_log(params=self.info)
        return ()

    def drive_letter(self):
        return self.info.get("drive_letter")

    def target(self):
        return self.info.get("target")

    def model_number(self):
        return self.info.get("model_number")

    def serial_number(self):
        return self.info.get("serial_number")

    def firmware_version(self):
        return self.info.get("firmware_version")

    def update_info(self):
        params = {
            "drive_letter": self.drive_letter(),
            "through_opt": self.target()
        }
        self.info.update(self.mcs.get_single_drive_info(params)[0])

    def poweroff(self):
        pass

    def get_special_fragement(self):
        pass


class AtaDrive(Drive):
    """
    ATA Entity
    """

    def __init__(self, info):
        super(AtaDrive, self).__init__(info)

    def smart(self):
        command = ["info show-dev",
                   "info show-rotation-rate",
                   "info show-ata-identify",
                   "info show-ata-smart",
                   "attr smart-status -m show",
                   "log smart-health",
                   "log show-ata-error",
                   "log show-ata-extend-error",
                   "log show-ata-extend-dst",
                   "log show-ata-phy-event",
                   "log show-hw-smart"
                   ]
        return self.mcs.get_disk_smart(self.drive_letter(), self.target(), self.serial_number(), command)

    def upgrade(self, params):
        """ATA disk firmware upgrade"""
        upgrade_params = {
            "drive_letter": self.info.get("drive_letter"),
            "through_opt": self.target()
        }
        upgrade_params.update(params)
        if "pmc" in self.target():
            upgrade_params.update({"fragment": "32768"})
        self.mcs.upgrade_sata_drive(upgrade_params)


class ScsiDrive(Drive):
    """
    SCSI Entity
    """

    def __init__(self, info):
        super(ScsiDrive, self).__init__(info)

    def smart(self):
        command = ["info show-dev",
                   "info show-rotation-rate",
                   "attr show-disk-erc",
                   "attr smart-status -m show",
                   "log smart-health",
                   "log show-scsi-supported-log",
                   "log show-scsi-error-count",
                   "log show-scsi-bms",
                   "log show-scsi-dst",
                   "log show-scsi-ie",
                   "log show-scsi-temperature",
                   "info show-defect-list -t grown",
                   "info show-defect-list -t primary"
                   ]
        return self.mcs.get_disk_smart(self.drive_letter(), self.target(), self.serial_number(), command)

    def upgrade(self, params):
        """SCSI disk firmware upgrade"""
        upgrade_params = {
            "drive_letter": self.info.get("drive_letter"),
            "through_opt": self.target()
        }
        upgrade_params.update(params)
        # 穿透PMC卡升级固件不支持64K分片，随机选择常用的32K
        if "pmc" in self.target():
            upgrade_params.update({"fragment": "32768"})
        self.mcs.upgrade_sas_drive(upgrade_params)


class NvmeDrive(Drive):
    """
    NVME Entity
    """

    def __init__(self, info):
        super(NvmeDrive, self).__init__(info)

    def smart(self):
        log_ids = self.mcs.get_pcie_support_log(self.drive_letter())
        command = ["info show-dev",
                   "info show-nvme-identify -t controller",
                   "info show-nvme-identify -t namespace",
                   "log show-nvme-log-supported",
                   "log show-hw-smart"]
        for log_id in log_ids:
            command.append("log show-nvme-log -p {}".format(log_id))
        return self.mcs.get_disk_smart(self.drive_letter(), self.target(), self.serial_number(), command)

    def upgrade(self, params):
        """NVMe disk firmware upgrade"""
        base_params = {
            "drive_letter": self.info.get("drive_letter")
        }
        base_params.update(params)
        self.mcs.download_nvme_firmware(base_params)
        self.mcs.active_nvme_firmware(base_params)
