#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 定义独立冗余磁盘阵列

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2024/2/29 created

"""
import os

from lib.core.tools.raid_card_tools import StoRCli64, ArcConf, HiRaidAdm
from lib.exception.dfs_exception import DFSException
from lib.utils.command import CommandUtil
from lib.utils.log import SingleLog


class CardFactoryMap(object):
    def __init__(self):
        self.log = SingleLog()

    def quqry_tool_path(self, raid_tool):
        cmd = "whereis -b {}".format(raid_tool)
        flag, msg, error = CommandUtil.exec_shell(cmd)
        if not flag:
            # 执行命令错误
            raise DFSException("excute whereis -b {} fail".format(raid_tool))
        if msg.strip() == "{}:".format(raid_tool):
            return ""
        return msg.split()[1]

    def get(self, exist_raid_info_list, is_get_specific_raid_info):
        # 判断工具存放路径是否存在
        storcli64_path, arcconf_path, hiraidadm_path = self._get_raid_tool_path()
        storcli64, arcconf, hiraidadm = None, None, None
        # 实例化各种工具
        if storcli64_path:
            storcli64 = StoRCli64(storcli64_path)
        if arcconf_path:
            arcconf = ArcConf(arcconf_path)
        if hiraidadm_path:
            hiraidadm = HiRaidAdm(hiraidadm_path)

        # 实例化工具
        result = []
        for raid_info in exist_raid_info_list:
            if raid_info["raid_tool"] == "storcli64":
                # 获取所有的Controller_id, raid_status
                if not storcli64:
                    # 没有工具此时
                    raise DFSException("storcli64 tool is not exist")
                controllerid_raidstatus_list = storcli64.get_cid_list()
                for controllerid_raidstatus in controllerid_raidstatus_list:
                    controller_id = controllerid_raidstatus["controller_id"]
                    raidstatus = controllerid_raidstatus["raid_status"]
                    # 申明 raid卡类
                    lsi = Lsi(controller_id, raidstatus, storcli64, raid_info["support_drive_version_list"],
                        raid_info["support_firmware_list"])
                    self._bound_specific_raid_info(lsi, is_get_specific_raid_info)
                    result.append(lsi)
            if raid_info["raid_tool"] == "arcconf":
                # 获取所有的Controller_id, raid_status
                if not arcconf:
                    # 没有工具此时
                    raise DFSException("arcconf tool is not exist")
                controllerid_raidstatus_list = arcconf.get_cid_list()
                for controllerid_raidstatus in controllerid_raidstatus_list:
                    controller_id = controllerid_raidstatus["controller_id"]
                    raidstatus = controllerid_raidstatus["raid_status"]
                    # 申明 raid卡类
                    pmc = Pmc(controller_id, raidstatus, arcconf, raid_info["support_drive_version_list"],
                              raid_info["support_firmware_list"])
                    self._bound_specific_raid_info(pmc, is_get_specific_raid_info)
                    result.append(pmc)
            if raid_info["raid_tool"] == "hiraidadm":
                # 获取所有的Controller_id
                if not hiraidadm:
                    # 没有工具此时
                    raise DFSException("hiraidadm tool is not exist")
                controllerid_list = hiraidadm.get_cid_list()
                for controller_id in controllerid_list:
                    hiraid = HiRaid(controller_id, hiraidadm, raid_info["support_drive_version_list"],
                           raid_info["support_firmware_list"])
                    self._bound_specific_raid_info(hiraid, is_get_specific_raid_info)
                    result.append(hiraid)
        return result

    def _get_raid_tool_path(self):
        if os.path.exists("/opt/MegaRAID/storcli/storcli64"):
            storcli64_path = "/opt/MegaRAID/storcli/storcli64"
        else:
            storcli64_path = self.quqry_tool_path("storcli64")

        if os.path.exists("/usr/Arcconf/arcconf"):
            arcconf_path = "/usr/Arcconf/arcconf"
        elif os.path.exists("/opt/SmartProvisioning/app/raid_cfg/tools/arcconf"):
            arcconf_path = "/opt/SmartProvisioning/app/raid_cfg/tools/arcconf"
        else:
            arcconf_path = self.quqry_tool_path("arcconf")
        hiraidadm_path = self.quqry_tool_path("hiraidadm")
        return storcli64_path, arcconf_path, hiraidadm_path

    def _bound_specific_raid_info(self, hiraid, is_get_specific_raid_info):
        if is_get_specific_raid_info:
            self.log.info("Get specific raid info")
            hiraid.get_raid_info()


class RaidCard(object):

    def __init__(self):
        self.chunk_size_tuple = (4096, 8192, 16384, 32768, 65536)
        # Raid卡下包含 drive信息
        self.drive_info = {}
        self.status = ""
        self.ctrl_id = ""
        self.init_drive_info = {}

    def chunk_size(self):
        return self.chunk_size_tuple


class Lsi(RaidCard):
    # , support_collect_version
    def __init__(self, controller_id, raidstatus, storcli64, support_drive_version=None,
                 support_firmware_version=None):
        super(Lsi, self).__init__()
        self.log = SingleLog()
        self.status = raidstatus
        self.ctrl_id = controller_id
        # 获取raid卡信息
        self.storcli = storcli64
        # 支持的驱动版本
        self.support_drive_version = support_drive_version
        # 支持的固件版本
        self.support_firmware_version = support_firmware_version

    def is_supported_log(self):
        """
            根据raid卡驱动版本和固件版本 是否支持内部日志收集
        Returns:

        """
        drive_dict = self._get_drive()
        self.log.warning("Current RAID drive version is {}".format(drive_dict.get("drive_version", "")))
        self.log.warning("Current RAID firmware version is {}".format(drive_dict.get("firmware_version", "")))
        if (drive_dict.get("drive_version", "") in self.support_drive_version and
                drive_dict.get("firmware_version", "") in self.support_firmware_version):
            return True
        return False

    def is_changed(self, sn):
        """
            升级后 状态对比
        Args:
            sn:

        Returns:

        """
        self.refresh_drive_group_by_sn(sn)
        if self.init_drive_info != self.drive_info:
            return True
        return False

    def refresh_drive_group_by_sn(self, sn):
        drive_group_id = self.drive_info.get(sn, {}).get("dg", {}).get("id", "")
        storcli64 = self.storcli
        # 刷新同物理组的信息
        param = {
            "controller_id": self.ctrl_id,
            "drive_group_id": drive_group_id
        }
        refresh_physical_virtual_info = storcli64.get_vd_info_by_drive_group(param)
        # 遍历所有的raid下的同组的raid信息
        for sn, raid in self.drive_info.items():
            if raid.get("dg", {}).get("id", "") == drive_group_id:
                self.drive_info.get(sn, {}).update(refresh_physical_virtual_info)

    def is_contain(self, sn):
        return sn in self.drive_info

    def is_normal(self):
        """
            升级前的状态检查
        Args:

        Returns:

        """
        # raid卡状态检查
        if self.status != "Opt":
            # 返回True/flase  记录日志
            self.log.error("Controller id {} raid status is error".format(self.ctrl_id))
            return False
        for sn, _ in self.init_drive_info.items():
            if self.drive_info.get(sn, {}).get("physic_state", "").upper() in ("JBOD", "UGOOD", "GHS", "DHS"):
                continue
            # 检查sn关联的物理 与 虚拟盘信息
            if self.drive_info.get(sn, {}).get("physic_state", "").upper() != "ONLN":
                self.log.error("{} raid physic_state is error, current raid physic_state is {}"
                           .format(sn,
                                   self.drive_info.get("sn", {}).get("physic_state", "")))
                return False
            for _, virtual_state in self.drive_info.get(sn, "").get("vd", {}).items():
                if virtual_state.upper() != "OPTL":
                    self.log.error("{} raid vir_health is error".format(sn))
                    return False
            # # 有后台任务就警告
            background_list = self.drive_info.get(sn, {}).get("dg", {}).get("bt", "")
            if any(element.upper() != 'N' for element in background_list):
                self.log.warning("{} background task in progress".format(sn))
        return True

    def get_raid_info(self):
        # 再获取 物理组信息
        param = {
            "controller_id": self.ctrl_id,
            "enclosure_id": "all",
            "slot_id": "all"
        }
        try:
            ret = self.storcli.get_pd_info(param)
        except DFSException as e:
            # 说明此时没有raid卡信息
            self.log.error(e.message)
            return
        # 组装物理数据
        for sn, drive_info in ret.items():
            temp = drive_info
            # 组装后台任务 和 虚拟盘信息
            # 如果物理盘状态是UGood F，错误状态，需要外部配置.drive_group_id参数错误，所以跳过虚拟状态获取
            if drive_info.get("physic_state", "").upper() == "UGOOD F":
                continue
            if not drive_info.get("dg", {}).get("id", "") in ("-", ""):
                drive_group_id = drive_info.get("dg", {}).get("id", "")
                param = {
                    "controller_id": self.ctrl_id,
                    "drive_group_id": drive_group_id
                }
                ret = self.storcli.get_vd_info_by_drive_group(param)
                temp.update(ret)
            self.drive_info[sn] = temp
        self.init_drive_info = self.drive_info

    def _get_drive(self):
        """
            获取raid卡驱动 和 固件版本
        Returns:
            {"drive_version": "", "firmware_version": ""}
            查询后的驱动信息
        """
        param = {
           "controller_id": self.ctrl_id
        }
        return self.storcli.get_drive_info(param)


class HiRaid(RaidCard):
    def __init__(self, controller_id,  hiraidadm, support_drive_version=None, support_firmware_version=None):
        super(HiRaid, self).__init__()
        self.log = SingleLog()
        self.ctrl_id = controller_id
        # msg 不同的control_id
        self.msg = {}
        # 获取raid卡信息
        self.hiraidadm = hiraidadm
        # 支持的驱动版本
        self.support_drive_version = support_drive_version
        # 支持的固件版本
        self.support_firmware_version = support_firmware_version

    def is_supported_log(self):
        drive_dict = self._get_drive()
        if (drive_dict.get("drive_version", "") in self.support_drive_version and
                drive_dict.get("firmware_version", "") in self.support_firmware_version):
            return True
        return False

    def is_changed(self, sn):
        """
            升级后 状态对比
        Args:
            sn:

        Returns:

        """
        self.refresh_drive_group_by_sn(sn)
        if self.init_drive_info == self.drive_info:
            return False
        return True

    def is_normal(self):
        """
            整个raid的状态检查
        Args:

        Returns:

        """
        # 遍历整个raid卡下的信息
        for sn, _ in self.init_drive_info.items():
            # 物理状态不正常
            if self.drive_info.get(sn, {}).get("physical_state", None).upper() != "ONLINE":
                self.log.error("{} physical_state is {}".
                               format(sn, self.drive_info.get(sn, {}).get("physical_state", None)))
                return False
            # 判断是否为JBOD模式
            if self.drive_info.get(sn, {}).get("dg", {}).get("id", None).upper() == "NA":
                continue
            for _, virtual_state in self.drive_info.get(sn, {}).get("vd", None).items():
                if virtual_state.upper() != "NORMAL":
                    self.log.error("{} raid vir_health is error".format(sn))
                    return False
                # 有后台任务就警告
                if self.drive_info.get(sn, {}).get("dg", {}).get("bt", None).upper() != 'NA':
                    self.log.warning("{} background task in progress".format(sn))
        return True

    def is_contain(self, sn):
        return sn in self.drive_info

    def refresh_drive_group_by_sn(self, target_sn):
        drive_group_id = self.drive_info.get(target_sn, {}).get("dg", {}).get("id", "")
        # JBOD模式刷新自己
        if drive_group_id in ("NA", ""):
            param = {
                "controller_id": self.ctrl_id,
                "enclosure_id": self.drive_info.get(target_sn, {}).get("enclosure_id", ""),
                "slot_id": self.drive_info.get(target_sn, "").get("slot_id", "")
            }
            physical_info = self.hiraidadm.get_pd_list(param)
            self.drive_info.get(target_sn, {}).update(physical_info)
            return
        # 刷新同组物理状态
        for sn, raid_msg in self.drive_info.items():
            if raid_msg.get("dg", {}).get("id", "") == drive_group_id:
                param = {
                    "controller_id": self.ctrl_id,
                    "enclosure_id": self.drive_info.get(sn, {}).get("enclosure_id", ""),
                    "slot_id": self.drive_info.get(sn, "").get("slot_id", "")
                }
                physical_info = self.hiraidadm.get_pd_list(param)
                self.drive_info.get(sn, {}).update(physical_info)
                #  更新虚拟信息
                param = {
                    "controller_id": self.ctrl_id
                }
                virtual_info = self.hiraidadm.get_vd_list(param)
                self.drive_info.get(sn, {}).get("vd", {}).update(virtual_info[drive_group_id])

    def get_raid_info(self):
        # 再获取 物理组信息
        param = {
            "controller_id": self.ctrl_id
        }
        enclosure_slot_list = self.hiraidadm.get_enclosure_slot_list(param)
        # 组装物理数据
        for enclosure_slot_msg in enclosure_slot_list:
            # 根据 control_id 与 槽位 信息 获取物理信息
            self._get_physic_virtual_msg(enclosure_slot_msg)

    def _get_drive(self):
        """
            绑定raid卡驱动
        Returns:
            {"drive_version": "", "firmware_version": ""}
            查询后的驱动信息
        """
        param = {
           "controller_id": self.ctrl_id
        }
        return self.hiraidadm.get_drive_info(param)

    def _get_physic_virtual_msg(self, enclosure_slot_msg):
        param = {
            "controller_id": self.ctrl_id,
            "enclosure_id": enclosure_slot_msg["enclosure_id"],
            "slot_id": enclosure_slot_msg["slot_id"]
        }
        physical_info = self.hiraidadm.get_pd_list(param)
        sn = physical_info.get("serial_number", "")
        # 组装物理信息
        self.drive_info[sn] = {
            "enclosure_id": enclosure_slot_msg["enclosure_id"],
            "slot_id": enclosure_slot_msg["slot_id"],
            "physical_state": physical_info.get("physical_state", None),
            "dg": {"id": physical_info.get("drive_group_id", "")},
            "vd": {}
        }
        if self.drive_info.get(sn, "").get("dg", "").get("id", None) != "NA":
            param = {
                "controller_id": self.ctrl_id
            }
            virtual_info = self.hiraidadm.get_vd_list(param)
            for drive_group_id, virtual_dict in virtual_info.items():
                if self.drive_info.get(sn, "").get("dg", {}).get("id", "") == drive_group_id:
                    self.drive_info[sn]["vd"] = virtual_dict
            # 组装后台任务
            param = {
                "controller_id": self.ctrl_id,
                "drive_group_id": self.drive_info.get(sn, {}).get("dg", {}).get("id", "")
            }
            background_task = self.hiraidadm.get_background_task_info(param)
            self.drive_info[sn]["dg"]["bt"] = background_task
        self.init_drive_info = self.drive_info

    def _is_raid_tool_exist(self, raid_tool):
        # 检测raid工具是否存在
        cmd = "whereis -b {}".format(raid_tool)
        flag, msg, error = CommandUtil.exec_shell(cmd)
        # 以
        if not flag:
            # 假若没有此工具
            raise DFSException("excute whereis -b {} fail".format(raid_tool))
        return True, msg.split(" ")[1]


class Pmc(RaidCard):

    def __init__(self, controller_id, raidstatus, arcconf, support_drive_version=None, support_firmware_version=None):
        super(Pmc, self).__init__()
        self.log = SingleLog()
        self.status = raidstatus
        self.ctrl_id = controller_id
        self.chunk_size_tuple = (4096, 8192, 16384, 32768)
        # msg 不同的control_id
        self.msg = {}
        # 获取raid卡信息
        self.arcconf = arcconf

        # 支持的驱动版本
        self.support_drive_version = support_drive_version
        # 支持的固件版本
        self.support_firmware_version = support_firmware_version

    def is_supported_log(self):
        drive_dict = self._get_drive()
        if (drive_dict.get("drive_version", "") in self.support_drive_version and
                drive_dict.get("firmware_version", "") in self.support_firmware_version):
            return True
        self.log.warning("Current drive version is {}".format(drive_dict.get("drive_version", "")))
        self.log.warning("Current firmware version is {}".format(drive_dict.get("firmware_version", "")))
        return False

    def is_normal(self):
        """
            升级前的状态检查
        Args:

        Returns:

        """
        # 遍历整个raid卡下的信息
        # raid卡状态检查
        if self.status.upper() != "OPTIMAL":
            # 返回True/flase  记录日志
            self.log.error("Controller_id :{} raid status is error".format(self.ctrl_id))
            return False
        for sn, _ in self.init_drive_info.items():
            # JBOD模式
            if self.drive_info.get(sn, {}).get("physic_state", "").upper() == "READY":
                continue
            # 检查sn关联的物理 与 虚拟盘信息
            if self.drive_info.get(sn, {}).get("physic_state", "").upper() != "ONLINE":
                self.log.error("{} raid physic_state is error,"" current raid physic_state is {}"
                           .format(sn, self.drive_info.get(sn, "").get("physic_state", "")))
                return False
            for _, virtual_state in self.drive_info.get(sn, {}).get("vd", {}).items():
                if virtual_state.upper() != "OPTIMAL":
                    self.log.error("{} raid vir_health is error, current raid virtual status is {}".
                                   format(sn, virtual_state))
                    return False
            # # 有后台任务就警告
            if self.drive_info.get("sn", {}).get("dg", {}).get("bt", None) is not None:
                self.log.warning("{} background task in progress".format(sn))
        return True

    def is_changed(self, sn):
        """
            升级后 状态对比
        Args:
            sn:

        Returns:

        """
        self.refresh_drive_group_by_sn(sn)
        if self.init_drive_info != self.drive_info:
            return True
        return False

    def refresh_drive_group_by_sn(self, target_sn):
        drive_group_id = self.drive_info.get(target_sn, {}).get("dg", {}).get("id", "")
        # 如果是JBOD模式 就刷新自己的物理信息
        if drive_group_id == "":
            channel = self.drive_info.get(target_sn, {}).get("Channel", "")
            device = self.drive_info.get(target_sn, {}).get("Device", "")
            param = {
                "controller_id": self.ctrl_id,
                "channel": channel,
                "device": device
            }
            physic_info = self.arcconf.get_pd_info(param)
            self.drive_info.get(target_sn, {}).update(physic_info)
            return
        # 刷新同物理组的信息
        for sn, raid_info in self.drive_info.items():
            if raid_info.get("dg", {}).get("id", "") == drive_group_id:
                channel = self.drive_info.get(sn, {}).get("Channel", "")
                device = self.drive_info.get(sn, {}).get("Device", "")
                param = {
                    "controller_id": self.ctrl_id,
                    "channel": channel,
                    "device": device
                }
                physic_info = self.arcconf.get_pd_info(param)
                self.drive_info.get(sn, {}).update(physic_info)
        param = {
            "controller_id": self.ctrl_id,
            "drive_group_id": drive_group_id
        }
        virture_info = self.arcconf.get_virtual_info_by_drive_group(param)
        self.drive_info.get(target_sn, {}).get("vd", {}).update(virture_info)

    def is_contain(self, sn):
        return sn in self.drive_info

    def get_raid_info(self):
        # 再获取 物理组信息
        param = {
            "controller_id": self.ctrl_id
        }
        ret = self.arcconf.get_pd_list(param)
        # 组装物理数据
        for sn, drive_info in ret.items():
            temp = drive_info
            # 组装后台任务 和 虚拟盘信息
            drive_group_id = drive_info["dg"]["id"]
            if drive_group_id != '':
                # 此时非JBOD模式
                param = {
                    "controller_id": self.ctrl_id,
                    "drive_group_id": drive_group_id
                }
                ret = self.arcconf.get_virtual_info_by_drive_group(param)
                temp["vd"].update(ret)
                # 获取后台任务
                param = {
                    "control_id": self.ctrl_id
                }
                ret = self.arcconf.get_background_task(param)
                temp["dg"].update(ret)
            self.drive_info[sn] = temp
        self.init_drive_info = self.drive_info

    def _get_drive(self):
        """
            绑定raid卡驱动
        Returns:
            {"drive_version": "", "firmware_version": ""}
            查询后的驱动信息
        """
        param = {
           "controller_id": self.ctrl_id
        }
        return self.arcconf.get_drive_info(param)

    def _is_raid_tool_exist(self, raid_tool):
        # 检测raid工具是否存在
        cmd = "whereis -b {}".format(raid_tool)
        flag, msg, error = CommandUtil.exec_shell(cmd)
        # 以
        if not flag:
            # 假若没有此工具
            raise DFSException("excute whereis -b {} fail".format(raid_tool))
        return True, msg.split(" ")[1]