#!/usr/bin/python
# -*- coding: UTF-8 -*-

"""
功    能: 提供博通卡基础wrapper

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2024/03/21 19:00 created

"""
import copy
import os
import re
from lib.exception.dfs_exception import DFSException
from lib.utils.command import CommandUtil
from lib.utils.validate import ValidateUtil


class StoRCli64(object):

    def __init__(self, tool_path):
        self.tool_path = tool_path

    def get_cid_list(self):

        def _parse_ret(raw_output):
            ret = {
                "status_code": "0",
                "output": []
            }
            cid_num = 0
            # 获取命令状态
            if not re.search(r"Status\s+=\s+success", raw_output, re.I):
                ret["status_code"] = "1"
                ret["output"] = {"error_info": raw_output}
                return ret
            for line in raw_output.split("\n"):
                if re.search(r"\s*\d+\s+[-\w]+\s+\d+\s+\d+\s+", line):
                    controller_info = line.split()
                    if re.search("3008", controller_info[1]):
                        cid_num -= 1
                        continue
                    dic = {
                        "controller_id": controller_info[0],
                        "raid_status": controller_info[-1]
                    }
                    ret["output"].append(dic)
            return ret

        cmd = "{} show".format(self.tool_path)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parse_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", [])

    @ValidateUtil.validate_param(params=dict)
    def get_pd_info(self, params):
        """
        获取硬盘详细信息
        """

        def _parse_ret(raw_output):
            ret = {
                    "physic_state": "",
                    "enclosure_id": "",
                    "slot_id": "",
                    "dg": {"id": ""}
            }
            result = {
                "status_code": "0",
                "output": {
                    }
            }
            if not re.search(r"Status\s+=\s+success", raw_output, re.I):
                result["status_code"] = "1"
                result["output"] = {"error_info": raw_output}
                return result
            for line in raw_output.split("\n"):
                # 特殊匹配出UGOOD
                disk_info = re.match(r"^(\d+:\d+)\s+(\d+)\s+(\w+)\s+(\S+)\s+", line)
                if disk_info:
                    ret["enclosure_id"] = line.split()[0]
                    ret["slot_id"] = line.split()[1]
                    ret['physic_state'] = line.split()[2]
                    ret['dg']["id"] = line.split()[3]
                    ret = self._update_raid_physic_state_an_drive_group(ret)
                key = None
                if re.search(r"^SN\s+=", line):
                    key = line.split("=")[-1].strip()
                if key is None:
                    continue
                result["output"][key] = copy.deepcopy(ret)
                ret = {
                    "physic_state": "",
                    "enclosure_id": "",
                    "slot_id": "",
                    "dg": {"bt": [], "id": ""}
                }
            return result

        controller_id = params.get("controller_id")
        enclosure_id = params.get("enclosure_id")
        slot_id = params.get("slot_id")
        if not all([controller_id, enclosure_id, slot_id]):
            raise DFSException(" Parameter 'controller_id',"
                               "'enclosure_id' or 'slot_id' "
                               "is None, please check.")
        cmd = "{} /c{}/e{}/s{} show all".format(self.tool_path, controller_id, enclosure_id, slot_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parse_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", {})

    @ValidateUtil.validate_param(params=dict)
    def get_vd_info_by_drive_group(self, params):
        """
            Args:
                1.controller_id
                2.raid组 id
        获取指定虚拟盘信息, 后台任务, 物理盘信息
        when：初始化、更新
            Returns
                    {
                        dgID:[
                            后台任务,
                            物理状态,
                            虚拟盘list[虚拟盘id:虚拟盘状态]
                        ]
                    }
                    1）虚拟盘ID
                    2）虚拟盘状态
                    3）硬盘组对应的后台任务
                    4）物理盘状态
        """

        def _parse_ret(raw_output, drive_group):
            ret = {
                       "dg": {
                           "id": drive_group,
                           "bt": ""
                       },
                       "vd": {}
            }
            result = {
                "status_code": "0",
                "output": {}
            }
            if not re.search(r"Status\s+=\s+success", raw_output, re.I):
                result["status_code"] = "1"
                result["output"] = {"error_info": raw_output}
                return result
            background_task = []
            for line in raw_output.split("\n"):
                # 正则匹配后台任务  0 0   0   252:0    2   DRIVE Onln  N  557.861 GB dsbl N  N   dflt -      N
                if re.search(r"^\d+\s+\d+\s+\d+\s+\d+:\d+\s+\d+\s+DRIVE", line.strip()):
                    array = line.split()
                    background_task.append(array[7])
                if re.search(r"^\d+\s+.\s+.\s+.\s+.\s+RAID", line.strip()):
                    array = line.split()
                    background_task.append(array[7])
                # 正则匹配 0/0   RAID0 Optl  RW     Yes     RWTD  -   ON  557.861 GB  虚拟信息
                if re.search(r"^\d+/\d+\s+RAID\d+\s+", line.strip()):
                    array = line.split()
                    virtual_id = array[0].split("/")[1]
                    virtual_status = array[2]
                    ret["vd"][virtual_id] = virtual_status
            ret["dg"]["bt"] = background_task
            result["output"] = ret
            return result

        controller_id = params.get("controller_id")
        drive_group = params.get("drive_group_id")
        if not controller_id:
            raise DFSException("Parameter 'controller_id' or 'drive_group_id' is None , please check.")
        if drive_group == "-":
            return {
                "status_code": "0",
                "output": {}
            }
        cmd = "{} /c{}/d{} show all".format(self.tool_path, controller_id, drive_group)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parse_ret(res[1], drive_group)
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", {})

    def get_drive_info(self, param):
        """
            获取当前raid控制器的驱动信息
        Args:
            param:
                controller_id
        Returns:

        """
        ret = {"status_code": "0", "output": {"drive_version": "", "firmware_version": ""}}

        def _parse_ret(raw_out):
            for line in raw_out:
                if re.search(r"^Driver Version\s[=]\s", line):
                    ret["output"]["drive_version"] = line.split("=")[1].strip()
                if re.search(r"^FW Version\s[=]\s", line):
                    ret["output"]["firmware_version"] = line.split("=")[1].strip()
            return  ret
        controller_id = param.get("controller_id")
        cmd = "{} /c{} show".format(self.tool_path, controller_id)
        res = CommandUtil.exec_shell(cmd)
        result = _parse_ret(res[1].split("\n"))
        if not res[0]:
            raise DFSException(res[1])
        ValidateUtil.check_status_code(result)
        return result.get("output", "")

    def _update_raid_physic_state_an_drive_group(self, ret):
        # 特殊情况处理 如physic_state为UGOOD F，并且其F字符将dg数据覆盖，则将physic_state还原为UGOOD F，dg置空
        for char in ret['dg']["id"]:
            ascii_value = ord(char)
            if ascii_value >= 65 and ascii_value <= 90:
                ret['physic_state'] = ret['physic_state'] + " " + ret['dg']["id"]
                ret['dg']["id"] = ""
                return ret
            if ascii_value >= 97 and ascii_value <= 122:
                ret['physic_state'] = ret['physic_state'] + " " + ret['dg']["id"]
                ret['dg']["id"] = ""
            return ret


class ArcConf(object):

    def __init__(self, tool_path):
        self.tool_path = tool_path
        os.chmod(tool_path, 755)

    def get_cid_list(self):
        """
        查询所有PMC卡controller id
        """

        def _parser_ret(raw_output):
            ret = {"status_code": "0", "output": []}
            controller_infos = re.findall(r"Controller\s+(\d+):\s+:\s+(.*)", raw_output)
            for controller_info in controller_infos:
                ret.get("output").append({
                    "controller_id": controller_info[0],
                    "raid_status": controller_info[1].split(',')[0].strip()
                })
            return ret

        cmd = "{} list".format(self.tool_path)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", [])

    @ValidateUtil.validate_param(params=dict)
    def get_pd_list(self, params):
        """
        PMC卡下查询指定硬盘信息
        """

        def _parser_ret(raw_output):
            result = {
                "status_code": "0",
                "output": {}
            }
            ret = {
                "physic_state": "",
                "Channel": "",
                "Device": "",
                "dg": {"id": "", "bt": []},
                "vd": {}
            }
            serial_number = ""
            for line in raw_output:
                if re.search(r"State\s+:\s+", line):
                    ret["physic_state"] = line.split(":")[1].strip()
                if re.search(r"Reported Channel,Device\(T:L\)\s+:\s+", line):
                    if ret["physic_state"] == "":
                        continue
                    ret["Channel"] = line.split(":")[2].split("(")[0].split(",")[0]
                    ret["Device"] = line.split(":")[2].split("(")[0].split(",")[1]
                if re.search(r"Serial number\s+:\s+", line):
                    serial_number = line.split(":")[1].strip()
                if re.search(r"^\s+Array\s+:", line):
                    # Onln 非JBOD模式 有物理组id
                    if ret["physic_state"] == "Online":
                        ret["dg"]["id"] = line.split(":")[1].strip()
                if serial_number != "":
                    result["output"][serial_number] = ret
                    serial_number = ""
                    ret = {
                        "physic_state": "",
                        "Channel": "",
                        "Device": "",
                        "dg": {"id": "", "bt": []},
                        "vd": {}
                    }
            return result

        controller_id = params.get("controller_id")
        if not all([controller_id]):
            raise DFSException("The parameter is incorrect, please check.")

        cmd = "{} getconfig {} pd".format(self.tool_path, controller_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1].split("\n"))
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", {})

    @ValidateUtil.validate_param(params=dict)
    def get_pd_info(self, params):
        """
        PMC卡下查询指定硬盘信息
        """

        def _parser_ret(raw_output):
            ret = {
                "status_code": "0",
                "output": {
                    "physic_state": ""
                }
            }
            for line in raw_output:
                if re.search(r"State\s+:\s+", line):
                    ret["output"]["physic_state"] = line.split(":")[1].strip()
            return ret

        control_id = params.get("controller_id")
        channel = params.get("channel")
        device = params.get("device")
        if not all([control_id, channel, device]):
            raise DFSException("The parameter is incorrect, please check.")

        cmd = "{} getconfig {} pd {} {}".format(self.tool_path, control_id, channel, device)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1].split("\n"))
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", {})

    @ValidateUtil.validate_param(params=dict)
    def get_virtual_info_by_drive_group(self, param):
        def _parser_ret(raw_output):
            ret = {
                "status_code": "0",
                "output": {

                }
            }
            for line in raw_output:
                if re.search(r"\sLogical\s\d\s", line):
                    logic_id = line.split(":")[0].split()[1]
                    ret["output"][logic_id] = line.split(":")[1].split("(")[0].strip()
            return ret

        controller_id = param.get("controller_id")
        drive_group_id = param.get("drive_group_id")
        if not all([controller_id, drive_group_id]):
            raise DFSException("The parameter is incorrect, please check.")
        cmd = "{} getconfig {} ar {}".format(self.tool_path, controller_id, drive_group_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1].split("\n"))
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", {})

    @ValidateUtil.validate_param(params=dict)
    def get_background_task(self, param):
        """
        PMC卡下查询raid后台任务
        """
        ret = {"status_code": "0", "output": {"bt": ""}}
        controller_id = param.get("control_id")
        if not controller_id:
            raise DFSException("The parameter is incorrect, please check.")

        cmd = "{} getstatus {}".format(self.tool_path, controller_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        bt_state_obj = re.search(r"Current operation\s+:\s+(.*)", res[1])
        if bt_state_obj:
            ret["output"]["bt"] = bt_state_obj.group(1)
        ValidateUtil.check_status_code(ret)
        return ret.get("output", {})

    def get_drive_info(self, param):
        """
            获取当前raid控制器的驱动信息
        Args:
            param:
                controller_id
        Returns:

        """
        ret = {"status_code": "0", "output": {"drive_version": "", "firmware_version": ""}}

        def _parse_ret(raw_out):
            for line in raw_out:
                if re.search(r"^Driver\s+[:]", line):
                    ret["output"]["drive_version"] = line.split(":")[1].strip().split()[1]
                if re.search(r"^Firmware\s+[:]", line):
                    ret["output"]["firmware_version"] = line.split(":")[1].strip()
            return ret
        controller_id = param.get("controller_id")
        cmd = "{} getversion {} ".format(self.tool_path, controller_id)
        res = CommandUtil.exec_shell(cmd)
        result = _parse_ret(res[1].split("\n"))
        if not res[0]:
            raise DFSException(res[1])
        ValidateUtil.check_status_code(result)
        return result.get("output", "")


class HiRaidAdm(object):

    def __init__(self, tool_path):
        self.tool_path = tool_path
        os.chmod(tool_path, 755)

    @ValidateUtil.validate_param(params=dict)
    def get_enclosure_slot_list(self, params):
        """查询指定raid卡下硬盘槽位"""

        def _parser_ret(raw_output):
            result = {
                "status_code": "0",
                "output": []
            }
            pd_num = 0
            if not re.search(r"Status\s+=\s+Success", raw_output):
                result["status_code"] = "1"
                result["output"] = {"error_info": raw_output}
                return result
            for line in raw_output.split("\n"):
                ret = {
                    "enclosure_id": "",
                    "slot_id": ""
                }
                disk_info = re.search(r"^(\d+)\s+(\d+)\s+(\d+)\s+\d+", line)
                if disk_info:
                    # 返回槽位信息
                    ret["enclosure_id"] = line.split()[1]
                    ret["slot_id"] = line.split()[2]
                    result["output"].append(ret)
                if re.search(r"Total\s+pd\s+number\s+is\s+\d+", line):
                    pd_num = int(line.split()[-1].strip())
            if pd_num != len(result["output"]):
                result["status_code"] = "1"
                result["output"] = {"error_info": raw_output}
            return result

        controller_id = params.get("controller_id")
        if not controller_id:
            raise DFSException("Parameter 'controller_id' is empty, please check.")
        cmd = "{} c{} show pdlist".format(self.tool_path, controller_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", [])

    @ValidateUtil.validate_param(params=dict)
    def get_pd_list(self, params):
        """查询指定硬盘"""

        def _pd_info_parse(raw_output):
            output = {
                "physical_state": "",
                "drive_group_id": "",
                "serial_number": ""
            }
            for line in raw_output.split("\n"):
                if re.search(r"Serial\s+Number\s+\|", line):
                    output["serial_number"] = line.split('|')[-1].strip()
                if re.search(r"^RAID\s+Group\s+ID\s+\|\s+", line):
                    output["drive_group_id"] = line.split('|')[-1].strip()
                if re.search(r"^Status\s+\|", line):
                    output["physical_state"] = line.split('|')[-1].strip()
            return output

        def _parser_ret(raw_output):
            ret = {
                "status_code": "0",
                "output": {},
            }
            if not re.search(r"Status\s+=\s+Success", raw_output):
                ret["status_code"] = "1"
                ret["output"] = {"error_info": raw_output}
                return ret
            ret["output"] = _pd_info_parse(raw_output)
            return ret

        controller_id = params.get("controller_id")
        enclosure_id = params.get("enclosure_id")
        slot_id = params.get("slot_id")
        if not all([controller_id, enclosure_id, slot_id]):
            raise DFSException("Parameter 'controller_id' or 'enclosure_id' or 'slot_id' is empty, please check.")
        cmd = "{} c{}:e{}:s{} show".format(
            self.tool_path, controller_id, enclosure_id, slot_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", {})

    @ValidateUtil.validate_param(params=dict)
    def get_vd_list(self, params):
        """查询指定硬盘"""

        def _vd_info_parse(raw_output):
            output = {}
            for line in raw_output.split("\n"):
                if re.search(r"^\d\s+\d\s+\w+", line):
                    drive_group_id = line.split()[1]
                    virtual_id = line.split()[0]
                    if drive_group_id in output:
                        output[drive_group_id].append({virtual_id: line.split()[3]})
                        break
                    output[drive_group_id] = {virtual_id: line.split()[3]}
            return output

        def _parser_ret(raw_output):
            ret = {
                "status_code": "0",
                "output": {},
            }
            if not re.search(r"Status\s+=\s+Success", raw_output):
                ret["status_code"] = "1"
                ret["output"] = {"error_info": raw_output}
                return ret
            ret["output"] = _vd_info_parse(raw_output)
            return ret

        controller_id = params.get("controller_id")
        if not controller_id:
            raise DFSException("Parameter 'controller_id' or 'enclosure_id' or 'slot_id' is empty, please check.")
        cmd = "{} c{} show vdlist".format(
            self.tool_path, controller_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", [])

    @ValidateUtil.validate_param(params=dict)
    def get_background_task_info(self, param):
        def _background_task_info_parse(raw_output):
            for line in raw_output.split("\n"):
                if re.search(r"^Current\sBackground\sTask\s", line):
                    background_task = line.split("|")[1].strip()
            return background_task

        def _parse_ret(raw_output):
            ret = {
                "status_code": "0",
                "output": {},
            }
            if not re.search(r"Status\s+=\s+Success", raw_output):
                ret["status_code"] = "1"
                ret["output"] = {"error_info": raw_output}
                return ret
            ret["output"] = _background_task_info_parse(raw_output)
            return ret

        controller_id = param.get("controller_id")
        drive_group_id = param.get("drive_group_id")
        if not all([controller_id, drive_group_id]):
            raise DFSException("Parameter 'controller_id' or 'rg_id' is empty, please check.")
        cmd = "{} c{}:rg{} show".format(self.tool_path, controller_id, drive_group_id)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parse_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", "")

    def get_cid_list(self):
        """查询获取所有controller id"""

        def _parse_ret(raw_output):
            ret = {
                "status_code": "0",
                # 和其他wrapper保持同样的数据结构
                "output": []
            }
            if not re.search(r"Status\s+=\s+Success", raw_output):
                ret["status_code"] = "1"
                ret["output"] = {"error_info": raw_output}
                return ret
            raid_num = 0
            for line in raw_output.split("\n"):
                if re.search(r"^Controllers\s+Number\s+=\s+\d+", line):
                    raid_num = int(line.split("=")[-1].strip())
                if re.search(r"^Controller\s+Id\s+\|", line):
                    ret["output"].append(line.split("|")[-1].strip())
            if raid_num != len(ret["output"]):
                ret["status_code"] = "1"
                ret["output"] = {"error_info": raw_output}
            return ret

        cmd = "{} show allctrl".format(self.tool_path)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parse_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", [])

    def get_drive_info(self, param):
        """
            获取当前raid控制器的驱动信息
        Args:
            param:
                controller_id
        Returns:

        """
        ret = {"status_code": "0", "output": {"drive_version": "", "firmware_version": ""}}

        def _parse_ret(fw_raw_out, drive_raw_out):
            for line in fw_raw_out:
                if re.search(r"^\w+\s+\w+\s+\w+\.\w+\.\w+\.\w+\s+", line):
                    ret["output"]["firmware_version"] = line.split()[2]
                    break
            for line in drive_raw_out:
                if re.search(r"^version[:]\s", line):
                    ret["output"]["drive_version"] = line.split(":")[1].strip()
                    break
            return ret
        controller_id = param.get("controller_id")
        cmd = "{} c{} show fw_version".format(self.tool_path, controller_id)
        fw_res = CommandUtil.exec_shell(cmd)
        cmd = "modinfo hiraid"
        drive_res = CommandUtil.exec_shell(cmd)
        result = _parse_ret(fw_res[1].split("\n"), drive_res[1].split("\n"))
        ValidateUtil.check_status_code(result)
        return result.get("output", "")