#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 服务入口

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2023/7/11 18:00 created

"""
import copy
import json
import os
import platform
import re
import shutil
import sys
import time
import xml.etree.ElementTree as et
from random import randint

from config.constants import ReturnStatus, SPECIAL_DISK_UPGRADE_PARAMS, DFSROOT
from config.raid_map import RaidMap
from lib.core.drives.drive import DriveFactoryMap
from lib.core.raid_cards.raid_card import CardFactoryMap
from lib.core.tools.mcscli import McsCli
from lib.core.tools.pack import Tar
from lib.core.tools.pack import Zip
from lib.exception.dfs_exception import DFSException
from lib.utils.command import CommandUtil
from lib.utils.csv_util import CsvUtil
from lib.utils.log import SingleLog
from lib.utils.manager import FirmwareManager, EXECUTING, ResourceManager, ProgressManager

G_PY_VERSION = platform.python_version().split(".")[0]

UPGRADE_DICT = {"sn_list": []}


class Service(object):
    def __init__(self, filter_sn, filter_model, filter_fw, log_storage_dir):
        self.filter_sn = filter_sn
        self.filter_model = filter_model
        self.log_storage_dir = log_storage_dir if log_storage_dir else DFSROOT
        self.mcs = McsCli()
        # 新增属性 fw_list
        self.fw = filter_fw
        self.drives = []
        self.project_path = os.path.realpath(os.path.dirname(os.path.dirname(__file__)))
        self.log = SingleLog()
        self.internal_progress_file = os.path.join(DFSROOT, "internal_progress.txt")  # 给一个默认的，在内部日志执行时修改
        self.upgrade_progress_file = os.path.join("upgrade_process.txt")
        self.tar = Tar()
        self.zip = Zip()
        self.return_status = {}
        self.dfs_log = os.path.join(DFSROOT, "dfs.log")
        self.mcs_log_path = os.path.join("/", "opt", "mcs", "log", "mcslog.txt")
        self.sys_message_path = os.path.join("/", "var", "log", "messages")

    @staticmethod
    def _output(dict_msg):
        print(json.dumps(dict_msg))
        sys.stdout.flush()

    @staticmethod
    def _match_upgrade_config(model_number, firmware_version, all_config_info, scene):
        """match the upgrade configuration based on the disk information"""

        def _match_info(match_firmware):
            for item in all_config_info:
                condition_model = model_number == item.get("Model")
                condition_fw = match_firmware in item.get("VersionList")
                check_new_fw = match_firmware != item.get("TargetVersion")
                if scene == "storage" and condition_model and check_new_fw:
                    return item
                elif condition_model and condition_fw and check_new_fw:
                    return item
            return {}

        match_info_list = list()
        tmp_firmware = firmware_version
        while True:
            match_info = _match_info(tmp_firmware)
            if match_info and match_info not in match_info_list:
                match_info_list.append(match_info)
                # update firmware, lookup again.
                tmp_firmware = match_info.get("TargetVersion")
                continue
            else:
                break
        return match_info_list

    @staticmethod
    def _read_xml(config_path):
        """parse version.xml"""
        ret = {
            "FirmwareFileName": "",
            "TargetVersion": "",
            "Model": "",
            "VersionList": "",
            "InternalResourceCheck": "",
            "SmartWhiteListcheck": "",
            "UpgradeRaidStatuscheck": ""
        }
        if not os.path.isfile(config_path):
            raise DFSException("Config file is not found! '%s'" % config_path)
        tree = et.ElementTree(file=config_path)
        for node in tree.iter():
            if node.tag == "FileName":
                ret["FirmwareFileName"] = node.text
            elif node.tag == "Version":
                ret["TargetVersion"] = node.text
            elif node.tag == "SupportModel":
                ret["Model"] = node.text
            elif node.tag == "RuleOldVersion":
                ret["VersionList"] = node.text.replace("|", ";")
            # 是否进行内部日志收集资源检查
            elif node.tag == "InternalResourceCheck":
                ret["InternalResourceCheck"] = node.text
            # 是否进行内部日志收集白名单检查
            elif node.tag == "InternalWhiteListcheck":
                ret["InternalWhiteListcheck"] = node.text
            # 是否进行Smart白名单检查
            elif node.tag == "SmartWhiteListcheck":
                ret["SmartWhiteListcheck"] = node.text
            # 固件升级状态检查
            elif node.tag == "UpgradeRaidStatuscheck":
                ret["UpgradeRaidStatuscheck"] = node.text
        return ret

    @staticmethod
    def _parse_storage_config(config_path):
        """parse hddfw.conf"""
        output = list()
        if not os.path.isfile(config_path):
            raise DFSException("Config file is not found! '%s'" % config_path)
        if G_PY_VERSION == "3":
            with open(config_path, 'r', encoding="gbk") as conf:
                all_lines = conf.readlines()
        else:
            with open(config_path, 'r') as conf:
                all_lines = conf.readlines()
        for line in all_lines:
            if line.startswith("//"):
                continue
            split_line = line.split(":")
            output.append({
                "FirmwareFileName": split_line[2],
                "TargetVersion": split_line[1],
                "Model": split_line[0],
                "VersionList": "",
                "FirmwareFile": os.path.join(os.path.dirname(config_path), split_line[2])
            })
        return output

    def get_all_disk(self):
        self.log.info("Start get_all_disk")
        self._target_disk()
        info = []
        for drive in self.drives:
            model_number = drive.model_number()
            firmware_version = drive.firmware_version()
            sn = drive.serial_number()
            firmware_target_version = "notFound"
            firmware_config_path = os.path.join(DFSROOT, "upgrade", "firmware_config.csv")
            if os.path.exists(firmware_config_path):
                fw_manager = FirmwareManager(firmware_config_path)
                match_info_list = fw_manager.get_firmware_info()
                if match_info_list and (drive.model_number(), drive.firmware_version()) in match_info_list.keys():
                    # 找对对应的升级文件
                    firmware_target_version = match_info_list[(drive.model_number(), drive.firmware_version())]\
                    [len(match_info_list[(drive.model_number(), drive.firmware_version())]) - 1][0]
            active = "false"
            match_model = SPECIAL_DISK_UPGRADE_PARAMS.get(model_number)
            if match_model and match_model.get("active") and firmware_version in match_model.get("currentFW"):
                active = "true"
            target = drive.target()
            info.append("{}:{}:{}:{}:{}:{}"
                        .format(sn, model_number, firmware_version, firmware_target_version, active, target))
        if info:
            result = {"status": ReturnStatus.SUCCESS, "info": "\n".join(info)}
        else:
            result = {"status": ReturnStatus.FAIL, "desc": "No target disk"}
        self._output(result)

    def smart(self):
        self.log.info("Start collect smart service")
        if not os.path.exists(self.log_storage_dir):
            result = {"status": ReturnStatus.FAIL, "desc": "Specify log storage dir not exist"}
            self._output(result)
            return
        self._target_disk()

        if not self.drives:
            self._output({"status": ReturnStatus.FAIL, "desc": "No target disk"})
            return

        #白名单检查
        check_dict = self._read_xml(os.path.join(DFSROOT, "version.xml"))
        if check_dict.get("SmartWhiteListcheck", None).upper() == "TRUE" and not self.is_in_white_list():
            result = {"status": ReturnStatus.FAIL, "desc": "Not in the RAID white list"}
            self._output(result)
            return

        task_id = "{}{}".format(int(time.time()), randint(1000, 9999))
        pack_smart_log = os.path.join(self.log_storage_dir, "smart_log_{}".format(task_id))
        # 存放的路径
        os.mkdir(pack_smart_log)
        for drive in self.drives:
            # 收集硬盘SMART
            smart_path = drive.smart()
            # 收集硬盘FARM LOG - 此日志收集时间较短,故放在SMART信息收集项完成
            farm_log_path = drive.farm_log()
            for f_log in farm_log_path:
                if os.path.exists(f_log):
                    # 移动farm_log文件到对应的smart目录 smart_log/sn/
                    shutil.move(f_log, smart_path)
            if os.path.exists(smart_path):
                shutil.move(smart_path, pack_smart_log)
        self.log.info("Mcs collect smart log done.")
        self._save_log_file(pack_smart_log)
        self.log.info("Start to make_archive.")
        shutil.make_archive(base_name=pack_smart_log, format='tar', root_dir=pack_smart_log)
        shutil.rmtree(pack_smart_log)
        full_tar_name = "smart_log_{}.tar".format(task_id)
        result = {"status": ReturnStatus.SUCCESS, "file_path": os.path.join(self.log_storage_dir, full_tar_name)}
        self._output(result)
        self.log.info("Complete collect smart service.")

    def internal_log(self):
        try:
            self.log.info("Start collect internal log")
            if not os.path.exists(self.log_storage_dir):
                self._output({"status": ReturnStatus.FAIL, "desc": "specify log storage dir not exist"})
                return
            self._target_disk()
            if not self.drives:
                self._output({"status": ReturnStatus.FAIL, "desc": "No target disk"})
                return
            # 读取是否需要白名单
            check_dict = self._read_xml(os.path.join(DFSROOT, "version.xml"))
            if check_dict.get("InternalWhiteListcheck", None).upper() == "TRUE" and not self.is_in_white_list():
                self._output({"status": ReturnStatus.FAIL, "desc": "Not in the RAID white list"})
                return
            task_id = "{}{}".format(int(time.time()), randint(1000, 9999))
            pack_logs = os.path.join(self.log_storage_dir, "internal_log_{}".format(task_id))
            os.mkdir(pack_logs)
            self.internal_progress_file = os.path.join(DFSROOT, "internal_progress_{}.txt".format(task_id))
            pm = ProgressManager(self.internal_progress_file)
            sn_list = [drive.serial_number() for drive in self.drives]
            pm.add_batch(sn_list)
        except Exception as err:
            self._output({"status": ReturnStatus.FAIL, "desc": str(err)})
            return
        self._output({"status": ReturnStatus.SUCCESS, "timeout": len(sn_list) * 300, "id": task_id})
        for drive in self.drives:
            sn = drive.serial_number()
            drive_letter = drive.drive_letter()
            target = drive.target()
            if not self.mcs.is_support_internal_log(target, drive_letter):
                # mcs不支持升级结果设置为成功
                self.log.warning(
                    "sn:{},model:{},not support collect internal log".format(sn, drive.model_number()))
                pm.set_success(sn)
                continue
            if (check_dict.get("InternalResourceCheck", None).upper() == "TRUE" and
                    not self._is_passed_resource_check(drive)):
                self.log.warning("Sn:{},model:{},resource check not passed".format(sn, drive.model_number()))
                pm.set_fail(sn, "Resource check is not passed")
                break
            res, log_path = drive.internal()
            if os.path.exists(log_path):
                shutil.move(log_path, pack_logs)
            if res:
                pm.set_success(sn)
            else:
                pm.set_fail(sn, "collect internal Fail,see dfs.log for details")
        sn_list = pm.query_sn_by_status(EXECUTING)
        pm.patch_set_a_state_to_fail(EXECUTING, sn_list, "not execute collect command")
        self._save_log_file(pack_logs)
        shutil.make_archive(base_name=pack_logs, format='tar', root_dir=pack_logs)
        shutil.rmtree(pack_logs)

    def is_in_white_list(self):
        self.log.warning("Start to check whether the environment configuration is in the RAID white list")
        # 判断lspci工具是否存在
        if not self.is_exist_lspci():
            self.log.warning("Lspci is not exist")
            return False
        # 维护中的raid卡信息
        raid_info_list = self._get_all_raid_list()
        # 利用工厂类 生产出 raid实例 找到与 当前盘对应的raid卡
        raid_list = CardFactoryMap().get(raid_info_list, False)
        for raid in raid_list:
            # 判断raid卡驱动和 固件版本是否在白名单内
            if not raid.is_supported_log():
                self.log.warning("Current RAID drive and firmware is not in the RAID white list")
                return False
        return True

    def internal_progress(self, task_id):
        """
        Returns:
            {"status":"success","file_path":"/root/path/to/internal_log.tar"}
            {"status":"failure","desc":"xxxxx","file_path":"/root/path/to/internal_log.tar"}
            {"status":"executing","progress":"43"}
        """
        if not task_id:
            ret = {"status": ReturnStatus.FAIL, "desc": "need to specify the taskid"}
            self._output(ret)
            return
        self.internal_progress_file = os.path.join(DFSROOT, "internal_progress_{}.txt".format(task_id))
        pm = ProgressManager(self.internal_progress_file)
        proc_result = pm.get_process()
        status = proc_result.get("status")
        info = proc_result.get("info")
        ret = {}
        if status == ReturnStatus.SUCCESS:
            ret = {"status": ReturnStatus.SUCCESS}
            # 删除进度文件
            os.remove(self.internal_progress_file)
        elif status == ReturnStatus.FAIL:
            ret = {"status": ReturnStatus.FAIL, "desc": info}
            os.remove(self.internal_progress_file)
        elif status == ReturnStatus.RUNNING:
            ret = {"status": ReturnStatus.RUNNING, "progress": info}
        self._output(ret)

    def upgrade_firmware(self):
        self.log.info("Start upgrade firmware")
        model_list = self.filter_model
        fw_list = self.fw

        # 验证参数
        if len(fw_list) != 0:
            if len(model_list) != len(fw_list):
                return_info = {"status": ReturnStatus.FAIL, "desc": "Model number length not equals fireware length"}
                self._output(return_info)
                return

        # 配置文件是否存在
        firmware_config_path = os.path.join(DFSROOT, "upgrade", "firmware_config.csv")
        if not self._is_exist_config_file(firmware_config_path):
            return_info = {"status": ReturnStatus.FAIL, "desc": "firmware config is no exist"}
            self._output(return_info)
            return
        self._target_disk()

        # 待升级列表是否为空
        if len(self.drives) == 0:
            return_info = {"status": ReturnStatus.FAIL, "desc": "no target drives to upgrade"}
            self._output(return_info)
            return

        # 根據sn， model 筛选出 (model, cur_fw)
        model_cur_fw = []
        for drive in self.drives:
            model_cur_fw.append((drive.model_number(), drive.firmware_version()))

        fw_manager = FirmwareManager(firmware_config_path)
        upgrade_fw_order = fw_manager.get_firmware_info()

        # 筛选出 升级版本匹配 以及 升级固件存在 的盘
        to_be_upgrade_drive_list = []
        for drive in self.drives:
            for model_current_fw, upgrade_info in upgrade_fw_order.items():
                # 是否匹配fw
                if (drive.model_number(), drive.firmware_version()) != model_current_fw:
                    continue
                if len(fw_list) != 0 and upgrade_info[len(upgrade_info) - 1][0] not in fw_list:
                    # 判断升级最终版本是否在 target_fw_list当中
                    continue
                # 添加目标版本及升级文件至硬盘对象
                if not self._is_upgrade_file_exist(upgrade_info):
                    continue
                #   将升级路径 [（fw，file）]添加至drive
                drive.add_upgrade_step(upgrade_info)
                to_be_upgrade_drive_list.append(drive)

        self.drives = to_be_upgrade_drive_list
        # 待升级列表是否为空
        if len(self.drives) == 0:
            return_info = {"status": ReturnStatus.FAIL, "desc": "no target drives to upgrade"}
            self._output(return_info)
            return
        self._create_process(self.upgrade_progress_file)
        result = {"status": ReturnStatus.SUCCESS, "timeout": len(self.drives) * 300}
        self._output(result)
        self._upgrade_all_match_list()
        sn_list = self.pm.query_sn_by_status(EXECUTING)
        self.pm.patch_set_a_state_to_fail(EXECUTING, sn_list, "not execute upgrade command")
        self._packaging_upgrade_log()
        self.log.info("Complete collect upgrade firmware")

    def get_upgrade_process(self):
        """
        get progress of firmware upgrade
        Returns:
            {"status":"failure", "desc":"sn exec upgrade cmd fail"}
        """
        if not os.path.exists(self.upgrade_progress_file):
            raise DFSException("Progress file is not found! '%s'" % self.upgrade_progress_file)
        pm = ProgressManager(self.upgrade_progress_file)
        proc_result = pm.get_process()
        status = proc_result.get("status")
        info = proc_result.get("info")
        ret = {}
        if status == ReturnStatus.SUCCESS:
            ret = {"status": ReturnStatus.SUCCESS}
            os.remove(self.upgrade_progress_file)
        elif status == ReturnStatus.FAIL:
            ret = {"status": ReturnStatus.FAIL, "desc": info}
            os.remove(self.upgrade_progress_file)
        elif status == ReturnStatus.RUNNING:
            ret = {"status": ReturnStatus.RUNNING, "progress": info}
        self._output(ret)

    def is_exist_lspci(self):
        # 检测raid工具是否存在
        cmd = "whereis -b {}".format("lspci")
        flag, msg, error = CommandUtil.exec_shell(cmd)
        # 以
        if not flag:
            # 假若没有此工具
            raise DFSException("excute whereis -b {} fail".format("lspci"))
        if msg.strip() == "lspci:":
            return False
        return True

    def _is_upgrade_file_exist(self, upgrade_info):
        for step_upgrade_info in upgrade_info:
            # 判断固件版本是否存在
            path = os.path.join(DFSROOT, "upgrade", step_upgrade_info[1])
            if not os.path.exists(path):
                self.log.warning("{} upgrade fireware no found".format(step_upgrade_info[1]))
                return False
        return True

    def _upgrade_drive_msg(self, upgrade_fw_order, target_model_fw):
        """
            校验 up
        Args:
            upgrade_fw_order:
            target_model_fw:

        Returns:

        """
        # 根据sn，model号筛选待升级硬盘信息得到model，cur_fw
        current_model_fw = []
        for model, _ in target_model_fw:
            for drive in self.drives:
                self._get_model_curfw(drive, model, current_model_fw, upgrade_fw_order, target_model_fw)

    def _get_model_curfw(self, drive, model, current_model_fw, upgrade_fw_order, target_model_fw):
        if drive.info["model_number"] == model:
            current_model_fw.append((model, drive.info["firmware_version"]))
            # 判断当前的model 和 fw 是否在 升级序列当中
            if (model, drive.info["firmware_version"]) not in upgrade_fw_order or \
                    not self._is_target_fw_right(upgrade_fw_order, target_model_fw,
                                                 (model, drive.info["firmware_version"])):
                # 从待升级列表中 剔除
                self.drives.remove(drive)
                self.log.warning("Target fireware match failed!")
                return
            upgrade_list = upgrade_fw_order[(model, drive.info["firmware_version"])]
            is_exit = True
            for item_tuple in upgrade_list:
                path = os.path.join(DFSROOT, "upgrade", item_tuple[1])
                if not os.path.exists(path):
                    self.log.warning("{} upgrade fireware no found".format(item_tuple[0]))
                    # 剔除
                    self.drives.remove(drive)
                    is_exit = False
                    break
            if not is_exit:
                return
            # 将升级包 放入drive当中
            drive.update_upgrade_step(upgrade_list)

    def _save_log_file(self, dest_dir):
        '''
        dfs.log internal_progress.txt mcslog.txt messages save to log dir
        Args:
            dest_dir:
        '''
        if os.path.exists(self.dfs_log):
            shutil.copy(self.dfs_log, dest_dir)
        if os.path.exists(self.mcs_log_path):
            shutil.copy(self.mcs_log_path, dest_dir)
        if os.path.exists(self.sys_message_path):
            shutil.copy(self.sys_message_path, dest_dir)
        if os.path.exists(self.upgrade_progress_file):
            shutil.copy(self.upgrade_progress_file, dest_dir)
        if os.path.exists(self.internal_progress_file):
            shutil.copy(self.internal_progress_file, dest_dir)

    def _exec_upgrade(self, match_info_list, drive, find_error=False):
        model_number = drive.model_number()
        sn = drive.serial_number()
        for match_info in match_info_list:
            params = {
                "file_path": match_info.get("FirmwareFile")
            }
            if model_number in SPECIAL_DISK_UPGRADE_PARAMS:
                params.update(SPECIAL_DISK_UPGRADE_PARAMS[model_number])
            before_firmware = drive.firmware_version()
            try:
                drive.upgrade(params)
                drive.update_info()
            except DFSException as error:
                self.log.warning("[Model:%s]Failed to execute the firmware upgrade command, %s-->%s, msg:%s"
                                 % (model_number, before_firmware, match_info.get("TargetVersion"), error.message))
            after_firmware = drive.firmware_version()
            # check firmware
            if after_firmware == match_info.get("TargetVersion"):
                self.log.info("[Model:%s]upgrade successfully, %s-->%s" %
                              (model_number, before_firmware, match_info.get("TargetVersion")))
                self.pm.set_success(sn)
            else:
                if params.get("active") == "PowerOff" and before_firmware in params.get("currentFW"):
                    msg = "[Model:%s]upgrade successfully, %s-->%s, " \
                          "but need power on and off for the firmware to take effect" % \
                          (model_number, before_firmware, match_info.get("TargetVersion"))
                    self.log.warning(msg)
                    self.pm.set_success(sn)
                    continue
                msg = "[Model:%s]upgrade failed, %s-->%s" % \
                      (model_number, before_firmware, match_info.get("TargetVersion"))
                self.log.error(msg)
                self.pm.set_fail(sn, msg)
                find_error = True
                break
        return find_error

    def _target_disk(self):
        # 下发扫盘命令,获取当前环境上所有的硬盘列表
        drives = self.mcs.get_drives_info()
        all_drive_info = "all drives info:\n"
        for drive in drives:
            all_drive_info += str(drive) + "\n"
        self.log.info(all_drive_info)
        # 如果用户输入的SN不为空,则使用SN进行过滤
        if self.filter_sn:
            drives = filter(lambda item: item.get("serial_number") in self.filter_sn, drives)
        # 如果用户输入的MN不为空,则使用MN进行过滤
        if self.filter_model:
            drives = filter(lambda item: item.get("model_number") in self.filter_model, drives)
        # 过滤之后的硬盘列表为最终的需要操作的列表
        for drive in drives:
            # 最终的信息组装在service的实例化中
            self.drives.append(DriveFactoryMap().get(drive))


    def _create_process(self, progress_file):
        """create process"""
        self.pm = ProgressManager(progress_file)
        sn_list = [drive.serial_number() for drive in self.drives]
        self.pm.add_batch(sn_list)

    def _upgrade_all_match_list(self):
        # 升级前 对 环境上所有的工具
        exist_raid_info_list = self._get_all_raid_list()
        try:
            raid_list = CardFactoryMap().get(exist_raid_info_list, True)
        except DFSException as e:
            self.log.error(e.message)
            return
        for raid in raid_list:
            self.log.info("Start to raid status check")
            check_dict = self._read_xml(os.path.join(DFSROOT, "version.xml"))
            if check_dict.get("UpgradeRaidStatuscheck", None).upper() == "TRUE" and not raid.is_normal():
                return
        # drive 和 raid绑定
        for drive in self.drives:
            sn = drive.info["serial_number"]
            for raid in raid_list:
                if raid.is_contain(sn):
                    drive.add(raid)
                    break
        for drive in self.drives:
            # 按drive 升级盘 is_error 测试过程中是否出现 状态检查出错 升级异常 raid卡状态错误
            if not self._upgrade_drive(drive):
                return

    def _upgrade_drive(self, drive):
        for upgrade_msg in drive.upgrade_step:
            upgrade_fw = upgrade_msg[0]
            upgrade_firmware = upgrade_msg[1]
            # 根据map 来给drive 绑定raid卡
            sn = drive.info["serial_number"]
            # 升级前状态检查
            drive_letter = drive.info["drive_letter"]
            is_normal_before_upgrade = drive.is_normal_before_upgrade(drive_letter)
            if not is_normal_before_upgrade:
                self.pm.set_fail(drive.info["serial_number"],
                                 "drive status is not normal before upgrade")
                return False
            upgrade_params = {
                "file_path": os.path.join(DFSROOT, "upgrade", upgrade_firmware)
            }
            try:
                drive.upgrade(upgrade_params)
            except Exception as e:
                self.pm.set_fail(drive.serial_number(), e.message.replace("\n", ""))
                return False
            # 升级后状态检查
            expect_fw_after_upgrade = upgrade_fw
            is_normal_after_upgrade = drive.is_normal_after_upgrade(expect_fw_after_upgrade)
            if not is_normal_after_upgrade:
                self.pm.set_fail(sn, "dirve is abnormal after upgrade")
                return False
            self.pm.set_success(sn)
            return True

    def _is_target_fw_right(self, upgrade_fw_order, target_model_fw, param):
        """

        Args:
            upgrade_fw_order: (model:currentFw):[(fw1,file1),(fw2,file2)]
            target_model_fw: [(model1,fw1),(model2,fw2)]
            param:(model,currentFw)
        Returns:
            bool
        """
        # 找到当前盘的升级列表

        current_upgrade_fw_order = upgrade_fw_order[param]
        current_target_fw = []
        length_list = len(target_model_fw)
        for i in range(length_list):
            if target_model_fw[i][0] == param[0]:
                current_target_fw.append(target_model_fw[i][1])
        if current_target_fw[0] == current_upgrade_fw_order[len(current_upgrade_fw_order) - 1][0]:
            return True
        return False

    def _is_exist_config_file(self, firmware_config_path):
        if not os.path.exists(firmware_config_path):
            return False
        return True

    def _packaging_upgrade_log(self):
        collect_time = int(time.time())

        pack_upgrade_log = os.path.join(".", "upgrade_log_{}".format(collect_time))
        os.mkdir(pack_upgrade_log)
        self._save_log_file(pack_upgrade_log)

        shutil.make_archive(base_name=pack_upgrade_log, format='tar', base_dir=pack_upgrade_log)
        shutil.rmtree(pack_upgrade_log)

    def _get_all_raid_list(self):
        """
            获取已经存在的工具 信息
        Returns:
            [{'raid_tool': 'storcli64'}]
        """
        cmd = "lspci -x"
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            self.log.warning("Lspci -x 执行失败")
            return []
        return self._parse_get_lspcix(res[1])

    def _parse_get_lspcix(self, param):
        """
            通过 解析四元组信息 获取 已经存在的工具
        Args:
            param:

        Returns:

        """
        lines = param.splitlines()
        output = []
        result = []
        for line in lines:
            if re.search(r"^00:\s{1}", line):
                msg = line.split(":")[-1].strip()
                line_list = msg.split(" ")
                result.append(line_list[1] + line_list[0])
                result.append(line_list[3] + line_list[2])
            if re.search(r"^20:\s{1}", line):
                msg = line.split(":")[-1].strip()
                line_list = msg.split(" ")
                result.append(line_list[-3] + line_list[-4])
                result.append(line_list[-1] + line_list[-2])
                if len(result) == 4:
                    output.append(result)
                    result = []
                continue
            if not re.search(r"[0-9]{2}:\s{1}", line):
                continue
        return self._upate_raid_info(output)

    def _upate_raid_info(self, groups):
        raids = []
        # 遍历每个四元祖信息 根据四元祖信息获取RAID卡相关信息
        for info in groups:
            self._get_raid_card_info(info, raids)
        return raids

    def _get_raid_card_info(self, info, raids):
        for raid_name, raid_list in RaidMap.raid_parameter_info.items():
            if info in raid_list:
                self.log.info("Current RAID name is {}".format(raid_name))
                # 2308IR/3008IR/3008IT/8060已经EOS,且可以视为直通处理,无需判断RAID卡兼容性
                if raid_name in ["LSI_3008IR", "LSI_2308", "LSI_3008IT", "8060"]:
                    continue
                # 获取raid信息
                raid = RaidMap.__dict__[raid_name]
                # 复制Raid卡信息
                raids.append(copy.deepcopy(raid))

    def _is_passed_resource_check(self, drive):
        """
            资源检查 CPU资源 内存利用率 每s输入输出利用率
        Returns:
            True or False
        """
        # 循环10次:
        self.log.info("Start to check resource")
        idle_cpu_cpu_use_list = []
        rate_mem_use_list = []
        time_r_await_list = []
        time_w_await_list = []
        rate_util_list = []
        # 资源检查策略为收10次 去掉最大最小值 取平均值
        for _ in range(10):
            try:
                idle_cpu_res = ResourceManager.idle_cpu_rate()
                if not idle_cpu_res:
                    self.log.warning("Get rate of cpu idle failed")
                    return False
                idle_cpu_cpu_use_list.append(float(idle_cpu_res))

                mem_use = ResourceManager.get_rate_mem_use()
                if not mem_use:
                    self.log.warning("Get rate of mem-use failed")
                    return False
                rate_mem_use_list.append(float(mem_use))
                ret = ResourceManager.get_rate_io_per_second(drive)
                if not ret:
                    self.log.warning("Get io per second fail")
                    return False
                r_await, w_await, time_util = ret
                time_r_await_list.append(r_await)
                time_w_await_list.append(w_await)
                rate_util_list.append(time_util)
            except DFSException as e:
                self.log.warning("Check resource failed")
                self.log.info(e.message)
                return False
        time.sleep(1)
        # 判断是否超出标准 去最大最小 取平均
        if self._calculate_the_average(idle_cpu_cpu_use_list) < 40:
            self.log.warning("Idle cpu is not enough, idle is {}%".format(str(idle_cpu_cpu_use_list)))
            return False

        if self._calculate_the_average(rate_mem_use_list) > 0.6:
            self.log.warning("Rate mem use overload, mem use is {}".format(str(rate_mem_use_list)))
            return False

        if self._calculate_the_average(time_r_await_list) > 20:
            self.log.warning("R_await is overload, r_await is {}".format(str(time_r_await_list)))
            return False

        if self._calculate_the_average(time_w_await_list) > 60:
            self.log.warning("W_await is overload, w_await is {}".format(str(time_w_await_list)))
            return False

        if self._calculate_the_average(rate_util_list) > 60:
            self.log.warning("Util is overload, util is {}%".format(str(rate_util_list)))
            return False
        return True

    def _calculate_the_average(self, calculate_list):
        calculate_list.sort()
        calculate_list = calculate_list[1:-1]
        return sum(calculate_list) / len(calculate_list)