#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 管理下发命令进程

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2022/08/04 18:00 created

"""
import platform
import subprocess
from threading import Timer
from argparse import ArgumentParser, RawTextHelpFormatter

from lib.utils.log import SingleLog

G_PY_VERSION = platform.python_version().split(".")[0]
LOGGER = SingleLog()


def cmd_line_parse():
    usage = "python3 dfs.py"
    parser = ArgumentParser(usage, formatter_class=RawTextHelpFormatter)
    parser.add_argument("-v", "--version", dest="version", action="store_true",
                        help="Show program's version number and exit")
    parser.add_argument("-o", dest="operation", help="specify the target operation\n"
                                                     "-o disks :scans all disk information on the server.\n"
                                                     "-o smart :collect SMART log about disks on the server.\n"
                                                     "-o internalLog :collect internal log about disks on the server.\n"
                                                     "-o internalLogProg :get the progress of collecting internal log\n"
                                                     "-o upgrade :upgrading the firmware of disks on the server\n"
                                                     "-o upgradeProg :get the progress of upgrading the firmware")
    parser.add_argument("-s", "--sn", dest="sn", help="specifying drives serial number.Ex:sn1,sn2..snk")
    parser.add_argument("-m", "--model", dest="model",
                        help="specifying drives model number.Ex:model1,model2...modelk")
    # refactor upgrade params
    parser.add_argument("-fw", "--firmware", dest="fw",
                        help="specifying drives firmware.Ex:firmware,firmware2...firmwarek")
    parser.add_argument("-d", dest="directory", help="specify the log where to storage")
    parser.add_argument("-t", dest="tid", help="specify which taskid to query execute progress")
    return parser


class CommandUtil(object):

    @classmethod
    def exec_cmd(cls, commands, work_dir=None, env_variable=None):
        """
        执行命令并返回subprocess.Popen对象
        :param commands: [string|list] 命令行参数
        :param work_dir: string 工作路径,默认为当前路径
        :param env_variable: string 环境变量，默认为空
        :return: subprocess.Popen
        """
        commands = ["/bin/sh", "-c"] + [commands]
        process = subprocess.Popen(args=commands, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE, shell=False, cwd=work_dir, env=env_variable)
        return process

    @classmethod
    def destroy(cls, process):
        """
        销毁进程
        :param process: object 待销毁subprocess.Popen 对象
        :return: None
        """
        if process and process.poll() is None:
            process.kill()

    @classmethod
    def get_result(cls, process, timeout, command):
        def call_back():
            LOGGER.error("command timeout, %s" % command)

        stdout = b''
        stderr = b''
        if G_PY_VERSION == "3":
            try:
                stdout, stderr = process.communicate(timeout=timeout)
            except subprocess.TimeoutExpired:
                LOGGER.error("command timeout, %s" % command)
            finally:
                cls.destroy(process)
        else:
            my_timer = Timer(timeout, call_back)
            my_timer.start()
            try:
                stdout, stderr = process.communicate()
            finally:
                if my_timer:
                    my_timer.cancel()
                cls.destroy(process)
        stdout_decode = stdout.decode("utf-8").strip()
        stderr_decode = stderr.decode("utf-8").strip()
        if process.poll() == 0:
            return True, stdout_decode, stderr_decode
        else:
            LOGGER.error("stdout:{}\nstderr:{}\ncommand:[{}]".format(stdout_decode, stderr_decode, command))
            return False, stdout_decode, stderr_decode

    @classmethod
    def exec_shell(cls, commands, timeout=300, work_dir=None, env_variable=None):
        return cls.get_result(cls.exec_cmd(commands, work_dir=work_dir, env_variable=env_variable), timeout, commands)
