#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 判断环境和获取相应工具

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2022/9/12 18:00 created

"""
import csv
import os
import platform
import stat

# Read/Write Mode
FLAGS = os.O_RDWR | os.O_CREAT
# Read/Write permission
MODES = stat.S_IWUSR | stat.S_IRUSR


class CsvUtil(object):
    """
    操作csv/Excel文件工具类
    """

    @classmethod
    def read_csv(cls, file_path, **kwargs):
        """
        读取csv文件，返回list
        :param file_path: 文件绝对路径
        :return: list
        """
        py_version = platform.python_version().split(".")[0]
        result = []
        with open(file_path, 'r') as csv_file:
            reader = csv.DictReader(csv_file, **kwargs)
            for line in reader:
                if py_version == "3":
                    result.append(line)
                else:
                    result.append({key.decode("utf-8"): val.decode("utf-8") for key, val in line.items()})
        return result

    @classmethod
    def create_csv(cls, file_path, field_names, dialect="excel"):
        """
        创建csv文件
        :param file_path: 文件名及路径
        :param field_names: 文件filed
        :param dialect: excel风格
        :return:
        """
        with os.fdopen(os.open(file_path, FLAGS, MODES), "w") as csv_file:
            dict_writer = csv.DictWriter(csv_file, field_names, dialect=dialect)
            dict_writer.writeheader()

    @classmethod
    def write_row(cls, file_path, field_names, info, dialect="excel"):
        """
        单行写入数据
        :param file_path: 文件名及路径
        :param field_names: 文件filed
        :param info: 待写入信息
        :param dialect: excel风格
        :return:
        """
        with os.fdopen(os.open(file_path, FLAGS, MODES), "a") as csv_file:
            dict_writer = csv.DictWriter(csv_file, field_names, dialect=dialect)
            dict_writer.writerow(info)

    @classmethod
    def write_rows(cls, file_path, field_names, info, dialect="excel"):
        """
        多行写入数据
        :param file_path: 文件名及路径
        :param field_names: 文件filed
        :param info: 待写入信息
        :param dialect: excel风格
        :return:
        """
        with os.fdopen(os.open(file_path, FLAGS, MODES), "a") as csv_file:
            dict_writer = csv.DictWriter(csv_file, field_names,
                                         dialect=dialect)
            dict_writer.writerows(info)
