#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: FirmwareManager固件配置管理类

版权信息: 华为技术有限公司，版权所有(C) 2023-2025

修改记录: 2024/2/29 16:00 created

"""
import os
import re

import fcntl

from config.constants import ReturnStatus
from lib.exception.dfs_exception import DFSException
from lib.utils.command import CommandUtil
from lib.utils.csv_util import CsvUtil, FLAGS, MODES
from lib.utils.validate import ValidateUtil

SUCCEED = ReturnStatus.SUCCESS
FAILED = ReturnStatus.FAIL
EXECUTING = ReturnStatus.RUNNING


class FirmwareManager(object):
    def __init__(self, tool_path):
        self.config_path = tool_path

    def get_firmware_info(self):
        """

        Args:
            Model	          Vendor VersionList	        TargetVersion	   FirmwareFileName
            MG06ACA800E	      TOSHIBA   080F	                 080A	           MG06_SATA_080A.ftd
            SDLL1CLR020T5CHS  SEAGATE   HW09	                 HW10	           dadasd.bin
            MG06ACA800E	      MEGATE    080A                   0807	           MG06_SATA_0807.ftd
            HWE62ST3480L003N  HUAWEI  10A7;10A6;10A8	     TARGET_FW	       MG06_SATA_0807.ftd
        Returns:
            type = dict
            key = model:version value = [(TargetVersion1, FirmwareFileName1),(TargetVersion2, FirmwareFileName2),...]
            ('MG06ACA800E', '080F') = {list: 2} [('080A', 'MG06_SATA_080A.ftd'), ('0807', 'MG06_SATA_0807.ftd')]
            ('SDLL1CLR020T5CHS', 'HW09') = {list: 1} [('HW10', 'dadasd.bin')]
            ('MG06ACA800E', '080A') = {list: 1} [('0807', 'MG06_SATA_0807.ftd')]
            ('HWE62ST3480L003N', '10A7') = {list: 1} [('TA-RGET_FW', 'MG06_SATA_0807.ftd')]
            ('HWE62ST3480L003N', '10A6') = {list: 1} [('TARGET_FW', 'MG06_SATA_0807.ftd')]
            ('HWE62ST3480L003N', '10A8') = {list: 1} [('TARGET_FW', 'MG06_SATA_0807.ftd')]
        """
        #
        # 读取到文件
        read_content = CsvUtil.read_csv(self.config_path)
        # 组装数据结构
        result_tree = {}
        for line in read_content:
            # 当前的(model, curfw)
            for cur_fw in line["VersionList"].split(";"):
                # 组装树 组装头
                result_tree[(line["Model"], cur_fw)] = [(line["TargetVersion"], line["FirmwareFileName"])]
                # 组装身 guard 头数据
                guard_model = line["Model"]
                guard_fw = cur_fw
                body_lines = [body_line for body_line in read_content if line["Model"] == guard_model]
                # 身 上 最新fw
                last_fw = line["TargetVersion"]

                self.find_next_body(lines=body_lines, guard=guard_model, start=guard_fw,
                                    result_tree=result_tree, last_fw=last_fw)

        return result_tree

    def find_next_body(self, lines, guard, start, result_tree, last_fw):
        if len(result_tree) == 0:
            return {}
        for line in lines:
            for body_cur_fw in line["VersionList"].split(";"):
                if body_cur_fw == last_fw:
                    # 找到匹配的 组装 并移除
                    result_tree[(guard, start)].append((line["TargetVersion"], line["FirmwareFileName"]))
                    lines.remove(line)
                    return self.find_next_body(lines, guard, start, result_tree, last_fw)
        return result_tree


class ResourceManager(object):

    @staticmethod
    def _get_drive_letter_by_sg_drive_letter(self, drive_letter):
        def _parser_ret(raw_output, drive_letter):
            ret = {"status_code": "0", "output": ""}
            res = re.search(r"(/dev/sd\w+)\s+(/dev/{})".format(drive_letter), raw_output)
            if res:
                ret["output"] = res.group(1).split("/")[2]
                return ret
            # 匹配不到盘符 尝试查找 Logical Volume
            res = re.search(r"(Logical Volume)\s+(.+)\s+(/dev/sd\w+)", raw_output)
            if res:
                ret["output"] = res.group(3).split("/")[2]
                return ret
            # 匹配不到盘符 尝试查找 RAID
            res = re.search(r"(RAID\w+)\s+\w+\s+(/dev/sd\w+)", raw_output)
            if res:
                ret["output"] = res.group(2).split("/")[2]
                return ret
            # 匹配失败
            return {"status_code": "1", "output": "{} match failed".format(drive_letter)}

        cmd = "lsscsi -g"
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1], drive_letter)
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", "")

    @classmethod
    def idle_cpu_rate(cls):
        def _parser_ret(raw_output):
            ret = {"status_code": "0", "output": ""}
            for line in raw_output.split("\n"):
                if re.search(r"\s*(\d+\.\d+)\s*(\d+\.\d+)\s*(\d+\.\d+)\s*(\d+\.\d+)\s*(\d+\.\d+)\s*(\d+\.\d+)",
                             line):
                    ret["output"] = line.split(" ")[len(line.split(" ")) - 1]
                    return ret
            return ret
        cmd = "iostat -x"
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", "")

    @classmethod
    def get_rate_mem_use(cls):
        def _parser_ret(raw_output):
            ret = {"status_code": "0", "output": ""}
            for line in raw_output.split("\n"):
                if re.search(r"^Mem:\s+", line):
                    total_mem = line.split()[1]
                    use_mem = line.split()[2]
                    ret["output"] = float(use_mem) / float(total_mem)
                    return ret
            return ret
        cmd = "free"
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1])
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", "")

    @classmethod
    def get_rate_io_per_second(cls, drive):
        def _parser_ret(raw_output, drive_letter):
            ret = {"status_code": "0", "output": ""}
            # 盘符是否为sg盘符 或者为 /bsg/hiraid0盘符
            if re.match(r"sg\d+", drive_letter) or re.match(r"bsg/hiraid\d+", drive_letter):
                # 下发lsscsi -g命令 找到正确的物理盘符
                drive_letter = ResourceManager._get_drive_letter_by_sg_drive_letter(cls, drive_letter)
            # 找到字段
            index_r_await = -1
            index_w_await = -1
            index_util = -1
            for line in raw_output.split("\n"):
                if re.search(r"^Device", line):
                    index_r_await = line.split().index("r_await")
                    index_w_await = line.split().index("w_await")
                    index_util = line.split().index("%util")
                    break
            if index_r_await == -1 or index_w_await == -1 or index_util == -1:
                return {"status_code": "1", "output": ""}
            for line in raw_output.split("\n"):
                if re.search(r"^{}\s+".format(drive_letter), line):
                    r_await = line.split()[index_r_await]
                    w_await = line.split()[index_w_await]
                    rate_util = line.split()[index_util]
                    ret["output"] = float(r_await), float(w_await), float(rate_util)
                    return ret
            for line in raw_output.split("\n"):
                if re.search(r"^{}\s+".format(drive_letter), line):
                    r_await = line.split()[10]
                    w_await = line.split()[11]
                    rate_util = line.split()[len(line.split()) - 1]
                    ret["output"] = "{} {} {}".format(r_await, w_await, rate_util)
                    return ret
            return ret
        # 盘符去除/dev/
        drive_letter = drive.drive_letter()[len("/dev/"):]
        cmd = "iostat -x"
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret_info = _parser_ret(res[1], drive_letter)
        ValidateUtil.check_status_code(ret_info)
        return ret_info.get("output", "")


class ProgressManager(object):
    """
        以sn的维度管理对硬盘操作的进度
    """

    def __init__(self, file_path):
        self.file_path = file_path

    def add_batch(self, sn_list):
        """
        Args:
            sn_list:需要跟踪sn的记录列表
        Returns:
            boolean
        """
        progress = {}
        for sn in sn_list:
            progress[sn] = [sn, EXECUTING, ""]
        self._flush_to_file(progress)

    def set_success(self, sn):
        """
        Args:
            sn:硬盘sn号
        Returns:
            boolean
        """
        cur_progress = {}
        with open(self.file_path, "r") as up_file:
            for line in up_file.readlines():
                cur_sn, cur_stat, cur_reason = line.split(";")
                cur_progress[cur_sn] = [cur_sn, cur_stat, cur_reason.strip()]
        cur_progress[sn] = [sn, SUCCEED, ""]
        self._flush_to_file(cur_progress)

    def set_fail(self, sn, reason):
        """
        Args:
            sn: 硬盘sn号
            reason: 失败原因
        Returns:
            boolean

        """
        # 进度文件通过分号来区分列数，原因中包含需要替换
        reason = reason.replace(";", ":")
        cur_progress = {}
        with open(self.file_path, "r") as up_file:
            for line in up_file.readlines():
                cur_sn, cur_stat, cur_reason = line.split(";")
                cur_progress[cur_sn] = [cur_sn, cur_stat, cur_reason.strip()]
        cur_progress[sn] = [sn, FAILED, reason]
        self._flush_to_file(cur_progress)

    def get_process(self):
        """
        Returns:
            {"status":"failed","info":"sn exec upgrade cmd fail"}
        """
        with open(self.file_path, "r") as upgrade_file:
            fcntl.flock(upgrade_file.fileno(), fcntl.LOCK_EX)
            lines = upgrade_file.readlines()
            content = "".join(lines)
        if EXECUTING in content:
            ug_cnt = float(content.count(EXECUTING))
            total = float(content.count("\n"))
            percent = 100 - int((ug_cnt / total) * 100)
            return {"status": EXECUTING, "info": str(percent)}
        if FAILED in content:
            all_reason = ""
            for line in lines:
                status = line.split(";")[1]
                if status == FAILED:
                    all_reason += line
            return {"status": FAILED, "info": all_reason}
        return {"status": SUCCEED, "info": ""}

    def query_sn_by_status(self, status):
        """
        Args:
            status:

        Returns:
            [sn1,sn2,sn3...]
        """
        with open(self.file_path, "r") as upgrade_file:
            fcntl.flock(upgrade_file.fileno(), fcntl.LOCK_EX)
            lines = upgrade_file.readlines()
            content = "".join(lines)
        result = []
        if status in content:
            for line in lines:
                msg = line.split(";")
                if msg[1] == status:
                    result.append(msg[0])
            return result
        return result

    def patch_set_a_state_to_fail(self, status, sn_list, fail_reason):
        """
        Args:
            status: 指定状态
            sn_list: sn组成的列表 type:list
            fail_reason: 错误原因
        Returns:
        """
        cur_progress = {}
        with open(self.file_path, "r") as upgrade_file:
            fcntl.flock(upgrade_file.fileno(), fcntl.LOCK_EX)
            lines = upgrade_file.readlines()
        # 筛选符合的line
        for line in lines:
            cur_sn, cur_status, cur_reason = line.split(";")
            cur_progress[cur_sn] = [cur_sn.strip(), cur_status.strip(), cur_reason.strip()]
            if cur_sn in sn_list and cur_status == status:
                cur_progress[cur_sn][1] = FAILED
                cur_progress[cur_sn][2] = fail_reason
        self._flush_to_file(progress=cur_progress)

    def _flush_to_file(self, progress):
        """
        Args:
            progress: {sn1:[sn1,stat1,reason1],sn2:[sn2,stat2,reason2]}
        Returns:
            None
        """
        if os.path.exists(self.file_path):
            os.remove(self.file_path)
        fd = os.open(self.file_path, os.O_WRONLY | os.O_CREAT, MODES)
        with os.fdopen(fd, 'w') as up_file:
            fcntl.flock(up_file.fileno(), fcntl.LOCK_EX)
            for _, s_p in progress.items():
                up_file.write("{};{};{}".format(s_p[0], s_p[1], s_p[2].strip() + '\n'))
