/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import com.obs.services.exception.ObsException;
import com.obs.services.model.KeyAndVersion;
import com.obs.services.model.ListObjectsRequest;
import com.obs.services.model.ObjectListing;
import com.obs.services.model.ObsObject;
import com.obs.services.model.fs.GetAttributeRequest;
import com.obs.services.model.fs.NewFolderRequest;
import com.obs.services.model.fs.ObsFSAttribute;
import com.obs.services.model.fs.RenameRequest;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.fs.obs.FileConflictException;
import org.apache.hadoop.fs.obs.OBSCommonUtils;
import org.apache.hadoop.fs.obs.OBSFileStatus;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.obs.OBSIOException;
import org.apache.hadoop.security.AccessControlException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class OBSPosixBucketUtils {
    private static final Logger LOG = LoggerFactory.getLogger(OBSPosixBucketUtils.class);

    private OBSPosixBucketUtils() {
    }

    static boolean fsIsFolder(ObsFSAttribute attr) {
        int ifDir = 16384;
        int mode = attr.getMode();
        if (mode < 0) {
            return false;
        }
        return (mode & 0x4000) != 0;
    }

    static boolean renameBasedOnPosix(OBSFileSystem owner, Path src, Path dst) throws IOException {
        String dstKey;
        String srcKey;
        Path dstPath;
        block8: {
            dstPath = dst;
            srcKey = OBSCommonUtils.pathToKey(owner, src);
            dstKey = OBSCommonUtils.pathToKey(owner, dstPath);
            if (srcKey.isEmpty()) {
                LOG.error("rename: src [{}] is root directory", (Object)src);
                return false;
            }
            try {
                FileStatus dstStatus = OBSCommonUtils.innerGetFileStatusWithRetry(owner, dstPath);
                if (dstStatus.isDirectory()) {
                    String newDstString = OBSCommonUtils.maybeAddTrailingSlash(dstPath.toString());
                    String filename = srcKey.substring(OBSCommonUtils.pathToKey(owner, src.getParent()).length() + 1);
                    dstPath = new Path(newDstString + filename);
                    dstKey = OBSCommonUtils.pathToKey(owner, dstPath);
                    LOG.debug("rename: dest is an existing directory and will be changed to [{}]", (Object)dstPath);
                    if (owner.exists(dstPath)) {
                        LOG.error("rename: failed to rename " + src + " to " + dstPath + " because destination exists");
                        return false;
                    }
                    break block8;
                }
                if (srcKey.equals(dstKey)) {
                    LOG.warn("rename: src and dest refer to the same file or directory: {}", (Object)dstPath);
                    return true;
                }
                LOG.error("rename: failed to rename " + src + " to " + dstPath + " because destination exists");
                return false;
            }
            catch (FileNotFoundException e) {
                LOG.debug("rename: dest [{}] does not exist", (Object)dstPath);
            }
            catch (FileConflictException e) {
                throw new ParentNotDirectoryException(e.getMessage());
            }
        }
        if (dstKey.startsWith(srcKey) && (dstKey.equals(srcKey) || dstKey.charAt(srcKey.length()) == '/')) {
            LOG.error("rename: dest [{}] cannot be a descendant of src [{}]", (Object)dstPath, (Object)src);
            return false;
        }
        return OBSPosixBucketUtils.innerFsRenameWithRetry(owner, src, dstPath, srcKey, dstKey);
    }

    static boolean innerFsRenameWithRetry(OBSFileSystem owner, Path src, Path dst, String srcKey, String dstKey) throws IOException {
        String newSrcKey = srcKey;
        String newDstKey = dstKey;
        int retryTime = 0;
        long startTime = System.currentTimeMillis();
        while (true) {
            boolean isRegularDirPath = newSrcKey.endsWith("/") && newDstKey.endsWith("/");
            try {
                LOG.debug("rename: {}-st rename from [{}] to [{}] ...", new Object[]{retryTime, newSrcKey, newDstKey});
                OBSPosixBucketUtils.innerFsRenameFile(owner, newSrcKey, newDstKey);
                return true;
            }
            catch (FileNotFoundException e) {
                if (owner.exists(dst)) {
                    LOG.debug("rename: successfully {}-st rename src [{}] to dest [{}] with SDK retry", new Object[]{retryTime, src, dst, e});
                    return true;
                }
                LOG.error("rename: failed {}-st rename src [{}] to dest [{}]", new Object[]{retryTime, src, dst, e});
                return false;
            }
            catch (IOException e) {
                if (e instanceof AccessControlException && isRegularDirPath) {
                    throw e;
                }
                try {
                    FileStatus srcFileStatus = OBSCommonUtils.innerGetFileStatusWithRetry(owner, src);
                    if (srcFileStatus.isDirectory()) {
                        newSrcKey = OBSCommonUtils.maybeAddTrailingSlash(newSrcKey);
                        newDstKey = OBSCommonUtils.maybeAddTrailingSlash(newDstKey);
                    } else if (e instanceof AccessControlException) {
                        throw e;
                    }
                }
                catch (FileConflictException e1) {
                    throw new AccessControlException((Throwable)e);
                }
                IOException lastException = e;
                LOG.warn("rename: failed {}-st rename src [{}] to dest [{}]", new Object[]{retryTime, src, dst, e});
                if (owner.exists(dst) && owner.exists(src)) {
                    LOG.warn("rename: failed {}-st rename src [{}] to dest [{}] with SDK retry", new Object[]{retryTime, src, dst, e});
                    return false;
                }
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                try {
                    Thread.sleep(delayMs);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
                if (System.currentTimeMillis() - startTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                LOG.error("rename: failed {}-st rename src [{}] to dest [{}]", new Object[]{retryTime, src, dst, lastException});
                throw lastException;
            }
            break;
        }
    }

    static void innerFsRenameFile(OBSFileSystem owner, String srcKey, String dstKey) throws IOException {
        LOG.debug("RenameFile path {} to {}", (Object)srcKey, (Object)dstKey);
        try {
            RenameRequest renameObjectRequest = new RenameRequest();
            renameObjectRequest.setBucketName(owner.getBucket());
            renameObjectRequest.setObjectKey(srcKey);
            renameObjectRequest.setNewObjectKey(dstKey);
            owner.getObsClient().renameFile(renameObjectRequest);
            owner.getSchemeStatistics().incrementWriteOps(1);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("renameFile(" + srcKey + ", " + dstKey + ")", srcKey, e);
        }
    }

    static boolean fsRenameToNewObject(OBSFileSystem owner, String srcKey, String dstKey) throws IOException {
        String newSrcKey = srcKey;
        String newdstKey = dstKey;
        newSrcKey = OBSCommonUtils.maybeDeleteBeginningSlash(newSrcKey);
        newdstKey = OBSCommonUtils.maybeDeleteBeginningSlash(newdstKey);
        return OBSPosixBucketUtils.renameBasedOnPosix(owner, OBSCommonUtils.keyToPath(newSrcKey), OBSCommonUtils.keyToPath(newdstKey));
    }

    private static int fsRemoveFile(OBSFileSystem owner, String sonObjectKey, List<KeyAndVersion> files) throws IOException {
        files.add(new KeyAndVersion(sonObjectKey));
        if (files.size() == owner.getMaxEntriesToDelete()) {
            OBSCommonUtils.removeKeys(owner, files, true, false);
            return owner.getMaxEntriesToDelete();
        }
        return 0;
    }

    static boolean fsDelete(OBSFileSystem owner, FileStatus status, boolean recursive) throws IOException, ObsException {
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        Path f = status.getPath();
        String key = OBSCommonUtils.pathToKey(owner, f);
        if (!status.isDirectory()) {
            LOG.debug("delete: Path is a file");
            OBSPosixBucketUtils.trashObjectIfNeed(owner, key);
        } else {
            LOG.debug("delete: Path is a directory: {} - recursive {}", (Object)f, (Object)recursive);
            key = OBSCommonUtils.maybeAddTrailingSlash(key);
            boolean isEmptyDir = OBSCommonUtils.isFolderEmpty(owner, key);
            if (key.equals("")) {
                return OBSCommonUtils.rejectRootDirectoryDelete(owner.getBucket(), isEmptyDir, recursive);
            }
            if (!recursive && !isEmptyDir) {
                LOG.warn("delete: Path is not empty: {} - recursive {}", (Object)f, (Object)recursive);
                throw new PathIsNotEmptyDirectoryException(f.toString());
            }
            if (isEmptyDir) {
                LOG.debug("delete: Deleting fake empty directory {} - recursive {}", (Object)f, (Object)recursive);
                try {
                    OBSCommonUtils.deleteObject(owner, key);
                }
                catch (FileConflictException e) {
                    LOG.warn("delete emtryDir[{}] has conflict exception, will retry.", (Object)key, (Object)e);
                    OBSPosixBucketUtils.trashFolderIfNeed(owner, key);
                }
            } else {
                LOG.debug("delete: Deleting objects for directory prefix {} to delete - recursive {}", (Object)f, (Object)recursive);
                OBSPosixBucketUtils.trashFolderIfNeed(owner, key);
            }
        }
        long endTime = System.currentTimeMillis();
        LOG.debug("delete Path:{} thread:{}, timeUsedInMilliSec:{}", new Object[]{f, threadId, endTime - startTime});
        return true;
    }

    private static void trashObjectIfNeed(OBSFileSystem owner, String key) throws ObsException, IOException {
        String destKeyWithNoSuffix;
        if (!OBSPosixBucketUtils.needToTrash(owner, key)) {
            OBSCommonUtils.deleteObject(owner, key);
            return;
        }
        OBSPosixBucketUtils.mkTrash(owner, key);
        String destKey = destKeyWithNoSuffix = owner.getTrashDir() + key;
        SimpleDateFormat df = new SimpleDateFormat("-yyyyMMddHHmmssSSS");
        if (owner.exists(new Path(destKey))) {
            destKey = destKeyWithNoSuffix + df.format(new Date());
        }
        long startTime = System.currentTimeMillis();
        int retryTime = 0;
        while (!OBSPosixBucketUtils.fsRenameToNewObject(owner, key, destKey)) {
            LOG.debug("Move file [{}] to [{}] failed, retryTime[{}].", new Object[]{key, destKey, retryTime});
            long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
            if (System.currentTimeMillis() - startTime + delayMs > OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
                LOG.error("Failed rename file [{}] to [{}] after retryTime[{}].", new Object[]{key, destKey, retryTime});
                throw new IOException("Failed to rename " + key + " to " + destKey);
            }
            try {
                Thread.sleep(delayMs);
            }
            catch (InterruptedException ie) {
                LOG.error("Failed rename file [{}] to [{}] after retryTime[{}].", new Object[]{key, destKey, retryTime});
                throw new IOException("Failed to rename " + key + " to " + destKey);
            }
            destKey = destKeyWithNoSuffix + df.format(new Date());
            ++retryTime;
        }
        LOG.debug("Moved file : '" + key + "' to trash at: " + destKey);
    }

    private static void trashFolderIfNeed(OBSFileSystem owner, String key) throws ObsException, IOException {
        String destKeyWithNoSuffix;
        if (!OBSPosixBucketUtils.needToTrash(owner, key)) {
            OBSPosixBucketUtils.fsRecursivelyDeleteDirWithRetry(owner, key, true);
            return;
        }
        OBSPosixBucketUtils.mkTrash(owner, key);
        StringBuilder sb = new StringBuilder(owner.getTrashDir());
        SimpleDateFormat df = new SimpleDateFormat("-yyyyMMddHHmmssSSS");
        int endIndex = key.endsWith("/") ? key.length() - 1 : key.length();
        sb.append(key, 0, endIndex);
        String destKey = destKeyWithNoSuffix = sb.toString();
        if (owner.exists(new Path(sb.toString()))) {
            destKey = destKeyWithNoSuffix + df.format(new Date());
        }
        String srcKey = OBSCommonUtils.maybeAddTrailingSlash(key);
        String dstKey = OBSCommonUtils.maybeAddTrailingSlash(destKey);
        long startTime = System.currentTimeMillis();
        int retryTime = 0;
        while (!OBSPosixBucketUtils.fsRenameToNewObject(owner, srcKey, dstKey)) {
            LOG.debug("Move folder [{}] to [{}] failed, retryTime[{}].", new Object[]{key, destKey, retryTime});
            long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
            if (System.currentTimeMillis() - startTime + delayMs > OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
                LOG.error("Failed rename folder [{}] to [{}] after retryTime[{}].", new Object[]{key, destKey, retryTime});
                throw new IOException("Failed to rename " + key + " to " + destKey);
            }
            try {
                Thread.sleep(delayMs);
            }
            catch (InterruptedException ie) {
                LOG.error("Failed rename folder [{}] to [{}] after retryTime[{}].", new Object[]{key, destKey, retryTime});
                throw new IOException("Failed to rename " + key + " to " + destKey);
            }
            destKey = destKeyWithNoSuffix + df.format(new Date());
            dstKey = OBSCommonUtils.maybeAddTrailingSlash(destKey);
            ++retryTime;
        }
        LOG.debug("Moved folder : '" + key + "' to trash at: " + destKey);
    }

    static void fsRecursivelyDeleteDirWithRetry(OBSFileSystem owner, String key, boolean deleteParent) throws IOException {
        long startTime = System.currentTimeMillis();
        int retryTime = 0;
        while (System.currentTimeMillis() - startTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
            try {
                long delNum = OBSPosixBucketUtils.fsRecursivelyDeleteDir(owner, key, deleteParent);
                LOG.debug("Recursively delete {} files/dirs when deleting {}", (Object)delNum, (Object)key);
                return;
            }
            catch (FileConflictException e) {
                LOG.warn("Recursively delete [{}] has conflict exception, retryTime[{}].", (Object)key, (Object)e);
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                try {
                    Thread.sleep(delayMs);
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        OBSPosixBucketUtils.fsRecursivelyDeleteDir(owner, key, deleteParent);
    }

    static long fsRecursivelyDeleteDir(OBSFileSystem owner, String parentKey, boolean deleteParent) throws IOException {
        long delNum = 0L;
        ArrayList<KeyAndVersion> subdirList = new ArrayList<KeyAndVersion>(owner.getMaxEntriesToDelete());
        ArrayList<KeyAndVersion> fileList = new ArrayList<KeyAndVersion>(owner.getMaxEntriesToDelete());
        ListObjectsRequest request = OBSCommonUtils.createListObjectsRequest(owner, parentKey, "/", owner.getMaxKeys());
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        while (true) {
            for (String commonPrefix : objects.getCommonPrefixes()) {
                if (commonPrefix.equals(parentKey)) continue;
                delNum += (long)OBSPosixBucketUtils.fsRemoveSubdir(owner, commonPrefix, subdirList);
            }
            for (ObsObject sonObject : objects.getObjects()) {
                String sonObjectKey = sonObject.getObjectKey();
                if (sonObjectKey.equals(parentKey)) continue;
                if (!sonObjectKey.endsWith("/")) {
                    delNum += (long)OBSPosixBucketUtils.fsRemoveFile(owner, sonObjectKey, fileList);
                    continue;
                }
                delNum += (long)OBSPosixBucketUtils.fsRemoveSubdir(owner, sonObjectKey, subdirList);
            }
            if (!objects.isTruncated()) break;
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        delNum += (long)fileList.size();
        OBSCommonUtils.removeKeys(owner, fileList, true, false);
        delNum += (long)subdirList.size();
        OBSCommonUtils.removeKeys(owner, subdirList, true, false);
        if (deleteParent) {
            OBSCommonUtils.deleteObject(owner, parentKey);
            ++delNum;
        }
        return delNum;
    }

    private static boolean needToTrash(OBSFileSystem owner, String key) {
        String trashPathKey;
        String newKey = key;
        newKey = OBSCommonUtils.maybeDeleteBeginningSlash(newKey);
        if (owner.isEnableTrash() && newKey.startsWith(trashPathKey = OBSCommonUtils.pathToKey(owner, new Path(owner.getTrashDir())))) {
            return false;
        }
        return owner.isEnableTrash();
    }

    private static int fsRemoveSubdir(OBSFileSystem owner, String subdirKey, List<KeyAndVersion> subdirList) throws IOException {
        OBSPosixBucketUtils.fsRecursivelyDeleteDirWithRetry(owner, subdirKey, false);
        subdirList.add(new KeyAndVersion(subdirKey));
        if (subdirList.size() == owner.getMaxEntriesToDelete()) {
            OBSCommonUtils.removeKeys(owner, subdirList, true, false);
            return owner.getMaxEntriesToDelete();
        }
        return 0;
    }

    private static void mkTrash(OBSFileSystem owner, String key) throws ObsException, IOException {
        String newKey = key;
        StringBuilder sb = new StringBuilder(owner.getTrashDir());
        newKey = OBSCommonUtils.maybeAddTrailingSlash(newKey);
        sb.append(newKey);
        sb.deleteCharAt(sb.length() - 1);
        sb.delete(sb.lastIndexOf("/"), sb.length());
        Path fastDeleteRecycleDirPath = new Path(sb.toString());
        if (!owner.exists(fastDeleteRecycleDirPath)) {
            owner.mkdirs(fastDeleteRecycleDirPath);
        }
    }

    static void fsCreateFolder(OBSFileSystem owner, String objectName) throws IOException {
        String newObjectName = OBSCommonUtils.maybeAddTrailingSlash(objectName);
        NewFolderRequest newFolderRequest = new NewFolderRequest(owner.getBucket(), newObjectName);
        newFolderRequest.setAcl(owner.getCannedACL());
        long len = newFolderRequest.getObjectKey().length();
        int retryTime = 0;
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() - startTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
            try {
                owner.getObsClient().newFolder(newFolderRequest);
                owner.getSchemeStatistics().incrementWriteOps(1);
                owner.getSchemeStatistics().incrementBytesWritten(len);
                return;
            }
            catch (ObsException e) {
                LOG.debug("Failed to create folder [{}], retry time [{}], exception [{}]", new Object[]{newObjectName, retryTime, e});
                IOException ioException = OBSCommonUtils.translateException("innerFsCreateFolder", newObjectName, e);
                if (!(ioException instanceof OBSIOException)) {
                    throw ioException;
                }
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                try {
                    Thread.sleep(delayMs);
                }
                catch (InterruptedException ie) {
                    throw ioException;
                }
            }
        }
    }

    static OBSFileStatus innerFsGetObjectStatus(OBSFileSystem owner, Path f) throws IOException {
        Path path = OBSCommonUtils.qualify(owner, f);
        String key = OBSCommonUtils.pathToKey(owner, path);
        LOG.debug("Getting path status for {}  ({})", (Object)path, (Object)key);
        if (key.isEmpty()) {
            LOG.debug("Found root directory");
            return new OBSFileStatus(path, owner.getShortUserName());
        }
        try {
            GetAttributeRequest getAttrRequest = new GetAttributeRequest(owner.getBucket(), key);
            ObsFSAttribute meta = owner.getObsClient().getAttribute(getAttrRequest);
            owner.getSchemeStatistics().incrementReadOps(1);
            if (OBSPosixBucketUtils.fsIsFolder(meta)) {
                LOG.debug("Found file (with /): fake directory");
                return new OBSFileStatus(path, OBSCommonUtils.dateToLong(meta.getLastModified()), owner.getShortUserName());
            }
            LOG.debug("Found file (with /): real file? should not happen: {}", (Object)key);
            return new OBSFileStatus(meta.getContentLength(), OBSCommonUtils.dateToLong(meta.getLastModified()), path, owner.getDefaultBlockSize(path), owner.getShortUserName());
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("getFileStatus", path, e);
        }
    }

    static ContentSummary fsGetDirectoryContentSummary(OBSFileSystem owner, String key) throws IOException {
        String newKey = key;
        newKey = OBSCommonUtils.maybeAddTrailingSlash(newKey);
        long[] summary = new long[]{0L, 0L, 1L};
        LOG.debug("Summary key {}", (Object)newKey);
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(owner.getBucket());
        request.setPrefix(newKey);
        request.setMaxKeys(owner.getMaxKeys());
        ObjectListing objects = OBSCommonUtils.listObjects(owner, request);
        while (true) {
            if (!objects.getCommonPrefixes().isEmpty() || !objects.getObjects().isEmpty()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Found path as directory (with /): {}/{}", (Object)objects.getCommonPrefixes().size(), (Object)objects.getObjects().size());
                }
                for (String prefix : objects.getCommonPrefixes()) {
                    if (prefix.equals(newKey)) continue;
                    summary[2] = summary[2] + 1L;
                }
                for (ObsObject obj : objects.getObjects()) {
                    if (!obj.getObjectKey().endsWith("/")) {
                        summary[0] = summary[0] + obj.getMetadata().getContentLength();
                        summary[1] = summary[1] + 1L;
                        continue;
                    }
                    if (obj.getObjectKey().equals(newKey)) continue;
                    summary[2] = summary[2] + 1L;
                }
            }
            if (!objects.isTruncated()) break;
            objects = OBSCommonUtils.continueListObjects(owner, objects);
        }
        LOG.debug(String.format("file size [%d] - file count [%d] - directory count [%d] - file path [%s]", summary[0], summary[1], summary[2], newKey));
        return new ContentSummary.Builder().length(summary[0]).fileCount(summary[1]).directoryCount(summary[2]).spaceConsumed(summary[0]).build();
    }

    static void innerFsTruncateWithRetry(OBSFileSystem owner, Path f, long newLength) throws IOException {
        int retryTime = 0;
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() - startTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
            try {
                OBSPosixBucketUtils.innerFsTruncate(owner, f, newLength);
                return;
            }
            catch (OBSIOException e) {
                OBSFileSystem.LOG.debug("Failed to truncate [{}] to newLength [{}], retry time [{}], exception [{}]", new Object[]{f, newLength, retryTime, e});
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                try {
                    Thread.sleep(delayMs);
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        OBSPosixBucketUtils.innerFsTruncate(owner, f, newLength);
    }

    private static void innerFsTruncate(OBSFileSystem owner, Path f, long newLength) throws IOException {
        LOG.debug("truncate {} to newLength {}", (Object)f, (Object)newLength);
        try {
            String key = OBSCommonUtils.pathToKey(owner, f);
            owner.getObsClient().truncateObject(owner.getBucket(), key, newLength);
            owner.getSchemeStatistics().incrementWriteOps(1);
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("truncate", f, e);
        }
    }
}

