/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs.input;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import com.obs.services.ObsClient;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.Queue;
import java.util.concurrent.ExecutorService;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ByteBufferReadable;
import org.apache.hadoop.fs.CanSetReadahead;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.obs.input.ReadAheadBuffer;
import org.apache.hadoop.fs.obs.input.ReadAheadTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OBSExtendInputStream
extends FSInputStream
implements CanSetReadahead,
ByteBufferReadable {
    public static final Logger LOG = LoggerFactory.getLogger(OBSExtendInputStream.class);
    private OBSFileSystem fs;
    private final ObsClient client;
    private FileSystem.Statistics statistics;
    private final String bucketName;
    private final String key;
    private long contentLength;
    private boolean closed;
    private int maxReadAhead;
    private long readaheadSize;
    private long pos;
    private long nextPos;
    private long lastBufferStart;
    private byte[] buffer;
    private long bufferRemaining;
    private ExecutorService readAheadExecutorService;
    private Queue<ReadAheadBuffer> readAheadBufferQueue = new ArrayDeque<ReadAheadBuffer>();

    public OBSExtendInputStream(OBSFileSystem obsFileSystem, Configuration conf, ExecutorService readAheadExecutorService, String bucketName, String key, Long contentLength, FileSystem.Statistics statistics) {
        LOG.info("use OBSExtendInputStream");
        this.fs = obsFileSystem;
        this.client = this.fs.getObsClient();
        this.statistics = statistics;
        this.bucketName = bucketName;
        this.key = key;
        this.contentLength = contentLength;
        this.readaheadSize = conf.getLong("fs.obs.readahead.range", 0x100000L);
        this.maxReadAhead = conf.getInt("fs.obs.readahead.max.number", 4);
        this.readAheadExecutorService = MoreExecutors.listeningDecorator((ExecutorService)readAheadExecutorService);
        this.nextPos = 0L;
        this.lastBufferStart = -1L;
        this.pos = 0L;
        this.bufferRemaining = 0L;
        this.closed = false;
    }

    private void validateAndResetReopen(long pos) throws EOFException {
        if (pos < 0L) {
            throw new EOFException("Cannot seek at negative position:" + pos);
        }
        if (pos > this.contentLength) {
            throw new EOFException("Cannot seek after EOF, contentLength:" + this.contentLength + " position:" + pos);
        }
        if (this.buffer != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Aborting old stream to open at pos " + pos);
            }
            this.buffer = null;
        }
    }

    private boolean isRandom(long position) {
        boolean isRandom = true;
        if (position == this.nextPos) {
            isRandom = false;
        } else {
            while (this.readAheadBufferQueue.size() != 0 && this.readAheadBufferQueue.element().getByteStart() != position) {
                this.readAheadBufferQueue.poll();
            }
        }
        return isRandom;
    }

    private void getFromBuffer() throws IOException {
        ReadAheadBuffer readBuffer = this.readAheadBufferQueue.poll();
        readBuffer.lock();
        try {
            readBuffer.await(ReadAheadBuffer.STATUS.INIT);
            this.buffer = (byte[])(readBuffer.getStatus() == ReadAheadBuffer.STATUS.ERROR ? null : readBuffer.getBuffer());
        }
        catch (InterruptedException e) {
            LOG.warn("interrupted when wait a read buffer");
        }
        finally {
            readBuffer.unlock();
        }
        if (this.buffer == null) {
            throw new IOException("Null IO stream");
        }
    }

    private synchronized void reopen(long position) throws IOException {
        this.validateAndResetReopen(position);
        long partSize = position + this.readaheadSize > this.contentLength ? this.contentLength - position : this.readaheadSize;
        boolean isRandom = this.isRandom(position);
        this.nextPos = position + partSize;
        int currentSize = this.readAheadBufferQueue.size();
        if (currentSize == 0) {
            this.lastBufferStart = position - partSize;
        } else {
            ReadAheadBuffer[] readBuffers = this.readAheadBufferQueue.toArray(new ReadAheadBuffer[currentSize]);
            this.lastBufferStart = readBuffers[currentSize - 1].getByteStart();
        }
        int maxLen = this.maxReadAhead - currentSize;
        for (int i = 0; i < maxLen && i < (currentSize + 1) * 2 && this.lastBufferStart + partSize * (long)(i + 1) < this.contentLength; ++i) {
            ReadAheadBuffer readBuffer;
            long byteStart = this.lastBufferStart + partSize * (long)(i + 1);
            long byteEnd = byteStart + partSize - 1L;
            if (byteEnd >= this.contentLength) {
                byteEnd = this.contentLength - 1L;
            }
            if ((readBuffer = new ReadAheadBuffer(byteStart, byteEnd)).getBuffer().length == 0) {
                readBuffer.setStatus(ReadAheadBuffer.STATUS.SUCCESS);
            } else {
                this.readAheadExecutorService.execute(new ReadAheadTask(this.bucketName, this.key, this.client, readBuffer));
            }
            this.readAheadBufferQueue.add(readBuffer);
            if (isRandom) break;
        }
        this.getFromBuffer();
        this.pos = position;
        this.bufferRemaining = partSize;
    }

    public synchronized int read() throws IOException {
        this.checkNotClosed();
        if (this.bufferRemaining <= 0L && this.pos < this.contentLength) {
            this.reopen(this.pos);
        }
        int byteRead = -1;
        if (this.bufferRemaining != 0L) {
            byteRead = this.buffer[this.buffer.length - (int)this.bufferRemaining] & 0xFF;
        }
        if (byteRead >= 0) {
            ++this.pos;
            --this.bufferRemaining;
        }
        this.incrementBytesRead(byteRead);
        return byteRead;
    }

    private void checkNotClosed() throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed!");
        }
    }

    private void validateReadArgs(byte[] buf, int off, int len) {
        if (buf == null) {
            throw new NullPointerException();
        }
        if (off < 0 || len < 0 || len > buf.length - off) {
            throw new IndexOutOfBoundsException();
        }
    }

    private void incrementBytesRead(long bytesRead) {
        if (this.statistics != null && bytesRead > 0L) {
            this.statistics.incrementBytesRead(bytesRead);
        }
    }

    public synchronized int read(byte[] buf, int off, int len) throws IOException {
        this.checkNotClosed();
        this.validateReadArgs(buf, off, len);
        if (len == 0) {
            return 0;
        }
        int byteRead = 0;
        while (this.pos < this.contentLength && byteRead < len) {
            if (this.bufferRemaining == 0L) {
                this.reopen(this.pos);
            }
            int bytes = 0;
            for (int i = this.buffer.length - (int)this.bufferRemaining; i < this.buffer.length; ++i) {
                buf[off + byteRead] = this.buffer[i];
                ++bytes;
                if (off + ++byteRead >= len) break;
            }
            if (bytes > 0) {
                this.pos += (long)bytes;
                this.bufferRemaining -= (long)bytes;
                continue;
            }
            if (this.bufferRemaining == 0L) continue;
            throw new IOException("Failed to read from stream. Remaining:" + this.bufferRemaining);
        }
        this.incrementBytesRead(byteRead);
        if (byteRead == 0 && len > 0) {
            return -1;
        }
        return byteRead;
    }

    public synchronized void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.buffer = null;
    }

    public synchronized int available() throws IOException {
        this.checkNotClosed();
        long remain = this.contentLength - this.pos;
        if (remain > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)remain;
    }

    public synchronized void seek(long position) throws IOException {
        this.checkNotClosed();
        if (position < 0L) {
            throw new EOFException("Cannot seek to a negative offset " + position);
        }
        if (this.contentLength <= 0L) {
            return;
        }
        if (this.pos == position) {
            return;
        }
        if (position > this.pos && position < this.pos + this.bufferRemaining) {
            long len = position - this.pos;
            this.pos = position;
            this.bufferRemaining -= len;
        } else {
            this.pos = position;
            this.bufferRemaining = 0L;
        }
    }

    public synchronized long getPos() throws IOException {
        this.checkNotClosed();
        return this.pos;
    }

    public boolean seekToNewSource(long targetPos) throws IOException {
        this.checkNotClosed();
        return false;
    }

    public int read(ByteBuffer byteBuffer) throws IOException {
        int len = byteBuffer.remaining();
        if (len == 0) {
            return 0;
        }
        byte[] buf = new byte[len];
        int size = this.read(buf, 0, len);
        if (size != -1) {
            byteBuffer.put(buf, 0, size);
        }
        return size;
    }

    public synchronized void setReadahead(Long readahead) throws IOException {
        this.checkNotClosed();
        if (readahead == null) {
            this.readaheadSize = 0x100000L;
        } else {
            Preconditions.checkArgument((readahead >= 0L ? 1 : 0) != 0, (Object)"Negative readahead value");
            this.readaheadSize = readahead;
        }
    }
}

