/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs.input;

import com.google.common.base.Preconditions;
import com.obs.services.ObsClient;
import com.obs.services.exception.ObsException;
import com.obs.services.model.GetObjectRequest;
import com.sun.istack.NotNull;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.ByteBufferReadable;
import org.apache.hadoop.fs.CanSetReadahead;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.obs.BasicMetricsConsumer;
import org.apache.hadoop.fs.obs.OBSCommonUtils;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.obs.OBSIOException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class OBSInputStream
extends FSInputStream
implements CanSetReadahead,
ByteBufferReadable {
    public static final Logger LOG = LoggerFactory.getLogger(OBSInputStream.class);
    private final FileSystem.Statistics statistics;
    private final ObsClient client;
    private final String bucket;
    private final String key;
    private final long contentLength;
    private final String uri;
    private OBSFileSystem fs;
    private long streamCurrentPos;
    private volatile boolean closed;
    private InputStream wrappedStream = null;
    private long readAheadRange = 0x100000L;
    private long nextReadPos;
    private long contentRangeFinish;
    private long contentRangeStart;

    OBSInputStream(String bucketName, String bucketKey, long fileStatusLength, ObsClient obsClient, FileSystem.Statistics stats, long readAheadRangeValue, OBSFileSystem obsFileSystem) {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)bucketName), (Object)"No Bucket");
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)bucketKey), (Object)"No Key");
        Preconditions.checkArgument((fileStatusLength >= 0L ? 1 : 0) != 0, (Object)"Negative content length");
        this.bucket = bucketName;
        this.key = bucketKey;
        this.contentLength = fileStatusLength;
        this.client = obsClient;
        this.statistics = stats;
        this.uri = "obs://" + this.bucket + "/" + this.key;
        this.fs = obsFileSystem;
        this.readAheadRange = readAheadRangeValue;
    }

    static long calculateRequestLimit(long targetPos, long length, long contentLength, long readahead) {
        return Math.min(contentLength, length < 0L ? contentLength : targetPos + Math.max(readahead, length));
    }

    private synchronized void reopen(String reason, long targetPos, long length) throws IOException {
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        if (this.wrappedStream != null) {
            this.closeStream("reopen(" + reason + ")", this.contentRangeFinish);
        }
        this.contentRangeFinish = OBSInputStream.calculateRequestLimit(targetPos, length, this.contentLength, this.readAheadRange);
        try {
            GetObjectRequest request = new GetObjectRequest(this.bucket, this.key);
            request.setRangeStart(targetPos);
            request.setRangeEnd(this.contentRangeFinish);
            if (this.fs.getSse().isSseCEnable()) {
                request.setSseCHeader(this.fs.getSse().getSseCHeader());
            }
            this.wrappedStream = this.client.getObject(request).getObjectContent();
            this.contentRangeStart = targetPos;
            if (this.wrappedStream == null) {
                throw new IOException("Null IO stream from reopen of (" + reason + ") " + this.uri);
            }
        }
        catch (ObsException e) {
            throw OBSCommonUtils.translateException("Reopen at position " + targetPos, this.uri, e);
        }
        this.streamCurrentPos = targetPos;
        long endTime = System.currentTimeMillis();
        LOG.debug("reopen({}) for {} range[{}-{}], length={}, streamPosition={}, nextReadPosition={}, thread={}, timeUsedInMilliSec={}", new Object[]{this.uri, reason, targetPos, this.contentRangeFinish, length, this.streamCurrentPos, this.nextReadPos, threadId, endTime - startTime});
    }

    public synchronized long getPos() throws IOException {
        this.fs.checkOpen();
        this.checkStreamOpen();
        return this.nextReadPos < 0L ? 0L : this.nextReadPos;
    }

    public synchronized void seek(long targetPos) throws IOException {
        this.fs.checkOpen();
        this.checkStreamOpen();
        if (targetPos < 0L) {
            throw new EOFException("Cannot seek to a negative offset " + targetPos);
        }
        if (this.contentLength <= 0L) {
            return;
        }
        this.nextReadPos = targetPos;
    }

    private void seekQuietly(long positiveTargetPos) {
        try {
            this.seek(positiveTargetPos);
        }
        catch (IOException ioe) {
            LOG.debug("Ignoring IOE on seek of {} to {}", new Object[]{this.uri, positiveTargetPos, ioe});
        }
    }

    private void seekInStream(long targetPos) throws IOException {
        this.checkStreamOpen();
        if (this.wrappedStream == null) {
            return;
        }
        long diff = targetPos - this.streamCurrentPos;
        if (diff > 0L) {
            boolean skipForward;
            int available = this.wrappedStream.available();
            long forwardSeekRange = Math.max(this.readAheadRange, (long)available);
            long remainingInCurrentRequest = this.remainingInCurrentRequest();
            long forwardSeekLimit = Math.min(remainingInCurrentRequest, forwardSeekRange);
            boolean bl = skipForward = remainingInCurrentRequest > 0L && diff <= forwardSeekLimit;
            if (skipForward) {
                LOG.debug("Forward seek on {}, of {} bytes", (Object)this.uri, (Object)diff);
                long skippedOnce = this.wrappedStream.skip(diff);
                while (diff > 0L && skippedOnce > 0L) {
                    this.streamCurrentPos += skippedOnce;
                    this.incrementBytesRead(skippedOnce);
                    skippedOnce = this.wrappedStream.skip(diff -= skippedOnce);
                }
                if (this.streamCurrentPos == targetPos) {
                    return;
                }
                LOG.info("Failed to seek on {} to {}. Current position {}", new Object[]{this.uri, targetPos, this.streamCurrentPos});
            }
        } else if (diff == 0L && this.remainingInCurrentRequest() > 0L) {
            return;
        }
        this.closeStream("seekInStream()", this.contentRangeFinish);
        this.streamCurrentPos = targetPos;
    }

    public boolean seekToNewSource(long targetPos) throws IOException {
        this.fs.checkOpen();
        this.checkStreamOpen();
        return false;
    }

    private void lazySeek(long targetPos, long len) throws IOException {
        int retryTime = 0;
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() - startTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
            long delayMs;
            try {
                this.seekInStream(targetPos);
            }
            catch (IOException e) {
                if (this.wrappedStream != null) {
                    this.closeStream("lazySeek() seekInStream has exception ", this.contentRangeFinish);
                }
                LOG.warn("IOException occurred in lazySeek, retry: {}", (Object)retryTime, (Object)e);
                delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs < OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
                    try {
                        Thread.sleep(delayMs);
                        continue;
                    }
                    catch (InterruptedException ie) {
                        throw e;
                    }
                }
                throw e;
            }
            try {
                if (this.wrappedStream == null) {
                    this.reopen("read from new offset", targetPos, len);
                }
                return;
            }
            catch (OBSIOException e) {
                LOG.debug("IOException occurred in lazySeek, retry: {}", (Object)retryTime, (Object)e);
                delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs < OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
                    try {
                        Thread.sleep(delayMs);
                        continue;
                    }
                    catch (InterruptedException ie) {
                        throw e;
                    }
                }
                throw e;
            }
        }
    }

    private void incrementBytesRead(long bytesRead) {
        if (this.statistics != null && bytesRead > 0L) {
            this.statistics.incrementBytesRead(bytesRead);
        }
    }

    private void sleepInLock(long sleepTime) throws InterruptedException {
        long start;
        long now = start = System.currentTimeMillis();
        while (now - start < sleepTime) {
            ((Object)((Object)this)).wait(start + sleepTime - now);
            now = System.currentTimeMillis();
        }
    }

    public synchronized int read() throws IOException {
        long position;
        boolean isTrue;
        this.fs.checkOpen();
        this.checkStreamOpen();
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        boolean bl = isTrue = this.contentLength == 0L || this.nextReadPos >= this.contentLength;
        if (isTrue) {
            return -1;
        }
        int byteRead = -1;
        try {
            this.lazySeek(this.nextReadPos, 1L);
        }
        catch (EOFException e) {
            this.onReadFailure(e, 1);
            return -1;
        }
        IOException exception = null;
        int retryTime = 0;
        long retryStartTime = System.currentTimeMillis();
        while (true) {
            try {
                byteRead = this.wrappedStream.read();
                exception = null;
            }
            catch (EOFException e) {
                this.onReadFailure(e, 1);
                return -1;
            }
            catch (IOException e) {
                exception = e;
                this.onReadFailure(e, 1);
                LOG.debug("read of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{this.uri, retryTime, e});
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                try {
                    this.sleepInLock(delayMs);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.error("read of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{this.uri, retryTime, e});
                    throw e;
                }
                if (System.currentTimeMillis() - retryStartTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
            }
            break;
        }
        if (exception != null) {
            long endTime = System.currentTimeMillis();
            if (this.fs.getMetricSwitch()) {
                BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("byteBuf", "read", false, endTime - startTime);
                OBSCommonUtils.setMetricsInfo(this.fs, record);
            }
            LOG.error("read of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{this.uri, retryTime, exception});
            throw exception;
        }
        if (byteRead >= 0) {
            ++this.streamCurrentPos;
            ++this.nextReadPos;
        }
        if (byteRead >= 0) {
            this.incrementBytesRead(1L);
        }
        long endTime = System.currentTimeMillis();
        long l = position = byteRead >= 0 ? this.nextReadPos - 1L : this.nextReadPos;
        if (this.fs.getMetricSwitch()) {
            BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("1byte", "read", true, endTime - startTime);
            OBSCommonUtils.setMetricsInfo(this.fs, record);
        }
        LOG.debug("read-0arg uri:{}, contentLength:{}, position:{}, readValue:{}, thread:{}, timeUsedMilliSec:{}", new Object[]{this.uri, this.contentLength, position, byteRead, threadId, endTime - startTime});
        return byteRead;
    }

    private synchronized void onReadFailure(IOException ioe, int length) throws IOException {
        LOG.debug("Got exception while trying to read from stream {} trying to recover: " + ioe, (Object)this.uri);
        int retryTime = 0;
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() - startTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) {
            try {
                this.reopen("failure recovery", this.streamCurrentPos, length);
                return;
            }
            catch (OBSIOException e) {
                LOG.debug("OBSIOException occurred in reopen for failure recovery, the {} retry time", (Object)retryTime, (Object)e);
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                try {
                    this.sleepInLock(delayMs);
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
        this.reopen("failure recovery", this.streamCurrentPos, length);
    }

    public synchronized int read(ByteBuffer byteBuffer) throws IOException {
        boolean isTrue;
        this.fs.checkOpen();
        this.checkStreamOpen();
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        LOG.debug("read byteBuffer: {}", (Object)byteBuffer.toString());
        int len = byteBuffer.remaining();
        if (len == 0) {
            return 0;
        }
        byte[] buf = new byte[len];
        boolean bl = isTrue = this.contentLength == 0L || this.nextReadPos >= this.contentLength;
        if (isTrue) {
            return -1;
        }
        try {
            this.lazySeek(this.nextReadPos, len);
        }
        catch (EOFException e) {
            this.onReadFailure(e, len);
            return -1;
        }
        int bytesRead = 0;
        IOException exception = null;
        int retryTime = 0;
        long startRetryTime = System.currentTimeMillis();
        while (true) {
            try {
                bytesRead = this.tryToReadFromInputStream(this.wrappedStream, buf, 0, len);
                if (bytesRead == -1) {
                    return -1;
                }
                exception = null;
            }
            catch (EOFException e) {
                this.onReadFailure(e, len);
                return -1;
            }
            catch (IOException e) {
                exception = e;
                this.onReadFailure(e, len);
                LOG.debug("read len[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{len, this.uri, retryTime, exception});
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                try {
                    this.sleepInLock(delayMs);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.error("read len[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{len, this.uri, retryTime, exception});
                    throw exception;
                }
                if (System.currentTimeMillis() - startRetryTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
            }
            break;
        }
        if (exception != null) {
            long endTime = System.currentTimeMillis();
            if (this.fs.getMetricSwitch()) {
                BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("byteBuf", "read", false, endTime - startTime);
                OBSCommonUtils.setMetricsInfo(this.fs, record);
            }
            LOG.error("read len[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{len, this.uri, retryTime, exception});
            throw exception;
        }
        if (bytesRead > 0) {
            this.streamCurrentPos += (long)bytesRead;
            this.nextReadPos += (long)bytesRead;
            byteBuffer.put(buf, 0, bytesRead);
        }
        this.incrementBytesRead(bytesRead);
        long position = bytesRead >= 0 ? this.nextReadPos - 1L : this.nextReadPos;
        long endTime = System.currentTimeMillis();
        if (this.fs.getMetricSwitch()) {
            BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("byteBuf", "read", true, endTime - startTime);
            OBSCommonUtils.setMetricsInfo(this.fs, record);
        }
        LOG.debug("Read-ByteBuffer uri:{}, contentLength:{}, destLen:{}, readLen:{}, position:{}, thread:{}, timeUsedMilliSec:{}", new Object[]{this.uri, this.contentLength, len, bytesRead, position, threadId, endTime - startTime});
        return bytesRead;
    }

    private int tryToReadFromInputStream(InputStream in, byte[] buf, int off, int len) throws IOException {
        int bytesRead;
        int bytes;
        for (bytesRead = 0; bytesRead < len; bytesRead += bytes) {
            bytes = in.read(buf, off + bytesRead, len - bytesRead);
            if (bytes != -1) continue;
            if (bytesRead != 0) break;
            return -1;
        }
        return bytesRead;
    }

    public synchronized int read(@NotNull byte[] buf, int off, int len) throws IOException {
        boolean isTrue;
        this.fs.checkOpen();
        this.checkStreamOpen();
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        this.validatePositionedReadArgs(this.nextReadPos, buf, off, len);
        if (len == 0) {
            return 0;
        }
        boolean bl = isTrue = this.contentLength == 0L || this.nextReadPos >= this.contentLength;
        if (isTrue) {
            return -1;
        }
        try {
            this.lazySeek(this.nextReadPos, len);
        }
        catch (EOFException e) {
            this.onReadFailure(e, len);
            return -1;
        }
        int bytesRead = 0;
        IOException exception = null;
        int retryTime = 0;
        long startRetryTime = System.currentTimeMillis();
        while (true) {
            try {
                bytesRead = this.tryToReadFromInputStream(this.wrappedStream, buf, off, len);
                if (bytesRead == -1) {
                    return -1;
                }
                exception = null;
            }
            catch (EOFException e) {
                this.onReadFailure(e, len);
                return -1;
            }
            catch (IOException e) {
                exception = e;
                this.onReadFailure(e, len);
                LOG.debug("read offset[{}] len[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{off, len, this.uri, retryTime, exception});
                long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
                ++retryTime;
                if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
                try {
                    this.sleepInLock(delayMs);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.error("read offset[{}] len[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{off, len, this.uri, retryTime, exception});
                    throw exception;
                }
                if (System.currentTimeMillis() - startRetryTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
            }
            break;
        }
        if (exception != null) {
            long endTime = System.currentTimeMillis();
            if (this.fs.getMetricSwitch()) {
                BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("seq", "read", false, endTime - startTime);
                OBSCommonUtils.setMetricsInfo(this.fs, record);
            }
            LOG.error("read offset[{}] len[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{off, len, this.uri, retryTime, exception});
            throw exception;
        }
        if (bytesRead > 0) {
            this.streamCurrentPos += (long)bytesRead;
            this.nextReadPos += (long)bytesRead;
        }
        this.incrementBytesRead(bytesRead);
        long endTime = System.currentTimeMillis();
        long costTime = endTime - startTime;
        this.readMetric(costTime);
        long position = bytesRead >= 0 ? this.nextReadPos - 1L : this.nextReadPos;
        LOG.debug("Read-3args uri:{}, contentLength:{}, destLen:{}, readLen:{}, position:{}, thread:{}, timeUsedMilliSec:{}", new Object[]{this.uri, this.contentLength, len, bytesRead, position, threadId, endTime - startTime});
        return bytesRead;
    }

    private void readMetric(long costTime) {
        if (this.fs.getMetricSwitch()) {
            BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("seq", "read", true, costTime);
            OBSCommonUtils.setMetricsInfo(this.fs, record);
        }
    }

    private void checkStreamOpen() throws IOException {
        if (this.closed) {
            throw new IOException(this.uri + ": " + "Stream is closed!");
        }
    }

    public synchronized void close() throws IOException {
        long startTime = System.currentTimeMillis();
        if (!this.closed) {
            this.fs.checkOpen();
            this.closeStream("close() operation", this.contentRangeFinish);
            super.close();
            this.closed = true;
        }
        long endTime = System.currentTimeMillis();
        if (this.fs.getMetricSwitch()) {
            BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("input", "close", true, endTime - startTime);
            OBSCommonUtils.setMetricsInfo(this.fs, record);
        }
    }

    private synchronized void closeStream(String reason, long length) throws IOException {
        if (this.wrappedStream != null) {
            try {
                this.wrappedStream.close();
            }
            catch (IOException e) {
                LOG.debug("When closing {} stream for {}", new Object[]{this.uri, reason, e});
                throw e;
            }
            LOG.debug("Stream {} : {}; streamPos={}, nextReadPos={}, request range {}-{} length={}", new Object[]{this.uri, reason, this.streamCurrentPos, this.nextReadPos, this.contentRangeStart, this.contentRangeFinish, length});
            this.wrappedStream = null;
        }
    }

    public synchronized int available() throws IOException {
        this.fs.checkOpen();
        this.checkStreamOpen();
        long remaining = this.remainingInFile();
        if (remaining > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)remaining;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    private synchronized long remainingInFile() {
        return this.contentLength - this.streamCurrentPos;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    private synchronized long remainingInCurrentRequest() {
        return this.contentRangeFinish - this.streamCurrentPos;
    }

    public boolean markSupported() {
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @InterfaceStability.Unstable
    public String toString() {
        OBSInputStream oBSInputStream = this;
        synchronized (oBSInputStream) {
            return "OBSInputStream{" + this.uri + " wrappedStream=" + (this.wrappedStream != null ? "open" : "closed") + " streamCurrentPos=" + this.streamCurrentPos + " nextReadPos=" + this.nextReadPos + " contentLength=" + this.contentLength + " contentRangeStart=" + this.contentRangeStart + " contentRangeFinish=" + this.contentRangeFinish + " remainingInCurrentRequest=" + this.remainingInCurrentRequest() + '}';
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void readFully(long position, byte[] buffer, int offset, int length) throws IOException {
        int nread;
        this.fs.checkOpen();
        this.checkStreamOpen();
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        this.validatePositionedReadArgs(position, buffer, offset, length);
        if (length == 0) {
            return;
        }
        OBSInputStream oBSInputStream = this;
        synchronized (oBSInputStream) {
            long oldPos = this.getPos();
            try {
                int nbytes;
                this.seek(position);
                for (nread = 0; nread < length; nread += nbytes) {
                    nbytes = this.read(buffer, offset + nread, length - nread);
                    if (nbytes >= 0) continue;
                    throw new EOFException("End of file reached before reading fully.");
                }
            }
            finally {
                this.seekQuietly(oldPos);
            }
        }
        long endTime = System.currentTimeMillis();
        if (this.fs.getMetricSwitch()) {
            BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord(null, "readFully", true, endTime - startTime);
            OBSCommonUtils.setMetricsInfo(this.fs, record);
        }
        LOG.debug("ReadFully uri:{}, contentLength:{}, destLen:{}, readLen:{}, position:{}, thread:{}, timeUsedMilliSec:{}", new Object[]{this.uri, this.contentLength, length, nread, position, threadId, endTime - startTime});
    }

    public int read(long position, byte[] buffer, int offset, int length) throws IOException {
        this.fs.checkOpen();
        this.checkStreamOpen();
        int len = length;
        long startTime = System.currentTimeMillis();
        this.validatePositionedReadArgs(position, buffer, offset, len);
        if (position < 0L || position >= this.contentLength) {
            return -1;
        }
        if (position + (long)len > this.contentLength) {
            len = (int)(this.contentLength - position);
        }
        if (this.fs.isReadTransformEnabled()) {
            int readSize = super.read(position, buffer, offset, len);
            long endTime = System.currentTimeMillis();
            if (this.fs.getMetricSwitch()) {
                BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("random", "read", true, endTime - startTime);
                OBSCommonUtils.setMetricsInfo(this.fs, record);
            }
            return readSize;
        }
        int readSize = this.randomReadWithNewInputStream(position, buffer, offset, len);
        long endTime = System.currentTimeMillis();
        if (this.fs.getMetricSwitch()) {
            BasicMetricsConsumer.MetricRecord record = new BasicMetricsConsumer.MetricRecord("random", "read", true, endTime - startTime);
            OBSCommonUtils.setMetricsInfo(this.fs, record);
        }
        return readSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int randomReadWithNewInputStream(long position, byte[] buffer, int offset, int length) throws IOException {
        long startTime = System.currentTimeMillis();
        long threadId = Thread.currentThread().getId();
        int bytesRead = 0;
        InputStream inputStream = null;
        IOException exception = null;
        GetObjectRequest request = new GetObjectRequest(this.bucket, this.key);
        request.setRangeStart(position);
        request.setRangeEnd(position + (long)length);
        if (this.fs.getSse().isSseCEnable()) {
            request.setSseCHeader(this.fs.getSse().getSseCHeader());
        }
        int retryTime = 0;
        long startRetryTime = System.currentTimeMillis();
        do {
            block18: {
                exception = null;
                try {
                    inputStream = this.client.getObject(request).getObjectContent();
                }
                catch (ObsException e) {
                    exception = OBSCommonUtils.translateException("Read at position " + position, this.uri, e);
                    LOG.debug("read position[{}] destLen[{}] offset[{}] readLen[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{position, length, offset, bytesRead, this.uri, retryTime, exception});
                    if (exception instanceof OBSIOException) break block18;
                    throw exception;
                }
            }
            if (exception == null) {
                try {
                    bytesRead = this.tryToReadFromInputStream(inputStream, buffer, offset, length);
                    if (bytesRead == -1) {
                        int e = -1;
                        return e;
                    }
                    exception = null;
                    break;
                }
                catch (EOFException e) {
                    this.onReadFailure(e, length);
                    int n = -1;
                    return n;
                }
                catch (IOException e) {
                    exception = e;
                    LOG.debug("read position[{}] destLen[{}] offset[{}] readLen[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{position, length, offset, bytesRead, this.uri, retryTime, exception});
                }
                finally {
                    if (inputStream != null) {
                        inputStream.close();
                    }
                }
            }
            long delayMs = OBSCommonUtils.getSleepTimeInMs(retryTime);
            ++retryTime;
            if (System.currentTimeMillis() - startTime + delayMs >= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY) continue;
            try {
                Thread.sleep(delayMs);
            }
            catch (InterruptedException ie) {
                LOG.error("read position[{}] destLen[{}] offset[{}] readLen[{}] of [{}] failed, retry time[{}], due to exception[{}]", new Object[]{position, length, offset, bytesRead, this.uri, retryTime, exception});
                throw exception;
            }
        } while (System.currentTimeMillis() - startRetryTime <= OBSCommonUtils.MAX_TIME_IN_MILLISECONDS_TO_RETRY);
        if (inputStream == null || exception != null) {
            IOException e = new IOException("read failed of " + this.uri + ", inputStream is " + (inputStream == null ? "null" : "not null"), exception);
            LOG.error("read position[{}] destLen[{}] offset[{}] len[{}] failed, retry time[{}], due to exception[{}]", new Object[]{position, length, offset, bytesRead, retryTime, exception});
            throw e;
        }
        long endTime = System.currentTimeMillis();
        LOG.debug("Read-4args uri:{}, contentLength:{}, destLen:{}, readLen:{}, position:{}, thread:{}, timeUsedMilliSec:{}", new Object[]{this.uri, this.contentLength, length, bytesRead, position, threadId, endTime - startTime});
        return bytesRead;
    }

    public synchronized void setReadahead(Long newReadaheadRange) throws IOException {
        this.fs.checkOpen();
        this.checkStreamOpen();
        if (newReadaheadRange == null) {
            this.readAheadRange = 0x100000L;
        } else {
            Preconditions.checkArgument((newReadaheadRange >= 0L ? 1 : 0) != 0, (Object)"Negative readahead value");
            this.readAheadRange = newReadaheadRange;
        }
    }
}

