#!/bin/bash

cd `dirname "$0"`
BASE_PATH=`pwd`
EXPECT_SCRIPT="${BASE_PATH}/exec_cli_cmd.ex"
#日志文件
LOG_FILE="/var/log/batch_handle_quota.log"
CLI_RESULT="${BASE_PATH}/tmpBatchHandleQuotaCliResult"
ERR_CLI="${BASE_PATH}/err_quota_cli"

# 修改配额命令前缀
CHANGE_QUOTA_CLI_PREFFIX="change quota normal"

TITLE_PATH="Quota Path"
TITLE_USER="User"
TITLE_ADVISE="Recommended Threshold"
TITLE_SOFT="Soft Threshold"
TITLE_TIME="Soft Threshold Grace Period"
TITLE_HARD="Hard Threshold"
PARAM_LIST=("${TITLE_PATH}" "${TITLE_USER}" "${TITLE_ADVISE}" "${TITLE_SOFT}" "${TITLE_TIME}" "${TITLE_HARD}")

# admin 密码
Passwd=""
path_ori=""
user_ori=""
err_task_count=0
create_count=0
change_count=0
limit_flag=0
# 不存在的用户
not_exist_user=()

config_path=$1

nas="${BASE_PATH}/nas.sh"
echo '#!/bin/bash' > $nas
echo "\"\$@\"" >> $nas
echo " " >> $nas
chmod +x $nas

#日志打印 日志量不大，可以手动清理
function LOG
{
    echo [`date`]$@ >> ${LOG_FILE}
}

function echo_red()
{
    echo -e "\033[31m$1\033[m"
}

function usage()
{
    local err_msg="${1}"

    echo "Error:"
    echo "       ${err_msg}"
    echo "Usage:"
    echo "       sh $0 <config_path>"
    echo "Parameter Description:"
    echo "       <config_path>        Quota config table, must be a normal CSV file."
    echo "For Example: sh $0 ${BASE_PATH}/quota_config.csv"
}


# 入参检查
function checkParam()
{
    if [ -z "${config_path}" ]; then
        usage "the 1st param is empty."
        exit 1
    fi

    if [ ! -f ${config_path} ]; then
        usage "${config_path} is not a file."
        exit 1
    fi

    echo ${config_path} | grep -q "\.csv$"
    if [ $? -eq 1 ]; then
        usage "${config_path} is not a normal CSV file."
        exit 1
    fi
}

# 输入密码，并检查密码的正确性
function inputPwd()
{
    read -s -p "Enter admin password: " Passwd
    echo "" 
    expect ${EXPECT_SCRIPT} ${Passwd} > /dev/null
    if [ $? -eq 1 ]; then
        echo "The admin password is incorrect."
        exit 1
    fi
}

# 检查csv对应参数是否合法，校验不通过返回1，处理下一行
function checkLineValid()
{
    local line=$1
    local count=$2
    local column=0
    local param_num=0
    local param_transform=""
    local param
    
    # 表头不处理
    if [ ${count} -eq 1 ]; then
        return 2
    fi

    # 只处理有逗号分隔符的行号
    ret=$(echo ${line} | awk -F, '{print NF}' | awk '{if($1==6)print 1}')
    if [ -z "${ret}" ]; then
        return 2
    fi

    # 跳过因清空csv数据残留的空行
    echo ${line} | grep -q "^,,,,,"
    if [ $? -eq 0 ]; then
        LOG "skip empty line(${count})"
        return 2
    fi

    for i in {0..5}
    do
        column=$(($i+1))
        param=$(echo ${line} | sed "s/\"//g" | tr -d "\n" | tr -d "\r" | awk -F, '{print $'"${column}"'}')
        if [ -z "${param}" ]; then
            echo_red "[ERR]the param of ${PARAM_LIST[$i]} is empty at line(${count}) of csv file"
            LOG "[check_err]the param of ${PARAM_LIST[$i]} is empty at line(${count}) of csv file"
            return 1
        fi

        if [ $i -lt 2 ]; then
            # path参数必须以'/'开头
            if [ $i -eq 0 ]; then
                echo ${param} | grep -q "^/"
                if [ $? -eq 1 ]; then
                    echo_red "[ERR]the param of ${PARAM_LIST[$i]}(${param}) must start with '/' at line(${count}) of csv file"
                    LOG "[check_err]the param of ${PARAM_LIST[$i]}(${param}) is not start with '/' at line(${count}) of csv file"
                    return 1
                fi
                # 检查path是否存在
                $nas [ ! -e "/mnt/fs/share${param}" ] && {
                    echo_red "[ERR]the ${TITLE_PATH}(${param}) does not exist at line(${count}) of csv file"
                    LOG "[check_err]the ${TITLE_PATH}(${param}) does not exist at line(${count}) of csv file"
                    return 1
                    }
                # 检查目录是否存在
                $nas [ ! -d "/mnt/fs/share${param}" ] && {
                    echo_red "[ERR]the ${TITLE_PATH}(${param}) is not a dir at line(${count}) of csv file"
                    LOG "[check_err]the ${TITLE_PATH}(${param}) is not a dir at line(${count}) of csv file"
                    return 1
                    }
                continue
            fi

            # 如果是之前检查出 不存在的用户，直接校验不通过，优化时间
            echo ${not_exist_user[@]} | grep -wq "${param}"
            if [ $? -eq 0 ]; then
                echo_red "[ERR]the ${TITLE_USER}(${param}) does not exist at line(${count}) of csv file"
                LOG "[check_err]the ${TITLE_USER}(${param}) does not exist at line(${count}) of csv file"
                return 1
            fi
            continue
        fi

        # 数值类配置
        param_num=$(echo ${param} | egrep '^[[:digit:]]+$')
        if [ -z ${param_num} ]; then
            echo_red "[ERR]the param of ${PARAM_LIST[$i]}(${param}) must be a Non-negative integer at line(${count}) of csv file"
            LOG "[check_err]the param of ${PARAM_LIST[$i]}(${param}) is not a Non-negative integer at line(${count}) of csv file"
            return 1
        fi
    done

    return 0
}

# 检查相应目录下对于用户的配额是否存在，存在打印配额id；没有则打印0
function checkQuotaIsExist()
{
    local path=$1
    local user=$2
    local tmpFile=${CLI_RESULT}Check
    local count=0
    local quota_id=0
    local cli_cmd="show quota normal all path=${path} user_or_group_name=${user}"

    expect ${EXPECT_SCRIPT} ${Passwd} "${cli_cmd}" > ${CLI_RESULT}
    # 将结果记录到日志中，便于问题定位
    sed '1,14d' ${CLI_RESULT} | awk '{print "[queryQuota]"$0}' >> ${LOG_FILE}

    # 检查命令是否出错，命令出错大概率是路径或用户名非法
    # 打出来并记录到命令出错文件里
    ret=$(grep "Error: Cmd" ${CLI_RESULT} | tee -a ${ERR_CLI})
    if [ ! -z "${ret}" ]; then
        return 1
    fi

    #过滤AD域类型用户的配额
    cat ${CLI_RESULT} | awk '{if($1=="id"||$1=="domain_type")print $3}' | grep -B1 "AD" | grep -v "AD" > ${tmpFile}
    count=$(cat ${tmpFile} | wc -l)
    if [ ${count} -eq 1 ]; then
        quota_id=$(cat ${tmpFile}) 
    fi

    LOG "query quota count(${count}) for path(${path})user(${user})"
    [ -f ${tmpFile} ] && rm -rf ${tmpFile}
    echo ${quota_id}
}

# 创建配额
function createQuota()
{
    local path=$1
    local user=$2
    local advise_threshold=$3
    local soft_threshold=$4
    local soft_threshold_extend_time=$5
    local hard_threshold=$6
    local count=$7
    local cli_cmd=""

    if [ ${limit_flag} -eq 1 ]; then
        echo_red "[ERR]The number of quotas has reached the upper limit. Can not create new quota for line(${count}) of csv file."
        LOG "[create_err]The number of quotas has reached the upper limit. Can not create new quota for path(${path}) user(${user}) at line(${count}) of csv file."
        let err_task_count++
        return
    fi
    
    # 创建配额命令:
    # create quota normal <resource_type> <quota_type> [advise_threshold] [soft_threshold] [soft_threshold_extend_time] [hard_threshold] <path> <user_or_group_name> [user_type] [domain_type]
    # <is_default>          normal: 普通配额
    # <resource_type>       CAPACITY：容量配额
    # <quota_type>          calculation_quota：计算配额，只监控存储空间使用情况不限制使用; mandatory_quota：强制配额，监控存储空间使用情况的同时限制存储空间的使用
    local CREATE_QUOTA_CLI_PREFFIX="create quota normal CAPACITY mandatory_quota"
    # <user_type>           USER：用户
    # <domain_type>         AD：使用AD（Active Directory）协议进行认证的域用户。
    local CREATE_QUOTA_CLI_SUFFIX="USER AD"
    cli_cmd=${CREATE_QUOTA_CLI_PREFFIX}" ${advise_threshold} ${soft_threshold}"
    if [ ${soft_threshold} -ne 0 ];then
        cli_cmd=${cli_cmd}" ${soft_threshold_extend_time}"
    fi
    cli_cmd=${cli_cmd}" ${hard_threshold} ${path} ${user} "${CREATE_QUOTA_CLI_SUFFIX}

    expect ${EXPECT_SCRIPT} ${Passwd} "${cli_cmd}" > ${CLI_RESULT}
    # 将结果记录到日志中，便于问题定位
    sed '1,14d' ${CLI_RESULT} | awk '{print "[createQuota]"$0}' >> ${LOG_FILE}

    # 检查命令是否执行成功
    grep -q "Command executed successfully." ${CLI_RESULT}
    if [ $? -eq 0 ]; then
        let create_count++
        return
    fi

    let err_task_count++
    grep -q "Error: The directory does not exist." ${CLI_RESULT}
    if [ $? -eq 0 ]; then
        echo_red "[ERR]the ${TITLE_PATH}(${path_ori}) does not exist at line(${count}) of csv file"
        LOG "[create_err]the ${TITLE_PATH}(${path_ori}) does not exist at line(${count}) of csv file"
        return
    fi

    grep -q "Error: The quota user does not exist." ${CLI_RESULT}
    if [ $? -eq 0 ]; then
        not_exist_user+=("${user_ori}")
        echo_red "[ERR]the ${TITLE_USER}(${user_ori}) does not exist at line(${count}) of csv file"
        LOG "[create_err]the ${TITLE_USER}(${user_ori}) does not exist at line(${count}) of csv file"
        return
    fi

    grep -q "Error: The number of quotas has reached the upper limit." ${CLI_RESULT}
    if [ $? -eq 0 ]; then
        limit_flag=1
        echo_red "[ERR]The number of quotas has reached the upper limit. Can not create new quota for line(${count}) of csv file."
        LOG "[create_err]The number of quotas has reached the upper limit. Can not create new quota for path(${path}) user(${user}) at line(${count}) of csv file."
        return
    fi

    echo_red "[ERR]create quota error for ${TITLE_PATH}(${path_ori}) ${TITLE_USER}(${user_ori}) at line(${count}) of csv file"
    LOG "[create_err]create quota error for path(${path}) user(${user}) at line(${count}) of csv file."
    return
}

# 修改配额
function changeQuota()
{
    local quota_id=$1
    local advise_threshold=$2
    local soft_threshold=$3
    local soft_threshold_extend_time=$4
    local hard_threshold=$5
    local count=$6
    local cli_cmd=""
    
    # 修改配额命令:
    # change quota normal <id> [quota_type=?] [advise_threshold=?] [soft_threshold=?] [soft_threshold_extend_time] [hard_threshold=?]
    cli_cmd="change quota normal ${quota_id} quota_type=mandatory_quota advise_threshold=${advise_threshold} hard_threshold=${hard_threshold} soft_threshold=${soft_threshold}"
    if [ ${soft_threshold} -ne 0 ]; then
        cli_cmd=${cli_cmd}" ${soft_threshold_extend_time}"
    fi

    expect ${EXPECT_SCRIPT} ${Passwd} "${cli_cmd}" > ${CLI_RESULT}
    # 将结果记录到日志中，便于问题定位
    sed '1,14d' ${CLI_RESULT} | awk '{print "[changeQuota]"$0}' >> ${LOG_FILE}

    # 检查命令是否执行成功
    grep -q "Command executed successfully." ${CLI_RESULT}
    if [ $? -eq 0 ]; then
        let change_count++
        return 0
    fi
    let err_task_count++

    echo_red "[ERR]change quota error for ${TITLE_PATH}(${path_ori}) ${TITLE_USER}(${user_ori}) at line(${count}) of csv file"
    LOG "[change_err]change quota error for quota_id(${quota_id}) at line(${count}) of csv file."
    return
}

function main()
{
    local count=0
    # <advise_threshold>  the resource_type is CAPACITY , the unit is GB
    local advise_threshold=0
    # <soft_threshold>    the resource_type is CAPACITY , the unit is GB
    local soft_threshold=0
    # <soft_threshold_extend_time>    Extend time of the soft_threshold, units:DAY(0-4294967294)
    local soft_threshold_extend_time=0
    # <hard_threshold>    the resource_type is CAPACITY , the unit is GB
    local hard_threshold=0
    local cli_cmd=""
    local quota_id=0
    # 任务总数, csv文件有效行数
    local valid_total=0
    # csv文件总行数
    local csv_line_total=0

    checkParam
    # 输入密码，并检查密码的正确性
    inputPwd
    
    csv_line_total=$(cat ${config_path} | wc -l)
    valid_total=$(cat ${config_path} | grep -v "Soft Threshold Grace Period(days)" | grep -v "^,,,,," | awk -F, '{print NF}' | awk '{if($1==6)print 1}' | wc -l)
    echo "batch handle quota task is start, there're ${valid_total} quota task to handle, please wait"
    while read -r line
    do
        let count++

        # 检查csv对应参数是否合法，校验不通过返回1，处理下一行
        checkLineValid "${line}" ${count}
        iRet=$?
        if [ $iRet -ne 0 ]; then
            if [ $iRet -eq 1 ]; then
                let err_task_count++
            fi
            continue
        fi

        path_ori=$(echo ${line} | sed "s/\"//g" | awk -F, '{print $1}')
        user_ori=$(echo ${line} | sed "s/\"//g" | awk -F, '{print $2}')
        advise_threshold=$(echo ${line} | awk -F, '{print $3}' | sed 's/[^0-9]//g')
        soft_threshold=$(echo ${line} | awk -F, '{print $4}' | sed 's/[^0-9]//g')
        soft_threshold_extend_time=$(echo ${line} | awk -F, '{print $5}' | sed 's/[^0-9]//g')
        hard_threshold=$(echo ${line} | awk -F, '{print $6}' | sed 's/[^0-9]//g')
        LOG "[quota_config]path(${path_ori})user(${user_ori})advise(${advise_threshold})soft(${soft_threshold})time(${soft_threshold_extend_time})hard(${hard_threshold})line(${count})"

        # 特殊字符转义
        path=$(echo ${path_ori} | sed -e 's/\\/\\\\/g' -e 's/ /\\s/g' -e 's/?/\\q/g' -e 's/|/\\|/g')
        user=$(echo ${user_ori} | sed -e 's/\\/\\\\/g' -e 's/ /\\s/g')
        LOG "after transform path(${path})user(${user})"

        if [ ${soft_threshold} -ne 0 -a ${soft_threshold_extend_time} -eq 0 ]; then
            echo_red "[ERR]The ${TITLE_TIME} must be greater than 0. Please check param at line(${count}) of csv file."
            LOG "[check_err]The ${TITLE_TIME} is 0 at line(${count}) of csv file."
            let err_task_count++
            continue
        fi

        if [ ${soft_threshold} -ne 0 -a ${soft_threshold} -le ${advise_threshold} ]; then
            echo_red "[ERR]The ${TITLE_ADVISE} must be smaller than the ${TITLE_SOFT}. Please check param at line(${count}) of csv file."
            LOG "[check_err]The ${TITLE_ADVISE}(${advise_threshold}) is not smaller than the ${TITLE_SOFT}(${soft_threshold}) at line(${count}) of csv file."
            let err_task_count++
            continue
        fi

        if [ ${hard_threshold} -ne 0 -a ${hard_threshold} -le ${advise_threshold} ]; then
            echo_red "[ERR]The ${TITLE_ADVISE} must be smaller than the ${TITLE_HARD}. Please check param at line(${count}) of csv file."
            LOG "[check_err]The ${TITLE_ADVISE}(${advise_threshold}) is not smaller than the ${TITLE_HARD}(${hard_threshold}) at line(${count}) of csv file."
            let err_task_count++
            continue
        fi

        if [ ${hard_threshold} -ne 0 -a ${hard_threshold} -le ${soft_threshold} ]; then
            echo_red "[ERR]The ${TITLE_SOFT} must be smaller than the ${TITLE_HARD}. Please check param at line(${count}) of csv file."
            LOG "[check_err]The ${TITLE_SOFT}(${soft_threshold}) is not smaller than the ${TITLE_HARD}(${hard_threshold}) at line(${count}) of csv file."
            let err_task_count++
            continue
        fi

        #检查相应目录下对于用户的配额是否存在，存在打印配额id，走修改流程；没有则配额id为0，走创建流程；报错返回1
        quota_id=$(checkQuotaIsExist $path $user)
        # 查询命令出错
        if [ $? -eq 1 ]; then
            echo_red "[ERR]show quota error for ${TITLE_PATH}(${path_ori}) ${TITLE_USER}(${user_ori}) at line(${count}) of csv file"
            LOG "[show_err]show quota error for ${TITLE_PATH}(${path_ori}) ${TITLE_USER}(${user_ori}) at line(${count}) of csv file."
            let err_task_count++
            continue
        fi
        if [ "${quota_id}" == "0" ]; then
            createQuota ${path} ${user} ${advise_threshold} ${soft_threshold} ${soft_threshold_extend_time} ${hard_threshold} ${count}
        else
            changeQuota ${quota_id} ${advise_threshold} ${soft_threshold} ${soft_threshold_extend_time} ${hard_threshold} ${count}
        fi

    done < ${config_path}

    echo "batch handle quota task is complete."
    LOG "[RESULT]csv_total(${csv_line_total}) valid_total(${valid_total})"
    LOG "[RESULT]success($(echo ${create_count} + ${change_count} | bc -l)) err(${err_task_count})"
    LOG "[RESULT]create(${create_count}) change(${change_count})"
    
    [ -f ${CLI_RESULT} ] && rm -rf ${CLI_RESULT}
}

main