#!/bin/bash -
###############################################
#
#    ivscenterdb_monitor.sh start          Start monitor and db service.
#    ivscenterdb_monitor.sh stop           Stop monitor and db service.
#    ivscenterdb_monitor.sh stop monitor   Stop monitor only.
#
###############################################



PS="/bin/ps"
OS=`uname`

program_dir="/home/center/bin"
old_dir=`pwd`
#获取当前路径的绝对路径
cd "${program_dir}"
program_dir=`pwd`
cd "${old_dir}"
LOG_FILE="${program_dir}/../logs/ivsweb_server_status.log"

ModuleName="center"


###############################################################
#write log to log file
###############################################################
function writelog
{
    nowtime=`date '+%Y-%-m-%d %H:%M:%S'`
    if [ -f ${LOG_FILE} ]; then
        filesize=`stat -c "%s" ${LOG_FILE}`
        if [ 10485760 -le ${filesize} ];then
            mv -f ${LOG_FILE} "${LOG_FILE}_bak"
        fi 
    fi
    echo "[${nowtime}][monitor]$*" >> $LOG_FILE
}

###############################################################
#print help
###############################################################
function printhelp
{
    echo "Usage:"
    echo "    su -c \"ivscenterdb_monitor.sh start &\" - center          Start monitor and db service."
    echo "    su -c \"ivscenterdb_monitor.sh stop\" - center                Stop monitor and db service."
    echo "    su -c \"ivscenterdb_monitor.sh stop monitor\" - center     Stop monitor only."
    return 0
}


###############################################################
#功能说明：获取业务当前状态
#返回值：0 正在运行；1 没有运行
###############################################################
function getcenterdbStatus
{

    ret=`/home/center/bin/pg_ctl status -D /DB_CENTER/data`
    ret1=`echo $ret | grep "server is running" | wc -l`
    ret2=`echo $ret | grep "no server running" | wc -l`
    
    if [ $ret1 -eq 1 ]; then
        return 0
    fi
    
    writelog "The ivscenterdb server is not running."
    touch /alarm/center
    return 1  
}

###############################################################
#功能说明：备份日志，日志文件达到10M时备份，并移除过时日志
#返回值：0 正在运行；1 没有运行
###############################################################
function backuplog
{
    nowtime=`date '+%Y-%-m-%d %H:%M:%S'`
    filesize=`stat -c "%s" "/home/center/logs/ivsweb_server_status.log"`
    if [ 10485760 -le ${filesize} ];then
        mv -f ${LOG_FILE} "${LOG_FILE}_bak"
        cd /home/center/logs
        touch "ivsweb_server_status.log"
        cd -
    fi 
    return 1  
}

###############################################################
#功能说明：获取数据库主备角色
#返回值：0 主机；1 备机；2 啥也不是；
###############################################################
function getcenterdbRole
{
    if [ -e /home/center/bin/primary.role ]; then
        return 0
    elif [ -e /home/center/bin/standby.role ]; then
        return 1
    else
        return 2
    fi 
}


###############################################################
#功能说明：启动ivswebservice
#返回值：0 启动成功；1 启动失败
###############################################################
function startcenterdb
{    
    typeset start_path=${program_dir}
    typeset i=0
    typeset max_count=10

    writelog "Begin to start ivs center server."


    #判断启动脚本所在的目录是否存在
    if [ ! -d ${start_path} ];then
        echo "The install path ${start_path} not a directory or not existed."
        writelog "The install path ${start_path} not a directory or not existed."
        return 1
    fi

    cd ${program_dir}
    
    getcenterdbRole
    CDBRole=$?
    if [ ${CDBRole} -eq 0 ]; then
        /home/center/bin/gs_ctl start -M primary -D /DB_CENTER/data
        echo "Start center db as primary"
        writelog "Start center db as primary"
    elif [ ${CDBRole} -eq 1 ]; then
        /home/center/bin/gs_ctl start -M standby -D /DB_CENTER/data
        echo "Start center db as standby"
        writelog "Start center db as standby"
        /home/center/bin/gs_ctl build -b full -M standby -D /DB_CENTER/data
        echo "Build slave center db"
        writelog "Build slave center db"
    else
        /home/center/bin/gs_ctl start -D /DB_CENTER/data
        echo "center db role file not found, start center db in normal mode"
        writelog "center db role file not found, start center db in normal mode"
    fi
    
    while [ $i -lt ${max_count} ]
    do
        sleep 5
        getcenterdbStatus
        if [ $? -eq 0 ];then
            break
        fi

        ((i=i+1))
    done

    if [ $i -ge ${max_count} ];then
        writelog "The ivs center monitor start service failed."
        return 1
    else
        writelog "The ivs center monitor start service successful."
        return 0
    fi
}
###############################################################
#功能说明：停止ivsdbservice
#返回值：
###############################################################
function stopcenterdb
{
    writelog "Begin to stop center postgres service process."
    while true
    do
        PGPID=`ps -fu center|grep -v "grep"|grep "postgres -D"|awk '{print $2}'`
        if [ "x$PGPID" = "x" ]; then
            PMPID=`ps -fu center|grep -v "grep"|grep "gaussdb -D"|awk '{print $2}'`
            if [ "x$PMPID" = "x" ]; then
                writelog "IVS postgres service Process for ${ModuleName} was killed" 
                return 0;
            fi            
        fi
        
        writelog " Trying to kill ${ModuleName} service process ..."
        
        cd ${program_dir}
        /home/center/bin/pg_ctl stop -m fast -D /DB_CENTER/data
        lsof -i:6001|awk '{print $2}'|xargs kill -9
        rm -rf /DB_CENTER/data/gaussdb.pid
        sleep 5
    done
}
######################################################################
#功能说明：启动监控ivsdbsvr脚本。每10秒查询一次状态，如果service已经
#          停止则尝试拉起3次，如果3次都失败则返回100，并停止监控
#返回值：100 拉起db service 3次失败；
#######################################################################
function startMonitor
{
    typeset count=0
    typeset max_count=300
    typeset proc_num=0
    typeset tmp_result="${program_dir}/result.tmp"
    typeset curr_pid=$$
    typeset monitor_cunt=0

    writelog "Begin to start ivs postgres monitor process."

    while [ ${monitor_cunt} -lt ${max_count} ]
    do
        ${PS} -ef |grep -v "grep" |grep -v "${curr_pid}"|grep "${ModuleName}"|grep -c "ivscenterdb_monitor.sh" > "${tmp_result}"
        proc_num=`cat ${tmp_result}`
        rm -rf "${tmp_result}"
        
        ###判断是否已经启动监控进程，如果已经启动则直接返回
        if [ ${proc_num} -le 0 ];then
            break
        fi
        ((monitor_cunt+=1))
    done

    if [ ${monitor_cunt} -ge ${max_count} ];then
        writelog "The ivs postgres monitor already start."
        return 0
    fi
    
    writelog "Start center postgres monitor successfully."
    
    #循环判断db service状态，如果service已经停止则尝试拉起3次
    while [ ${count} -lt ${max_count} ]
    do

        #日志到达10M时备份日志
        backuplog
        
        getcenterdbStatus
        if [ $? -ne 0 ];then
            writelog "The center postgres server is not running,try to start cneter postgres server."
            stopcenterdb
            startcenterdb
            getcenterdbStatus
            if [ $? -ne 0 ];then
                ((count+=1))
            else
                count=0
            fi
        else
            count=0
        fi

        sleep 10
    done
        
    writelog "The center postgres server is not running,try ${max_count} number to start ivs postgres server failed."
    return 100    
}

###############################################################
#功能说明：停止ivsdb监控脚本
#返回值：
###############################################################
function stopMonitor
{
    typeset pid_list=$$
    typeset curr_pid=$$
    typeset proc_num=0
    typeset tmp_result="${program_dir}/result.tmp"
    
    writelog "Begin to stop ivs postgres monitor process."
    ${PS} -ef |grep -v "grep"|grep -v "${curr_pid}"|grep "${ModuleName}"|grep -c "ivscenterdb_monitor.sh" > "${tmp_result}"
    proc_num=`cat ${tmp_result}`
    rm -rf "${tmp_result}"
    
    if [ ${proc_num} -gt 0 ];then
        ${PS} -ef |grep -v "grep"|grep -v "${curr_pid}"|grep "${ModuleName}"|grep "ivscenterdb_monitor.sh" | awk '{print $2}' > "${tmp_result}"
        pid_list=`cat "${tmp_result}"`
        rm -rf "${tmp_result}"

        for pid in ${pid_list}
        do
            if [ "X${pid}" != "X${curr_pid}" ];then
                kill -9 ${pid} >/dev/null 2>&1
            fi
        done
        
        writelog "Stop center postgres monitor process, kill process ${pid_list}, current process id is ${curr_pid}."
    else
        writelog "No center postgres monitor process run."
        return 0
    fi
}

function main
{
    typeset operate_type=$1
    typeset install_path=${program_dir}
    
    if [ $# -eq 1 -a $1 = "start" ];then
        operate_type="start"
    elif [ $# -eq 1 -a $1 = "stop" ];then
        operate_type="stop"
    elif [ $# -eq 2 -a $1 = "stop" -a $2 = "monitor" ];then
        operate_type="stopmonitor"
    else
        printhelp
        return 1
    fi
    
    if [ "X${operate_type}" = "Xstart" ];then
        startMonitor
    elif [ "X${operate_type}" = "Xstop" ];then
        stopMonitor
        stopcenterdb
    elif [ "X${operate_type}" = "Xstopmonitor" ];then
        stopMonitor
    fi
}

#本脚本只能用center用户运行，否则会出问题！！！
user=`whoami`
if [ "X${user}" != "Xcenter" ];then
    echo "This Script Must Be Runned By Center User!"
    
    printhelp
    exit 1
fi

main $@
     
