#!/bin/bash
#
# Description: 框架集群脚本公共文件 
# Athor: 肖阳阳
# Last Modified: 2021-9-17
# Revision: v0.1

source ./patch_modules.sh
source ./common/InfoStreamDs_1.0_Patch_Common

SSH_OUTPUT_LOG="/home/logs/ssh_output.log"
SSH_INSTALL_LOG="/home/logs/ssh_install.log"
SSHPASS_EXE="sshpass"
# ssh exits with the exit status of the remote command or with 255 if an error occurred
SSH_ERROR=255

readonly E_OK=0
readonly E_ERROR=1

readonly node_types=(
    mmu 
    vpu
)

mmu_master_ip=""
mmu_slave_ip=""
vpu_ips=()
ssh_username=""
ssh_password=""
remote_patch_parent_dir="/root"

#
# 检查node是否合法, 以及模块是否属于node
# 
function check_node_and_modules()
{
    local node="$1"
    shift
    local units=("$@")

    [ ${#units[*]} -eq 0 ] && return 1
    if [ "$node" != "mmu" ] &&  [ "$node" != "vpu" ] && [ "$node" != "all" ]; then
        return 1
    fi

    if [ "${units[0]}" == "all" ]; then
        return 0
    fi

    if [ "$node" == "all" ]; then
        for unit in "${units[@]}"; do
            grep -q "$unit" <<< "${common_modules[*]} ${mmu_modules[*]} ${vpu_modules[*]}"
            [ $? -ne 0 ] && return 1
        done
        return 0
    fi
    
    if [ "$node" == "mmu" ]; then
        for unit in "${units[@]}"; do
            grep -q "$unit" <<< "${common_modules[*]} ${mmu_modules[*]}"
            [ $? -ne 0 ] && return 1
        done
        return 0
    fi

    if [ "$node" == "vpu" ]; then
        for unit in "${units[@]}"; do
            grep -q "$unit" <<< "${common_modules[*]} ${vpu_modules[*]}"
            [ $? -ne 0 ] && return 1
        done
        return 0
    fi 

    return 1
}

function check_node()
{
    local node="$1"
    if [ "$node" == "mmu" ] || [ "$node" == "vpu" ] || [ "$node" == "all" ] ; then
        return 0
    fi
    return 1
}

function get_node_all_modules()
{
    local node="$1"
    if [ "$node" == "mmu" ]; then
        echo "${common_modules[*]} ${mmu_modules[*]}"
    fi
    if [ "$node" == "vpu" ]; then
        echo "${common_modules[*]} ${vpu_modules[*]}"
    fi
    echo ""
}

function query_module_dir()
{
    local module="$1"
    if [ "$module" == "common_config" ]; then
        echo "${module^^}"
    else
        echo "IVS_${module^^}"
    fi
}

#
# 返回值同ssh返回错误码
#
function remote_exec()
{
    local remote_ip="$1"
    shift
    local cmds="$@"
    sed -i "/$remote_ip/d" ${HOME}/.ssh/known_hosts || true
    "$SSHPASS_EXE" -p "$ssh_password" ssh -q -t -o StrictHostKeyChecking=no ${ssh_username}@${remote_ip} "${cmds}" &>"${SSH_OUTPUT_LOG}"
}

# 返回值同ssh返回错误码
#
function remote_exec_cluster()
{
    local cmds="$@"
    local ret=0
    for remote_ip in ${mmu_master_ip} ${mmu_slave_ip} "${vpu_ips[@]}"; do 
        sed -i "/$remote_ip/d" ${HOME}/.ssh/known_hosts || true
        "$SSHPASS_EXE" -p "$ssh_password" ssh -q -t -o StrictHostKeyChecking=no ${ssh_username}@${remote_ip} "${cmds}" &>"${SSH_OUTPUT_LOG}"
        [ $? -ne 0 ] && ret=1
    done
    return $ret
}
#
# 拷贝文件或目录到集群的某个节点
#
function copy_file_to_remote()
{
    local src="$1"
    local remote_ip="$2"
    local dest="$3"
    local ret=0
    "$SSHPASS_EXE" -p "$ssh_password" scp -o StrictHostKeyChecking=no -r "${src}" "${ssh_username}@${remote_ip}:${dest}" &>"${SSH_OUTPUT_LOG}"
    if [ $? -ne 0 ]; then
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" \
            "scp failed. remote ip: ${remote_ip}, patch dir: ${src}, patch remote path: ${dest}"
        ret=1
    fi
    return $ret    
}

function copy_patch_to_remote()
{
    local remote_ip="$1"
    local ret=0
    "$SSHPASS_EXE" -p "$ssh_password" scp -o StrictHostKeyChecking=no -r "${PATCH_TOP_DIR}" "${ssh_username}@${remote_ip}:${remote_patch_parent_dir}" &>"${SSH_OUTPUT_LOG}"
    if [ $? -ne 0 ]; then
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" \
            "scp failed. remote ip: ${remote_ip}, patch dir: ${PATCH_TOP_DIR}, patch remote path: ${remote_patch_parent_dir}"
        ret=1
    fi
    return $ret
}

function copy_patch_to_remote_cluster()
{
    local ret=0
    for remote_ip in ${mmu_master_ip} ${mmu_slave_ip} "${vpu_ips[@]}"; do 
        "$SSHPASS_EXE" -p "$ssh_password" scp -o StrictHostKeyChecking=no -r "${PATCH_TOP_DIR}" "${ssh_username}@${remote_ip}:${remote_patch_parent_dir}" &>"${SSH_OUTPUT_LOG}"
        if [ $? -ne 0 ]; then
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" \
                "scp failed. remote remote_ip: ${remote_ip}, patch dir: ${PATCH_TOP_DIR}, patch remote path: ${remote_patch_parent_dir}"
            ret=1
        fi
    done
    return $ret
}
#
# 返回值 0 - 已经存在
#       1 - 未被复制
#       2 - 其他错误错误
#       3 - patch目录存在，单补丁是旧补丁
function check_patch_has_been_copied()
{
    local remote_ip="$1"
    local ret=0
    remote_exec "$remote_ip" "test -d ${remote_patch_parent_dir}/${PATCH_DIR}"
    ret=$?
    if [ $ret -eq $SSH_ERROR ]; then
        log_and_print_console "ssh execute cmds failed: ssh related error, remote: ${remote_ip}"
        log_file ${SSH_OUTPUT_LOG}
        ret=2
    elif [ $ret -ne 0 ]; then
        ret=1
    fi
    
    [ $ret -ne 0 ] && return $ret

    remote_exec "$remote_ip" "grep -q ${VERSION} ${remote_patch_parent_dir}/${PATCH_DIR}/VERSION"
    ret=$?
    if [ $ret -eq $SSH_ERROR ]; then
        log_and_print_console "ssh execute cmds failed: ssh related error, remote: ${remote_ip}"
        log_file ${SSH_OUTPUT_LOG}
        ret=2
    elif [ $ret -ne 0 ]; then
        ret=3
    fi      
    return $ret
}

# FIXME: 返回值应该是列表, 记录每个节点的补丁是否存在信息
function check_patch_has_been_copied_cluster()
{
    local ret=0
    for remote_ip in ${mmu_master} ${mmu_slave} "${vpus[@]}"; do 
        remote_exec "$remote_ip" "ls ${remote_patch_parent_dir} | grep -q "${PATCH_DIR}""
        ret=$?
        if [ $ret -eq $SSH_ERROR ]; then
            log_and_print_console "ssh execute cmds failed: ssh related error, remote: ${remote_ip}"
            log_file ${SSH_OUTPUT_LOG}
            ret=1
        elif [ $ret -ne 0 ]; then
            ret=1
        fi
        remote_exec "$remote_ip" "grep -q ${VERSION} ${remote_patch_parent_dir}/VERSION"
        ret=$?
        if [ $ret -eq $SSH_ERROR ]; then
            log_and_print_console "ssh execute cmds failed: ssh related error, remote: ${remote_ip}"
            log_file ${SSH_OUTPUT_LOG}
            ret=1
        elif [ $ret -ne 0 ]; then
            ret=1
        fi        
    done
    return $ret
}

function easy_remote_exec()
{
    local remote_ip="$1"
    shift
    local cmds="$@"
    local ret=0

    check_patch_has_been_copied "$remote_ip"
    ret=$?
    if [ $ret -eq 1 ]; then
        # 补丁文件未拷贝到remote host
        if ! copy_patch_to_remote "$remote_ip"; then
            return 1
        fi
    elif [ $ret -eq 2 ]; then
        # ssh连接或者其他未知错误
        return 1
    elif [ $ret -eq 3 ]; then
        # 补丁是旧补丁, 要先把旧补录删除掉
        remote_exec "$remote_ip" "cd ${remote_patch_parent_dir}; rm -fr ./${PATCH_DIR}"
        # 补丁文件未拷贝到remote host
        if ! copy_patch_to_remote "$remote_ip"; then
            return 1
        fi        
    fi

    if ! remote_exec "$remote_ip" $cmds; then
        return 1
    fi
    return 0
}

function extract_single_value_type()
{
    local var="$1"
    local file="$2"
    local value
    value=$(grep -a "$var" "$file" | awk -F ':' '{print $2}')
    value="$(echo -n "${value//[[:space:]]/}")"
    echo -n "$value"
}

function extract_multi_value_type()
{
    local var="$1"
    local file="$2"
    local value
    value=$(grep -a "$var" "$file" | awk -F ':' '{print $2}')
    value="$(echo -n "${value//[[:space:]]/}")"
    value="${value//','/' '}"
    echo -n "$value"
}

function extract_multiline_multi_value_type()
{
    local var="$1"
    local file="$2"
    local value
    OLDIFS=$IFS
    IFS=$'\n'
    
    for line in $(grep -a "$var" "$file"); do
        tmpvalue=$(awk -F ':' '{print $2}' <<< "$line")
        tmpvalue="$(echo -n "${tmpvalue//[[:space:]]/}")"
        tmpvalue="${tmpvalue//','/' '}"
        if [[ "$tmpvalue" == [[:space:]]* ]]; then
            continue
        fi
        value="${value} ${tmpvalue}"
    done
    IFS=$OLDIFS
    echo -n "$value"
}

function read_cluster_info()
{
    local file="$1"
    local var
    local value
    local ret=0

    var='ssh_username'
    value=$(extract_single_value_type "$var" "$file")
    [ -z "$value" ] && ret=1
    ssh_username="$value"

    var='ssh_password'
    value=$(extract_single_value_type "$var" "$file")
    [ -z "$value" ] && ret=1
    ssh_password="$value"

    var='mmu_master_ip'
    value=$(extract_single_value_type "$var" "$file")
    [ -z "$value" ] && ret=1
    mmu_master_ip="$value"

    var='mmu_slave_ip'
    value=$(extract_single_value_type "$var" "$file")
    [ -z "$value" ] && ret=1
    mmu_slave_ip="$value"        

    var='vpu_ips'
    value=$(extract_multiline_multi_value_type "$var" "$file")
    [ -z "$value" ] && ret=1
    vpu_ips=($value)

    return $ret
}

function print_cluster_info()
{
    echo "mmu_master: ${mmu_master_ip}"
    echo "mmu_slave:  ${mmu_slave_ip}"
    echo "vpu:"
    for remote_ip in "${vpu_ips[@]}"; do 
        echo "${remote_ip}" ; 
    done
}

function check_cluster_network()
{
    local ret=0
    for remote_ip in "$mmu_master_ip" "$mmu_slave_ip" "${vpu_ips[@]}"; do
        remote_exec "$remote_ip" "pwd"
        if [ $? -ne 0 ]; then
            echo "ssh to remote host ${remote_ip} failed" 
            ret=1
        fi
    done
    return $ret
}