#!/bin/bash -
#=======================================================================
#
#          FILE:  collect.sh
#
#         USAGE:  ./collect.sh
#
#   DESCRIPTION:  IVS Informations Export program.
#
#       OPTIONS:  ---
#  REQUIREMENTS:  ---
#          BUGS:  ---
#         NOTES:  ---
#        AUTHOR:
#       COMPANY: Huawei Tech. Co., Ltd.
#       CREATED:
#      REVISION:  ---
#=======================================================================

########################################################################
# Environment Settings.
########################################################################
#Conf Path

if test -x /opt/SmartData/bin/smio_watch.sh; then
    HAS_SAFE_VIDEO=true
else
    HAS_SAFE_VIDEO=false
fi

PCGTomcatPath=/home/ivs_pcg/mss-apache-tomcat

DBHOME=/DB_DATA/data
SCUHOME=/home/ivs_scu
SMUHOME=/home/ivs_smu
MUHOME=/home/ivs_mu
DCGHOME=/home/ivs_dcg
IMGUHOME=/home/ivs_imgu
PCGHOME=/home/ivs_pcg/mss-apache-tomcat
OMUHOME=/home/ivs_omu
OMUPORTALHOME=/home/ivs_omu_portal
MAUQDHOME=/home/ivs_mau_qd
MTUHOME=/home/ivs_mtu/vau
HACSHOME=/opt/hacs
UOAHOME=/home/uoausr

BackLogDir=`mount | grep nfs | grep mnt |  awk -F' ' '{ print $3  }'`
hostip=$(/sbin/ifconfig | grep inet | head -n1 | awk -F" " '{print $2}')
MUBackLogDir=$BackLogDir"/ivs_mu_bak/"$hostip                                  #MU日志备份路径
DCGBackLogDir=$BackLogDir"/ivs_dcg_bak/"$hostip                                #DCG日志备份路径
IMGUBackLogDir=$BackLogDir"/ivs_imgu_bak/"$hostip                              #IMGU日志备份路径
MAUQDBackLogDir=$BackLogDir"/ivs_mauqd_bak/"$hostip                           #MAU_QD日志备份路径
OMUBackLogDir=$BackLogDir"/ivs_omu_bak/"$hostip                                #OMU日志备份路径
PCGBackLogDir=$BackLogDir"/ivs_pcg_bak/"$hostip                                #PCG日志备份路径
SCUBackLogDir=$BackLogDir"/ivs_scu_bak/"$hostip                                #SCU日志备份路径
SMUBackLogDir=$BackLogDir"/ivs_smu_bak/"$hostip                                #SMU日志备份路径
CMUBackLogDir=$BackLogDir"/ivs_cmu_bak/"$hostip                                #CMU日志备份路径
CSRBackLogDir=$BackLogDir"/ivs_csr_bak/"$hostip                                #CSR日志备份路径
DBCENTERBackLogDir=$BackLogDir"/CenterDB"                      #CenterDB日志备份路径
ZookeeperBackLogDir=$BackLogDir"/zookeeper"                    #CMU日志备份路径
UoaBackLogDir="/Backup/ivslog/UOA"                                #UOA日志备份路径

DownloadMaxsize=200
MUDownloadMaxsize=$DownloadMaxsize                           #MU模块允许下载的最大备份日志大小
SCUDownloadMaxsize=$DownloadMaxsize                          #SCU模块允许下载的最大备份日志大小
DCGDownloadMaxsize=$DownloadMaxsize                          #DCG模块允许下载的最大备份日志大小
IMGUDownloadMaxsize=$DownloadMaxsize                         #IMGU模块允许下载的最大备份日志大小
OMUDownloadMaxsize=$DownloadMaxsize                          #OMU模块允许下载的最大备份日志大小
SMUDownloadMaxsize=$DownloadMaxsize                          #SMU模块允许下载的最大备份日志大小
PCGDownloadMaxsize=$DownloadMaxsize                          #PCG模块允许下载的最大备份日志大小
MAUQDDownloadMaxsize=$DownloadMaxsize                        #MAUQD模块允许下载的最大备份日志大小
ClusterDownloadMaxsize=$DownloadMaxsize                      #集群相关模块允许下载的最大备份日志大小
UOADownloadMaxsize=$DownloadMaxsize                      #UOA相关模块允许下载的最大备份日志大小

IVSTOOLHOME=/home/ivstool/bin                                   #IVSTOOL Bin路径
Hostname=`hostname`
NowTime=`date +%Y%m%d%H%M%S`                                    #当前时间
ExportTime=$NowTime                                             #导出信息时间
CollectDir="/opt/FileServer/"                                                  #收集压缩包存放目录
ExportDir=$CollectDir"IVS_INF_"$1"_"$hostip"_"$NowTime        #导出信息文件路径

DBExportDir=$ExportDir"/postgres"                               #数据库信息导出临时路径
SCUExportDir=$ExportDir"/SCU"                                   #SCU信息导出临时路径
SMUExportDir=$ExportDir"/SMU"                                   #SMU信息导出临时路径
MUExportDir=$ExportDir"/MU"                                     #MU信息导出临时路径
DCGExportDir=$ExportDir"/DCG"                                   #DCG信息导出临时路径
IMGUExportDir=$ExportDir"/IMGU"                                 #IMGU信息导出临时路径
PCGExportDir=$ExportDir"/PCG"                                   #PCG信息导出临时路径
OMUExportDir=$ExportDir"/OMU"                                   #OMU信息导出临时路径
OMUPORTALExportDir=$ExportDir"/OMUPORTAL"                       #OMUPORTAL信息导出临时路径
MAUQDExportDir=$ExportDir"/MAU_QD"                              #MAU QD信息导出临时路径
MTUExportDir=$ExportDir"/MTU"                                   #MTU信息导出临时路径
PORTALExportDir=$ExportDir"/PORTAL"                             #PORTAL信息导出临时路径
HACSExportDir=$ExportDir"/HACS"                                 #HACS信息导出临时路径
SYSExportDir=$ExportDir"/SYS"                                   #SYS信息导出临时路径
UOAExportDir=$ExportDir"/UOA"                                   #UOA信息导出临时路径

#----start---- 新增的集群故障收集 ----start----


DBCENTERHOME=/DB_CENTER/data
CMUHOME=/home/ivs_cmu
CSRHOME=/home/ivs_csr
GSRHOME=/home/ivs_gsr

DBCENTERExportDir=$ExportDir"/center"        #中心数据库信息导出临时路径
CMUExportDir=$ExportDir"/CMU"                #CMU信息导出临时路径
CSRExportDir=$ExportDir"/CSR"                #CSR信息导出临时路径
GSRExportDir=$ExportDir"/GSR"                #GSR信息导出临时路径
SUSEExportDir=$ExportDir"/SUSE"              #GSR信息导出临时路径
ZookeeperExportDir=$ExportDir"/zookeeper"    #zookeeper信息导出临时路径

#----end---- 新增的集群故障收集 ----end----

#===  FUNCTION  ================================================================
#          NAME:  _RunAsRootUser
#   DESCRIPTION:  检查执行用户是否为root
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function _RunAsRootUser ()
{
    _WHOAMI='/usr/bin/whoami'
    if [ -x "${_WHOAMI}" ]; then
        _ruser=$(${_WHOAMI} 2>/dev/null)
        if [ "-root" != "-${_ruser}" ]; then
            echo "ERROR! Please execute the application as root. CurrentUser=${_ruser}"
            exit 1
        fi
    else
        _ruid=$(id -u 2>/dev/null)
        if [ "-0" != "-${_ruid}" ]; then
            echo "ERROR! Please execute the application as root. CurrentUserID=${_ruid}"
            exit 1
        fi
    fi
}


#===  FUNCTION  ================================================================
#          NAME:  Usage
#   DESCRIPTION:  脚本提示信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function Usage ()
{
cat <<USAGEOF
    Usage:
        $0 [MODULE ]

    MODULE:
        all
        db
        scu
        smu
        mu
        pcg
        dcg
        imgu
        omu
        portal
        omuportal
        mauqd
        cmu
        hacs
        sys
        suse
        safevideo
        **NOTE**: By default, all the modules as shown in the above were included in MODULE.

    EXAMPLE:
        $0 scu                 Export SCU information;
        $0 suse                Export SUSE information;
        $0 all                 Export ALL information but SUSE;
USAGEOF
}

#===  FUNCTION  ================================================================
#          NAME:  _ParserArgs
#   DESCRIPTION:  解析脚本传入参数
#    PARAMETERS:  Module
#       RETURNS:
#===============================================================================
function _ParserArgs ()
{
    if [ $# -eq 1 ];then
        case $1 in
            'all')
            ExportAll
            ;;
            'db')
            ExportDB
            ;;
            'scu')
            ExportSCU
            ;;
            'smu')
            ExportSMU
            ;;
            'mu')
            ExportMU
            ;;
            'pcg')
            ExportPCG
            ;;
            'dcg')
            ExportDCG
            ;;
            'imgu')
            ExportIMGU
            ;;
            'omu')
            ExportOMU
            ;;
            'omuportal')
            ExportOMUPORTAL
            ;;
            'mauqd')
            ExportMAUQD
            ;;
            'cmu')
            ExportCMU
            ;;
            'hacs')
            ExportHACS
            ;;
            'sys')
            ExportSYS
            ;;
            'suse')
            ExportSUSE
            ;;
            'safevideo')
            ExportSafeVideo
            ;;
            'uoa')
            ExportUOA
            ;;
            *)
            Usage
            exit 1
            ;;
        esac
    else
        Usage
        exit 1
    fi

    return $?
}

#===  FUNCTION  ================================================================
#          NAME:  ChangeStrToUnixTime
#   DESCRIPTION:  将14位数字时间转成unix时间戳
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ChangeStrToUnixTime()
{
    #将时间参数(本地时间)转成Y-m-d H:M:S格式
    TempTime=$1
    TempTime="${TempTime:0:4}-${TempTime:4:2}-${TempTime:6:2} ${TempTime:8:2}:${TempTime:10:2}:${TempTime:12:2}"

    #转时间戳
    TempTime=`date -d "$TempTime" +%s`
    
    #这里的echo作为函数的返回输出不能删除,且本函数只允许有一处echo命令！
    if [ $? -ne 0 ];then
        echo "1"
    else
        echo "$TempTime"
    fi
}

#===  FUNCTION  ================================================================
#          NAME:  ExportBackLogByTime
#   DESCRIPTION:  按照给定时间范围收集备份日志
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportBackLogByTime()
{
    local Module=$1
    local ModuleBackLogDir=$2
    local ExportTempDir=$3
    local MaxExportTempDirSize=$4
    local MinCollectDirSize=$((150*1024))
    
    #时间参数校验，主要保证是14位数字形式
    FromTime=`echo $StartTime | grep -E "^[0-9]{14}$"`
    ToTime=`echo $EndTime | grep -E "^[0-9]{14}$"`
    
    if [ "$FromTime" == "" ] || [ "$ToTime" == "" ] || [ ! -d $ModuleBackLogDir ] || (($FromTime > $ToTime)) ;then
        return 0
    fi
    
    #传入的最大可收集日志大小小于0则返回2
    if (($MaxExportTempDirSize < 0)) ;then
        return 2
    fi
    
    #如果是dbcenter需要将时间参数转换成unix时间戳 
    if [ "$Module" == "DBCENTER" ];then
        FromTime=`ChangeStrToUnixTime $FromTime`
        echo "FromTime=$FromTime"
        [ "$FromTime" == "1" ] && return 0
        
        ToTime=`ChangeStrToUnixTime $ToTime`
        echo "ToTime=$ToTime"
        [ "$ToTime" == "1" ] && return 0
    fi
    
    mkdir -p $ExportTempDir
    
    echo "$ModuleBackLogDir"
    find $ModuleBackLogDir -name "*.zip" -o -name "*.tar" | grep -E "run_|debug_|security|center" > $ExportDir/backupnum.txt
    echo "start collect backLogs"

    #每次往故障收集目录拷贝100个备份日志，直到所有在时间范围内的日志都已拷贝到临时目录或者临时目录的大小超过规定值
    while true
    do
        local ExportTempDirSize=`du -hsm $ExportTempDir | awk '{print $1}'`
        local CollectDirSize=`df $CollectDir | grep -v Filesystem|awk '{print $4}'`
        
        if [ $ExportTempDirSize -lt $MaxExportTempDirSize  ] && [ $CollectDirSize -gt $MinCollectDirSize ];then
             if [ ! -s $ExportDir/backupnum.txt ];then
                 break
             fi
             for file in $(sed -n '1,100p' $ExportDir/backupnum.txt)
             do
                 fileTime=$(echo "${file##*_}" | awk -F '.' '{print $1}')
                 fileTime=`echo $fileTime | grep -E "^[0-9]*$"`
                 if [ "$fileTime" != "" ] && (($FromTime <= $fileTime)) && (($ToTime >= $fileTime));then
                     cp  $file  $ExportTempDir
                 fi
             done       
             sed -i '1,100d' $ExportDir/backupnum.txt
        else 
             #收集的备份日志大小超过规定值或者临时目录空间不足150M则删除，并返回2
             rm -rf $ExportTempDir
             rm -f $ExportDir/backupnum.txt
             return 2
        fi
    done
    
    #集群相关模块的备份日志收集大小为一总值，每收集一个模块就减去相应的大小
    if [ "$Module" == "CMU" -o "$Module" == "CSR" -o  "$Module" == "DBCENTER" ];then
        ClusterDownloadMaxsize=`expr $ClusterDownloadMaxsize - $ExportTempDirSize` 
    fi
       
    rm -f $ExportDir/backupnum.txt
    return 0
}

#===  FUNCTION  ================================================================
#          NAME:  ExportDB
#   DESCRIPTION:  导出postgres信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportDB()
{
    if [ -d $DBHOME ];then
        echo "Start to collect postgres information..."
        PythonXml db running
        mkdir -p $DBExportDir
        ExecuteCommand "su - postgres -c \"psql -d postgres -c 'SELECT SPCNAME as tablespaceName,PG_SIZE_PRETTY(PG_TABLESPACE_SIZE(SPCNAME)) as tablespaceSize FROM PG_TABLESPACE;'\"" $DBExportDir/tablespace_conn_info.txt
        ExecuteCommand "su - postgres -c \"psql -d postgres -c 'SELECT COUNT(*) as connectionsNum FROM PG_STAT_ACTIVITY;'\"" $DBExportDir/tablespace_conn_info.txt
        
        cp -rf $DBHOME/pg_log $DBExportDir
        cp -rf $DBHOME/postgresql.conf $DBExportDir
        DelSensitiveFile $ExportDir
        
        cd $ExportDir && zip -r "DB_"$ExportTime postgres >/dev/null 2>&1

        if [ -d $DBExportDir ]; then
            rm -rf $DBExportDir
        fi
        echo "Collect postgres information OK."
        PythonXml db ok
    else
        echo "postgres is not installed."
        PythonXml db none
    fi
}

#===  FUNCTION  ================================================================
#          NAME:  ExportSCU
#   DESCRIPTION:  导出SCU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportSCU()
{
    if [ -d $SCUHOME ];then
        echo "Start to collect scu information..."
        PythonXml scu running
        mkdir -p $SCUExportDir
        cp -rf $SCUHOME/log $SCUExportDir
        cp -rf $SCUHOME/config $SCUExportDir
        rm -rf $SCUExportDir/config/deviceparameter
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime "SCU" "$SCUBackLogDir" "$SCUExportDir/log/backup/" "$SCUDownloadMaxsize"
        ret=$?
            
        cd $ExportDir && zip -r "SCU_"$ExportTime SCU >/dev/null 2>&1

        if [ -d $SCUExportDir ]; then
            rm -rf $SCUExportDir
        fi
        echo "Collect scu information OK."
        PythonXml scu ok
        
        return $ret 
    else
        echo "SCU is not installed."
        PythonXml scu none
        return 0
    fi
}   # ----------  end of function ExportSCU  ----------


#===  FUNCTION  ================================================================
#          NAME:  ExportSMU
#   DESCRIPTION:  获取SMU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportSMU()
{
    if [ -d $SMUHOME ];then
        echo "Start to collect smu information..."
        PythonXml smu running
        mkdir -p $SMUExportDir
        cp -rf $SMUHOME/log $SMUExportDir
        cp -rf $SMUHOME/config $SMUExportDir
        rm -rf $SMUExportDir/config/isap
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime "SMU" "$SMUBackLogDir" "$SMUExportDir/log/backup/" "$SMUDownloadMaxsize"
        ret=$?
        
        cd $ExportDir && zip -r "SMU_"$ExportTime SMU >/dev/null 2>&1
        if [ -d $SMUExportDir ]; then
            rm -rf $SMUExportDir
        fi
        echo "Collect smu information OK."
        PythonXml smu ok       
        return $ret
    else
        echo "SMU is not installed."
        PythonXml smu none
        return 0
    fi
}   # ----------  end of function ExportSMU  ----------


#===  FUNCTION  ================================================================
#          NAME:  ExportMU
#   DESCRIPTION:  获取MU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportMU()
{
    if [ -d $MUHOME ];then
        echo "Start to collect mu information..."
        PythonXml mu running
        mkdir -p $MUExportDir
        cp -rf $MUHOME/log $MUExportDir
        cp -rf $MUHOME/config $MUExportDir
        mkdir -p $MUExportDir/bsm
        cp -rf $MUHOME/bsm/etc $MUExportDir/bsm
        if [ -d /opt/SmartData/log ];then
            mkdir -p $MUExportDir/SmartData
            cp -rf /opt/SmartData/log $MUExportDir/SmartData
        fi
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime  "MU" "$MUBackLogDir" "$MUExportDir/log/backup/"  "$MUDownloadMaxsize"
        ret=$?
      
        cd $ExportDir && zip -r "MU_"$ExportTime MU >/dev/null 2>&1
        if [ -d $MUExportDir ]; then
            rm -rf $MUExportDir
        fi
        echo "Collect mu information OK."
        PythonXml mu ok
        
        return $ret
    else
        echo "MU is not installed."
        PythonXml mu none
        return 0
    fi
}   # ----------  end of function ExportMU  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportDCG
#   DESCRIPTION:  获取DCG信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportDCG()
{
    if [ -d $DCGHOME ];then
        echo "Start to collect dcg information..."
        PythonXml dcg running
        mkdir -p $DCGExportDir
        cp -rf $DCGHOME/log $DCGExportDir
        cp -rf $DCGHOME/config $DCGExportDir
		cp -rf $DCGHOME/plugins/onvif_2.0.0/log $DCGExportDir
		cp -rf $DCGHOME/plugins/hwsdk_2.0.0/log $DCGExportDir
		cp -rf $DCGHOME/plugins/sdkset_2.0.0/log $DCGExportDir
		
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime "DCG" "$DCGBackLogDir" "$DCGExportDir/log/backup/" "$DCGDownloadMaxsize"
        ret=$?       
        cd $ExportDir && zip -r "DCG_"$ExportTime DCG >/dev/null 2>&1

        if [ -d $DCGExportDir ]; then
            rm -rf $DCGExportDir
        fi
        echo "Collect dcg information OK."
        PythonXml dcg ok
        return $ret
    else
        echo "DCG is not installed."
        PythonXml dcg none
        return 0
    fi
}   # ----------  end of function ExportDCG  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportIMGU
#   DESCRIPTION:  获取IMGU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportIMGU()
{
    if [ -d $IMGUHOME ];then
        echo "Start to collect imgu information..."
        PythonXml imgu running
        mkdir -p $IMGUExportDir
        cp -rf $IMGUHOME/log $IMGUExportDir
        cp -rf $IMGUHOME/config $IMGUExportDir
        mkdir -p $IMGUExportDir/bsm
        cp -rf $IMGUHOME/bsm/etc $IMGUExportDir/bsm
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime "IMGU" "$IMGUBackLogDir" "$IMGUExportDir/log/backup/" "$IMGUDownloadMaxsize"
        ret=$?
        cd $ExportDir && zip -r "IMGU_"$ExportTime IMGU >/dev/null 2>&1
        if [ -d $IMGUExportDir ]; then
            rm -rf $IMGUExportDir
        fi
        echo "Collect imgu information OK."
        PythonXml imgu ok
        return $ret
    else
        echo "IMGU is not installed."
        PythonXml imgu none
        return 0
    fi
    
}   # ----------  end of function ExportIMGU  ----------


#===  FUNCTION  ================================================================
#          NAME:  ExportPCG
#   DESCRIPTION:  获取PCG信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportPCG()
{
    if [ -d $PCGHOME ];then
        echo "Start to collect pcg information..."
        PythonXml pcg running
        mkdir -p $PCGExportDir/{log,config/classes,config/WEB-INFconf}

        if [[ $(ls $PCGTomcatPath/logs/) ]]; then
            cp -rf $PCGTomcatPath/logs/* $PCGExportDir/log
        fi


        cp -rf $PCGHOME/conf/* $PCGExportDir/config
        cp -rf $PCGTomcatPath/bin/catalina.sh $PCGExportDir/config
        cp -rf $PCGHOME/webapps/PCG/WEB-INF/conf/* $PCGExportDir/config/WEB-INFconf

        conflist=`ls -la ${PCGHOME}/webapps/PCG/WEB-INF/classes | grep "^-" | awk '{print $NF}'`
        for f in ${conflist}; do
            cp -p $PCGHOME/webapps/PCG/WEB-INF/classes/$f $PCGExportDir/config/classes
        done
        find $ExportDir -name db.properties | xargs rm -rf
        find $ExportDir -name nss_net.xml | xargs rm -rf
        DelSensitiveFile $ExportDir
        ExportBackLogByTime "PCG" "$PCGBackLogDir" "$PCGExportDir/log/backup/" "$PCGDownloadMaxsize"
        ret=$?        
        cd $ExportDir && zip -r "PCG_"$ExportTime PCG >/dev/null 2>&1

        if [ -d $PCGExportDir ]; then
            rm -rf $PCGExportDir
        fi
        echo "Collect pcg information OK."
        PythonXml pcg ok
        return $ret
    else
        echo "PCG is not installed."
        PythonXml pcg none
        return 0
    fi
}   # ----------  end of function ExportPCG  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportOMU
#   DESCRIPTION:  获取OMU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportOMU()
{
    if [ -d $OMUHOME ];then
        echo "Start to collect omu information..."
        PythonXml omu running
        mkdir -p $OMUExportDir

        cp -rf $OMUHOME/log $OMUExportDir
        cp -rf $OMUHOME/config $OMUExportDir
        DelSensitiveFile $ExportDir

        ExportBackLogByTime "OMU" "$OMUBackLogDir" "$OMUExportDir/log/backup/" "$OMUDownloadMaxsize"
        ret=$?    
        cd $ExportDir && zip -r "OMU_"$ExportTime OMU >/dev/null 2>&1

        if [ -d $OMUExportDir ]; then
            rm -rf $OMUExportDir
        fi
        echo "Collect omu information OK."
        PythonXml omu ok
        return $ret
    else
        echo "OMU is not installed."
        PythonXml omu none
        return 0
    fi
}   # ----------  end of function ExportOMU  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportOMUPORTAL
#   DESCRIPTION:  获取OMU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportOMUPORTAL()
{
    if [ -d $OMUPORTALHOME ];then
        echo "Start to collect omuportal information..."
        PythonXml omuportal running
        mkdir -p $OMUPORTALExportDir/{log,config}

        cp -rf $OMUPORTALHOME/logs $OMUPORTALExportDir/log
        mkdir -p $OMUPORTALExportDir/log/Runtime
        cp -rf $OMUPORTALHOME/htdocs/Runtime/Logs $OMUPORTALExportDir/log/Runtime
        cp -rf $OMUPORTALHOME/httpd/conf $OMUPORTALExportDir/config

        DelSensitiveFile $ExportDir
        cd $ExportDir && zip -r "OMUPORTAL_"$ExportTime OMUPORTAL >/dev/null 2>&1

        if [ -d $OMUPORTALExportDir ]; then
            rm -rf $OMUPORTALExportDir
        fi
        echo "Collect omuportal information OK."
        PythonXml omuportal ok
    else
        echo "OMUPORTAL is not installed."
        PythonXml omuportal none
    fi
}

#===  FUNCTION  ================================================================
#          NAME:  ExportMAUQD
#   DESCRIPTION:  获取MAUQD信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportMAUQD()
{
    if [ -d $MAUQDHOME ];then
        echo "Start to collect mauqd information..."
        PythonXml mauqd running

        mkdir -p $MAUQDExportDir
        cp -rf $MAUQDHOME/log $MAUQDExportDir
        cp -rf $MAUQDHOME/config $MAUQDExportDir
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime "MAUQD" "$MAUQDBackLogDir" "$MAUQDExportDir/log/backup/" "$MAUQDDownloadMaxsize"
        ret=$?  
        cd $ExportDir && zip -r "MAU_QD_${ExportTime}" MAU_QD >/dev/null 2>&1
        if [ -d $MAUQDExportDir ]; then
            rm -rf $MAUQDExportDir
        fi
        echo "Collect mau_qd information OK."
        PythonXml mauqd ok
        return $ret
    else
        echo "MAU_QD is not installed."
        PythonXml mauqd none
        return 0
    fi
}   # ----------  end of function ExportMAUQD  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportUOA
#   DESCRIPTION:  获取UOA信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportUOA()
{
    if [ -d $UOAHOME ];then
        echo "Start to collect uoa information..."
        PythonXml uoa running

        mkdir -p $UOAExportDir
        cp -rf $UOAHOME/uoasetup/log $UOAExportDir
        cp -rf $UOAHOME/uoasetup/cfg $UOAExportDir
        
        mkdir -p $UOAExportDir/cfg/system
        cp -f $UOAHOME/uoasetup/cfg/uoa_common.ini $UOAExportDir/cfg
        cp -f $UOAHOME/uoasetup/cfg/system/uoa_system.ini $UOAExportDir/cfg/system
        cp -f $UOAHOME/uoasetup/cfg/system/adapter.acl $UOAExportDir/cfg/system        
        DelSensitiveFile $ExportDir
        
        ExportBackLogByTime "UOA" "$UoaBackLogDir" "$UOAExportDir/log/backup/" "$UOADownloadMaxsize"
        ret=$?  
        cd $ExportDir && zip -r "UOA_${ExportTime}" UOA >/dev/null 2>&1
        if [ -d $UOAExportDir ]; then
            rm -rf $UOAExportDir
        fi
        echo "Collect UOA information OK."
        PythonXml uoa ok
        return $ret
    else
        echo "MAU_QD is not installed."
        PythonXml uoa none
        return 0
    fi
}   # ----------  end of function ExportUOA  ----------



#===  FUNCTION  ================================================================
#          NAME:  ExportMTU
#   DESCRIPTION:  获取MTU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportMTU()
{
    if [ -d $MTUHOME ];then
        echo "Start to collect mtu information..."
        PythonXml mtu running
        mkdir -p $MTUExportDir/{livehttp,transcode,rtsp_gw}

        cp -rf $MTUHOME/livehttp/log $MTUExportDir/livehttp
        cp -rf $MTUHOME/livehttp/conf $MTUExportDir/livehttp

        cp -rf $MTUHOME/transcoder/log $MTUExportDir/transcode
        cp -rf $MTUHOME/transcoder/conf $MTUExportDir/transcode

        cp -rf $MTUHOME/rtsp_gw/log $MTUExportDir/rtsp_gw
        cp -rf $MTUHOME/rtsp_gw/conf $MTUExportDir/rtsp_gw

        cd $ExportDir && zip -r "MTU_"$ExportTime MTU >/dev/null 2>&1

        if [ -d $MTUExportDir ]; then
            rm -rf $MTUExportDir
        fi
        echo "Collect mtu information OK."
        PythonXml mtu ok
    else
        echo "MTU is not installed."
        PythonXml mtu none
    fi
}   # ----------  end of function ExportMTU  ----------


#===  FUNCTION  ================================================================
#          NAME:  ExportSafeVideo
#   DESCRIPTION:  获取SafeVideo信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportSafeVideo()
{
    if $HAS_SAFE_VIDEO; then
        echo "Start to collect SafeVideo information..."
        PythonXml safevideo running

        mkdir -p $ExportDir/SafeVideo

        (
            cd $ExportDir/SafeVideo

            rm -rf Log*_*.tar.gz

            /opt/SmartData/bin/smio_watch.sh tar
            tar -zxhf Log*_*.tar.gz && rm -rf Log*_*.tar.gz
			DelSensitiveFile $ExportDir/SafeVideo/
        )
    
        echo "Collect SafeVideo information OK."
        PythonXml safevideo ok
    else
        echo "SafeVideo is not installed."
        PythonXml safevideo none
    fi
}


#----start---- 新增的集群故障收集 ----start----#

#===  FUNCTION  ================================================================
#          NAME:  ExportCMU
#   DESCRIPTION:  获取CMU信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportCMU()
{  
    ret=0

    if [ -d $CMUHOME ];then
        echo "Start to collect cmu information..."
        PythonXml cmu running

        mkdir -p $CMUExportDir

        cp $CMUHOME/log_zk.log $CMUExportDir >/dev/null 2>&1
        cp -rf $CMUHOME/log $CMUExportDir >/dev/null 2>&1
        cp -rf $CMUHOME/config $CMUExportDir
        ExportCSR
        ExportDBCENTER
        ExportZookeeper 
        DelSensitiveFile $ExportDir
        
        #收集备份日志
        if [ -d $CMUExportDir ];then
            ExportBackLogByTime  "CMU" "$CMUBackLogDir" "$CMUExportDir/log/backup/"  "$ClusterDownloadMaxsize"        
            ret=$?
        fi        
        if [ "$ret" != "2" ] && [ -d $CSRExportDir ];then
            ExportBackLogByTime  "CSR" "$CSRBackLogDir" "$CSRExportDir/log/backup/"  "$ClusterDownloadMaxsize"
            ret=$?
        fi
        if [ "$ret" != "2" ] && [ -d $DBCENTERExportDir ];then
            ExportBackLogByTime  "DBCENTER" "$DBCENTERBackLogDir" "$DBCENTERExportDir/logs/backup/"  "$ClusterDownloadMaxsize"
            ret=$?
        fi

        #如果收集的备份日志总大小超过规定值则删除前面收集的备份日志
        if [ "$ret" == "2" ];then
            rm -rf $CMUExportDir/log/backup
            rm -rf $CSRExportDir/log/backup
            rm -rf $DBCENTERExportDir/logs/backup
            #rm -rf $ZookeeperExportDir/log/backup
        fi
        
        cd $ExportDir && zip -r "CMU_"$ExportTime CMU CSR center zookeeper >/dev/null 2>&1

        rm -rf $CMUExportDir
        rm -rf $CSRExportDir
        rm -rf $DBCENTERExportDir
        rm -rf $ZookeeperExportDir
           
        echo "Collect cmu information OK."
        PythonXml cmu ok
    else
        echo "cmu is not installed."
        PythonXml cmu none
    fi

    return $ret
}   # ----------  end of function ExportCMU  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportZookeeper
#   DESCRIPTION:  获取Zookeeper信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportZookeeper()
{
    if [ -d $SCUHOME/lib/zookeeper ];then
        echo "Start to collect zookeeper information..."
        PythonXml zookeeper running

        mkdir -p $ZookeeperExportDir
        cp -rf $SCUHOME/lib/zookeeper/log $ZookeeperExportDir
        cp $SCUHOME/lib/zookeeper/bin/zkEnv.sh $ZookeeperExportDir
        cp $SCUHOME/lib/zookeeper/conf/zoo.cfg $ZookeeperExportDir
        cp $SCUHOME/config/db.xml $ZookeeperExportDir

        echo "Collect zookeeper information OK."
        PythonXml zookeeper ok
    else
        echo "zookeeper is not installed."
        PythonXml zookeeper none
    fi
}   # ----------  end of function ExportCSR  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportCSR
#   DESCRIPTION:  获取CSR信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportCSR()
{
    if [ -d $CSRHOME ];then
        echo "Start to collect csr information..."
        PythonXml csr running

        mkdir -p $CSRExportDir
        DelSensitiveFile $ExportDir
        cp -rf $CSRHOME/log $CSRExportDir >/dev/null 2>&1
        cp -rf $CSRHOME/config $CSRExportDir
      
        echo "Collect csr information OK."
        PythonXml csr ok
    else
        echo "csr is not installed."
        PythonXml csr none
    fi
}   # ----------  end of function ExportCSR  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportGSR
#   DESCRIPTION:  获取GSR信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportGSR()
{
    if [ -d $GSRHOME ];then
        echo "Start to collect gsr information..."
        PythonXml gsr running

        mkdir -p $GSRExportDir
        DelSensitiveFile $ExportDir
        cp -rf $GSRHOME/log $GSRExportDir >/dev/null 2>&1
        cp -rf $GSRHOME/config $GSRExportDir

        echo "Collect gsr information OK."
        PythonXml gsr ok
    else
        echo "gsr is not installed."
        PythonXml gsr none
    fi
}   # ----------  end of function ExportGSR  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportDBCENTER
#   DESCRIPTION:  导出postgres中心数据库信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportDBCENTER()
{
    if [ -d $DBCENTERHOME ];then
        echo "Start to collect center db information..."
        PythonXml center running

        mkdir -p $DBCENTERExportDir
        ExecuteCommand "su - center -c \"psql -d center -c 'SELECT SPCNAME as tablespaceName,PG_SIZE_PRETTY(PG_TABLESPACE_SIZE(SPCNAME)) as tablespaceSize FROM PG_TABLESPACE;'\"" $DBCENTERExportDir/tablespace_conn_info.txt
        ExecuteCommand "su - center -c \"psql -d center -c 'SELECT COUNT(*) as connectionsNum FROM PG_STAT_ACTIVITY;'\"" $DBCENTERExportDir/tablespace_conn_info.txt

        cp -rf $DBCENTERHOME/pg_log $DBCENTERExportDir
        cp -rf $DBCENTERHOME/postgresql.conf $DBCENTERExportDir
        cp -rf /home/center/logs $DBCENTERExportDir
        cp /usr/local/etc/odbc.ini $DBCENTERExportDir
        
        echo "Collect center db information OK."
        PythonXml center ok
    else
        echo "center db is not installed."
        PythonXml center none
    fi
}

#----end---- 新增的集群故障收集 ----end----#

#===  FUNCTION  ================================================================
#          NAME:  ExportHACS
#   DESCRIPTION:  获取所有信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportHACS ()
{
    if [ -d $HACSHOME ]
    then
        echo "Start to collect hacs information..."
        PythonXml hacs running
        mkdir -p $HACSExportDir/{conf,log}

        ExecuteCommand "/opt/hacs/version.sh" $HACSExportDir/conf/hacs_status.txt
        ExecuteCommand "ps -C corosync" $HACSExportDir/conf/hacs_status.txt
        ExecuteCommand "ps -ef | grep heartbeat | grep -v grep" $HACSExportDir/conf/hacs_status.txt
        ExecuteCommand "/usr/sbin/corosync-cfgtool -s" $HACSExportDir/conf/hacs_status.txt
        ExecuteCommand "/usr/sbin/crm_resource -L" $HACSExportDir/conf/hacs_status.txt
        ExecuteCommand "crm_mon -ft1" $HACSExportDir/conf/hacs_status.txt
        ExecuteCommand "crm configure show" $HACSExportDir/conf/hacs_configure.txt
        ExecuteCommand "/usr/sbin/crm_verify -Vx /var/lib/heartbeat/crm/cib.xml" $HACSExportDir/conf/hacs_configure.txt
        ExecuteCommand "ldd /usr/sbin/crm_mon" $HACSExportDir/conf/hacs_configure.txt

        cp -rf /etc/corosync/corosync.conf $HACSExportDir/conf
        cp -rf /etc/corosync/service.d/pcmk $HACSExportDir/conf
        cp -rf /etc/corosync/uidgid.d/corosync $HACSExportDir/conf
        cp -rf /etc/corosync/authkey $HACSExportDir/conf
        cp -rf /var/lib/heartbeat/crm/cib.xml $HACSExportDir/conf

        if [ -f /proc/drbd ]
        then
            cp -rf /proc/drbd $HACSExportDir/conf
            cp -rf /etc/drbd.conf $HACSExportDir/conf
            cp -rf /etc/drbd.d/r0.res $HACSExportDir/conf
        fi

        mkdir -p $HACSExportDir/log/corosync
        cp -rf /var/log/corosync* $HACSExportDir/log/corosync
        mkdir -p $HACSExportDir/log/hacs
        cp -rf /opt/hacs/log/* $HACSExportDir/log/hacs
        
        DelSensitiveFile $ExportDir
        
        cd $ExportDir && zip -r "HACS_"$ExportTime HACS >/dev/null 2>&1

        if [ -d $HACSExportDir ]; then
            rm -rf $HACSExportDir
        fi

        echo "Collect hacs information OK."
        PythonXml hacs ok
    else
        echo "HACS is not installed."
        PythonXml hacs none
    fi
}   # ----------  end of function ExportHACS  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportSYS
#   DESCRIPTION:  获取系统类信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportSYS()
{
    echo "Start to collect sys information..."
    PythonXml sys running

    mkdir -p $SYSExportDir
    cp -rf /var/log/messages* $SYSExportDir

    ExportCoreInfo $SYSExportDir
    GetDiskInfo $SYSExportDir
    GetNICFlow $SYSExportDir
    GetNetworkSetting $SYSExportDir
    GetNetstatSU $SYSExportDir
    GetSysInfo $SYSExportDir
    GetCronStatus $SYSExportDir
    GetNtpStatus $SYSExportDir
    GetISCSIStatus $SYSExportDir
    GetEnvInfo $SYSExportDir
    GetBPSysInfo $SYSExportDir
    
    DelSensitiveFile $ExportDir
    cd $ExportDir && zip -r "SYS_"$ExportTime SYS >/dev/null 2>&1

    if [ -d $SYSExportDir ]
    then
        rm -rf $SYSExportDir
    fi
    PythonXml sys ok
}   # ----------  end of function ExportSYS  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportAll
#   DESCRIPTION:  获取所有信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportAll()
{
    for model in DB SCU SMU MU IMGU PCG DCG OMU PORTAL OMUPORTAL MAUQD CMU HACS SYS SafeVideo UOA; do
        Export${model}
    done
}   # ----------  end of function ExportAll  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportSUSE
#   DESCRIPTION:  获取所有信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportSUSE ()
{
    echo "Start to install supportutils-1.20-65.1.noarch.rpm."
    PythonXml suse running
    rpm -Uvh ./supportutils-1.20-65.1.noarch.rpm >/dev/null 2>&1

    #收集操作系统信息供SUSE定位系统问题
    supportconfig -A >/dev/null 2>&1

    echo "Start to collect suse information..."
    if [[ $(ls /var/log/ | grep ^nts.*tbz$) ]];then
        mkdir -p $SUSEExportDir
        cp -rf /var/log/nts*.tbz $SUSEExportDir
        cd $ExportDir && zip -r "SUSE_"$ExportTime SUSE >/dev/null 2>&1
        if [ -d $SUSEExportDir ]; then
            rm -rf $SUSEExportDir
        fi
        echo "Collect suse information OK."
        PythonXml suse ok
    else
        echo "suse /var/log/nts*.tbz file does not exist."
        PythonXml suse none
    fi

}   # ----------  end of function ExportAll  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExecuteCommand
#   DESCRIPTION:  执行命令，写入结果文件
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExecuteCommand()
{
    echo "====================Begin==========$1====================" >> $2 2>&1
    echo "# $1:" >> $2 2>&1
    eval $1 >> $2 2>&1
    echo "====================End==========$1====================" >> $2 2>&1
    echo "" >> $2 2>&1
    echo "" >> $2 2>&1
}   # ----------  end of function ExecuteCommand  ----------

#===  FUNCTION  ================================================================
#          NAME:  PythonXml
#   DESCRIPTION:  用Python写入xml文件
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function PythonXml()
{
    python /home/ivstool/collect/xmlhub.py $1 $2
}   # ----------  end of function PythonXml  ----------

#===  FUNCTION  ================================================================
#          NAME:  ExportCoreInfo
#   DESCRIPTION:  获取系统MRU/MDU/DCG的core文件的堆栈信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function ExportCoreInfo()
{
    PythonXml corefiles running
    CoreFileDir=/opt/corefiles
    if [ ! -d $CoreFileDir ]; then
        PythonXml corefiles true
        return 1
    fi
    echo "Start to collect system corefiles information."
    Size=`df $CollectDir | grep -v Filesystem|awk '{print $4}'`
    CoreFileSize=`du -s /opt/corefiles|awk '{print $1}'`
    if [ $CoreFileSize -gt $Size ];then
        echo "The disk size is not enough, please check $CoreFileDir for corefiles!"
        PythonXml corefiles full
        return 0
    else
        cp -rf /opt/corefiles $1
        PythonXml corefiles true
    fi

    echo "Collect system corefiles information OK."
}   # ----------  end of function ExportCoreInfo  ----------

#===  FUNCTION  ================================================================
#          NAME:  GetNICFlow
#   DESCRIPTION:  获取网卡流量，每秒钟检测一次，检测30秒
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetNICFlow()
{
    echo "Start to collect NIC flow information."

    nic_inf=`ifconfig | grep -E "bond|eth" | awk '{print $1}'`

    for n in ${nic_inf}; do
        COUNT=0
        while [ $COUNT -lt 30 ]
        do
            eth=$n
            RXpre=$(cat /proc/net/dev | grep -w $eth | tr : " " | awk '{print $2}')
            TXpre=$(cat /proc/net/dev | grep -w $eth | tr : " " | awk '{print $10}')
            sleep 1
            RXnext=$(cat /proc/net/dev | grep -w $eth | tr : " " | awk '{print $2}')
            TXnext=$(cat /proc/net/dev | grep -w $eth | tr : " " | awk '{print $10}')

            echo -e "`date +%k:%M:%S` RX   TX " >> $1/"nic_flow_"$n".txt" 2>&1

            if [ -z $RXnext -o -z $RXpre ]
            then
                break
            fi

            RX=`expr $RXnext - $RXpre`
            TX=`expr $TXnext - $TXpre`

            if [[ $RX -lt 1024 ]]
            then
                RX="${RX}B/s"
            elif [[ $RX -gt 1048576 ]]
            then
                RX=$(echo $RX | awk '{print $1/1048576 "MB/s"}')
            else
                RX=$(echo $RX | awk '{print $1/1024 "KB/s"}')
            fi

            if [[ $TX -lt 1024 ]]
            then
                TX="${TX}B/s"
            elif [[ $TX -gt 1048576 ]]
            then
                TX=$(echo $TX | awk '{print $1/1048576 "MB/s"}')
            else
                TX=$(echo $TX | awk '{print $1/1024 "KB/s"}')
            fi

            echo -e "$eth \t $RX   $TX " >> $1/"nic_flow_"$n".txt" 2>&1

            COUNT=`expr $COUNT + 1`
        done
    done

    echo "Collect NIC flow information OK."
}   # ----------  end of function GetNICFlow  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetNetworkSetting
#   DESCRIPTION:  获取网卡配置信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetNetworkSetting()
{
    echo "Start to collect network setting information."

    ExecuteCommand "ifconfig " $1/network_setting.txt
    ExecuteCommand "ifconfig -a" $1/network_setting.txt
    ExecuteCommand "route" $1/network_setting.txt
    ExecuteCommand "cat /proc/net/dev" $1/network_setting.txt

    if [ -d /etc/sysconfig/network ];then
        cp -rf /etc/sysconfig/network $1
    fi

    nic_inf=`ifconfig | grep -E "bond|eth" | awk '{print $1}'`

    for n in ${nic_inf}; do
        ExecuteCommand "ethtool ${n}" $1/network_setting.txt
    done

    echo "Collect network setting information OK."
}   # ----------  end of function GetNetworkSetting  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetDiskInfo
#   DESCRIPTION:  获取硬盘状态
#                 1.1 Modify
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetDiskInfo()
{
    echo "Start to collect disk information."

    ExecuteCommand "/sbin/fdisk -l 2>/dev/null | grep Disk" $1/disk_info.txt
    ExecuteCommand "cat /proc/partitions" $1/disk_info.txt
    ExecuteCommand "cat /etc/fstab" $1/disk_info.txt
    ExecuteCommand "/bin/mount" $1/disk_info.txt
    ExecuteCommand "cat /proc/mounts" $1/disk_info.txt
    ExecuteCommand "cat /etc/mtab" $1/disk_info.txt
    ExecuteCommand "df -hT" $1/disk_info.txt
    ExecuteCommand "blkid" $1/disk_info.txt

    echo "====================Begin==========df -h====================" >> $1/disk_info.txt 2>&1
    df -h >> $1/disk_info.txt 2>&1

    disk_status=`df -h | awk '{print $5"&&"$6}' | grep -v Record`

    for dp in ${disk_status} ; do
        used=`echo ${dp} | awk -F% '{print $1}'`
        mounted=`echo ${dp} | awk -F"&&" '{print $2}'`
        if [[ $used -gt 70 ]]; then
            echo "$mounted" | grep -q "Record"
            if [ $? -eq 1 ];
            then
                ExecuteCommand "du -h --max-depth=6 ${mounted}" $1/disk_info.txt
            fi
        fi
    done
    echo "====================End==========df -h====================" >> $1/disk_info.txt 2>&1

    ExecuteCommand "/bin/ls -lR --time-style=long-iso /dev/disk/" $1/disk_info.txt

    echo "" >> $1/disk_info.txt 2>&1
    echo "" >> $1/disk_info.txt 2>&1

    echo "Collect disk information OK."
}   # ----------  end of function GetDiskInfo  ----------



#===  FUNCTION  ================================================================
#          NAME:  GetNetstatSU
#   DESCRIPTION:  获取系统底层协议栈收发报信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetNetstatSU()
{
    echo "Start to collect netstat su information."

    FirstTime=`date +%k:%M:%S`
    echo "====================FirstTime:${FirstTime}====================" >> $1/netstat_su.txt 2>&1
    netstat -su >> $1/netstat_su.txt 2>&1

    sleep 60

    SecondTime=`date +%k:%M:%S`
    echo "====================SecondTime:${SecondTime}====================" >> $1/netstat_su.txt 2>&1
    netstat -su >> $1/netstat_su.txt 2>&1

    echo "Collect netstat su information OK."
}   # ----------  end of function GetNetstatSU  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetCronStatus
#   DESCRIPTION:  获取定时任务状态
#                 1.1 ADD
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetCronStatus()
{
    echo "Start to collect cron information."

    ExecuteCommand "/sbin/chkconfig cron --list" $1/cron.txt
    ExecuteCommand "/etc/init.d/cron status" $1/cron.txt
    ExecuteCommand "cat /etc/crontab" $1/cron.txt

    cronusr=`ls /var/spool/cron/tabs`
    for usr in ${cronusr}
    do
        ExecuteCommand "crontab -l -u ${usr}" $1/cron.txt
    done

    ExecuteCommand "ls -lhR /opt/dbdata/backup" $1/cron.txt
    ExecuteCommand "ls -lhR /home/ivslog/backup" $1/cron.txt

    echo "Collect cron information OK."
}   # ----------  end of function GetCronStatus  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetNtpStatus
#   DESCRIPTION:  获取NTP服务状态
#                 1.1 ADD
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetNtpStatus()
{
    echo "Start to collect ntp information."

    ExecuteCommand "rpm -qa | grep ntp | grep -v yast" $1/ntp.txt
    ExecuteCommand "/sbin/chkconfig ntp --list" $1/ntp.txt
    ExecuteCommand "/etc/init.d/ntp status" $1/ntp.txt
    ExecuteCommand "/usr/sbin/ntpq -p" $1/ntp.txt

    #ExecuteCommand "cat /home/ivstool/bin/sntpRun.sh" $1/ntp.txt
    ExecuteCommand "cat /etc/ntp.conf" $1/ntp.txt

    ntpdriftfile=`cat /etc/ntp.conf | grep driftfile | awk '{print $2}'`
    ExecuteCommand "cat ${ntpdriftfile}" $1/ntp.txt

    ntplogfile=`cat /etc/ntp.conf | grep logfile | awk '{print $2}'`
    ExecuteCommand "cat ${ntplogfile}" $1/ntp.txt

    echo "Collect ntp information OK."
}   # ----------  end of function GetNtpStatus  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetISCSIStatus
#   DESCRIPTION:  获取NTP服务状态
#                 1.1 ADD
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetISCSIStatus()
{
    echo "Start to collect iscsi information."

    ExecuteCommand "rpm -qa | grep iscsitarget" $1/iscsi.txt
    ExecuteCommand "/sbin/chkconfig iscsitarget --list" $1/iscsi.txt
    ExecuteCommand "/etc/init.d/iscsitarget status" $1/iscsi.txt
    ExecuteCommand "/usr/sbin/ietadm --op show" $1/iscsi.txt
    ExecuteCommand "cat /etc/ietd.conf" $1/iscsi.txt
    ExecuteCommand "rpm -qa | grep open-iscsi" $1/iscsi.txt
    ExecuteCommand "/sbin/chkconfig boot.open-iscsi" $1/iscsi.txt
    ExecuteCommand "/sbin/chkconfig open-iscsi --list" $1/iscsi.txt
    ExecuteCommand "/etc/init.d/open-iscsi status" $1/iscsi.txt
    ExecuteCommand "cat /etc/iscsid.conf" $1/iscsi.txt

    echo "Collect iscsi information OK."
}   # ----------  end of function GetISCSIStatus  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetEnvInfo
#   DESCRIPTION:  获取NTP服务状态
#                 1.1 ADD
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetEnvInfo()
{
    echo "Start to collect env information."

    ExecuteCommand "env" $1/env.txt
    ExecuteCommand "cat /etc/profile | grep ^export" $1/env.txt
    EnvDir=`cat /etc/passwd|grep ivs|awk -F ':' '{print $6}'`
    for temp in $EnvDir
    do
        ExecuteCommand "cat $temp/.bashrc" $1/env.txt
    done

    echo "Collect env information OK."
}   # ----------  end of function GetEnvInfo  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetSysInfo
#   DESCRIPTION:  获取操作系统配置信息
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetSysInfo()
{
    echo "Start to collect system config information."

    ExecuteCommand "lsb_release -a" $1/os_version.txt
    ExecuteCommand "/bin/uname -a" $1/os_version.txt
    ExecuteCommand "cat /etc/SuSE-release" $1/os_version.txt

    ExecuteCommand "/bin/date -R" $1/time_info.txt
    ExecuteCommand "hwclock" $1/time_info.txt
    ExecuteCommand "cat /etc/sysconfig/clock" $1/time_info.txt

    ExecuteCommand "cat /proc/cpuinfo" $1/sys_info.txt
    ExecuteCommand "cat /proc/meminfo" $1/sys_info.txt
    ExecuteCommand "free -m" $1/sys_info.txt
    ExecuteCommand "top -bc -n 5 -d 2" $1/sys_info.txt
    ExecuteCommand "sysctl -a" $1/sys_info.txt
    ExecuteCommand "ulimit -a" $1/sys_info.txt

    ExecuteCommand "ps -ef" $1/ps_info.txt

    ExecuteCommand "iostat -x 2 15" $1/disk_io.txt

    #1.1 ADD
    ExecuteCommand "/sbin/chkconfig --list" $1/chkconfig.txt
    ExecuteCommand "/bin/ls -lR --time-style=long-iso /etc/init.d" $1/chkconfig.txt

    echo "Collect system config information OK."
}   # ----------  end of function GetSysInfo  ----------


#===  FUNCTION  ================================================================
#          NAME:  GetBPSysInfo
#   DESCRIPTION:  收集BP周期性检测系统状态结果，当前至少包含内存使用情况，CPU使用情况、IO使用情况
#    PARAMETERS:
#       RETURNS:
#===============================================================================
function GetBPSysInfo()
{
    filePath=$1
    if [ -d /var/log/statistics ];then
        echo "Start to collect BP sysinfo."
        cp -rf /var/log/statistics ${filePath}
        echo "Collect BP sysinfo OK."
    else
        echo "/var/log/statistics not exists."
    fi
}

#删除业务模块敏感数据（C）,不包含pcg
function DelSensitiveFile()
{
    filePath=$1
    # find $filePath -name db.xml | xargs rm -rf
    # find $filePath -name net.xml | xargs rm -rf
    # find $filePath -name log.xml | xargs rm -rf
    # find $filePath -name key.xml | xargs rm -rf
    # find $filePath -name key.ini | xargs rm -rf
    
    # find $filePath -name server.key|xargs rm -f
    # find $filePath -name userData|xargs rm -rf
    # find $filePath -name oracle|xargs rm -rf
    # sed -i "s/pass\|pwd\|key\|iv/@@@@@/ig" `find $filePath -name "*.*" |xargs grep -ilE 'pass|pwd|key|iv'`
    # sed -i "/@@@@@/d"  `find $filePath -name "*.*" |xargs grep -ilE '@@@@@'`
}

########################################################################
# Entry;
########################################################################

#20200413 delete by t00533446 :The omu_monitor process is started by the ivsoper user, and there is no root authority when collecting fault results.
#_RunAsRootUser

StartTime=$2                                 #备份日志导出开始时间
EndTime=$3                                   #备份日志导出结束时间

_ParserArgs $1
_ret=$?

#返回值为2说明收集的日志大小超过阈值


#Hostname=`hostname`
Hostname=$hostip
ExportFileName=IVS_INF_$(echo $1 | tr [a-z] [A-Z])
CurrentExportDir="IVS_INF_"$1"_"$Hostname"_"$NowTime

if [ -f $CollectDir$ExportFileName".zip" ]; then
#如果压缩包已存在则删除
    rm $CollectDir$ExportFileName".zip"
fi

if [ -d $ExportDir ];then
    echo "Start to compress information file..."
    cd $CollectDir && zip -r $ExportFileName $CurrentExportDir >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        chmod 644 $CollectDir$ExportFileName".zip"
        echo "Compress information file OK."       
        if [ "${_ret}" == "2" ];then
           echo "backLogs collected are too large, please change the time selected"
           PythonXml $1 limit
        else
           PythonXml $1 true
        fi
    else
        echo false
        PythonXml $1 false
    fi
    rm -rf $ExportDir
else
    echo null
    PythonXml $1 null
fi

exit ${_ret}
