#!/bin/sh

source ./common.sh

THIS_SCRIPT="$(basename $(pwd))/$(basename $0)"
THIS_PATCH_DIR="${PATCH_TOP_DIR}/IVS_PCG"
patch_name="PCG patch"

    
function show_usage
{
    echo "usage: sh $0 [mmu_master | mmu_slave]"
    echo "       sh $0 [mmu_master | mmu_slave] status"
}

# 输入参数检查
function check_param
{
    if [ $# -lt 1 ];then
        return 1
    fi
    return 0
}


function backup()
{
    if [ ! -f  "${BACKUP_DIR}/${PCG_BACKUP_ARCHIVE}" ]; then
        cd "${PCG_DIR}"
        tar czf "${BACKUP_DIR}/${PCG_BACKUP_ARCHIVE}" \
            $(find . -name '*.class') \
            "$PCG_CONFIG_FILE"  \
            "/home/ivs_pcg/hacs/hacs_stop_pcg.sh" \
            "/home/ivs_pcg/hacs/hacs_start_pcg.sh"

        if [ $? -ne 0 ]; then
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "backup pcg files failed"
			cd - 1>/dev/null
            return 1
        fi
		cd - 1>/dev/null
        return 0
    fi
}

function change_bin_attr()
{
    local mybins=("$@")
    for mybin in "${mybins[@]}"; do
        chown ivspcg:ivs "$mybin"
        chmod 700 "$mybin"
    done
}

function install_scripts()
{
    cp -f ./patch_script/hacs_start_pcg.sh "/home/ivs_pcg/hacs/hacs_start_pcg.sh"
    cp -f ./patch_script/hacs_stop_pcg.sh  "/home/ivs_pcg/hacs/hacs_stop_pcg.sh"
    cp -f ./patch_script/on_pcg_start.sh   "/home/ivs_pcg/hacs/on_pcg_start.sh"
    cp -f ./patch_script/on_pcg_stop.sh    "/home/ivs_pcg/hacs/on_pcg_stop.sh"
    cp -f ./patch_script/InfoStreamDs_6.0_Crontab_Restart_PCG.sh /home/ivstool/bin/
    chmod 500 /home/ivstool/bin/InfoStreamDs_6.0_Crontab_Restart_PCG.sh
    change_bin_attr "/home/ivs_pcg/hacs/hacs_start_pcg.sh" \
        "/home/ivs_pcg/hacs/hacs_stop_pcg.sh" "/home/ivs_pcg/hacs/on_pcg_start.sh" \
        "/home/ivs_pcg/hacs/on_pcg_stop.sh"
}

function update_config()
{
    Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "begin to update config"

    update_pcg_properties_file 'pcg.media.rtsp.clearRepeat' 'false' 'is clear media rtsp when received request has same subject(true: clear, false: not clear)'

    update_pcg_properties_file 'pcg.eachExdomain.mediaSession.control' '1' 'eachExdomain.mediaSession.control. 0:no, 1:yes'

    update_pcg_properties_file 'pcg.connectCode.force.check'  'true' 'connectCode force check flag'

    update_pcg_properties_file 'pcg.rtspSession.timeout'  '5'  'timeout to refresh rtspSession'

    update_pcg_properties_file 'pcg.rtspSession.cleanOldest'  'true' 'enable auto clean oldest rtsp session when session pool is full'

    update_pcg_properties_file 'pcg.rtspSession.enableAutoClean'  'false' 'enable auto clean rtsp session' 

    update_pcg_properties_file 'pcg.rtspSession.autoCleanThreshold'  '800' 'rtsp session auto clean threshold'

    update_pcg_properties_file 'pcg.rtspSession.autoCleanStartTime'  '2' 'rtsp session auto clean start time(hour) in one day'

    update_pcg_properties_file 'pcg.selfcure.switch'  'false' 'pcg self-cure switch,default value [true] is open self-cure check'

    update_pcg_properties_file 'pcg.selfcure.queue.threshold'  '0.9' 'turn on self-cure check conditions -- queue length threshold'

    update_pcg_properties_file 'pcg.task.execution.timeout'  '5' 'Task execution timeout,defalut value [5 minutes]'

    update_pcg_properties_file 'pcg.reboot.time'  '2000-1-1 12:00:00' 'The time of the last reboot'

    update_pcg_properties_file 'pcg.reboot.interval'  '3'  'Self-cure dormant period,in days,defalut value 3 days'

    update_pcg_properties_file 'pcg.selfcure.test.number'  '5' 'the number of test to reboot pcg'

    Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "update config succeed"    
}

function install()
{
	local mmu_type="$1"

    backup
    [ $? -ne 0 ] && return 1
	
    if [[ "$mmu_type" == "mmu_master" ]]; then
        stop_process
    fi
	
    # 拷贝脚本
    install_scripts

    # 替换class文件
    find "./WEB-INF" -name '*.class' -exec cp -f --parents {} ${PCG_DIR} \;

    cd "${PCG_DIR}"
    for f in $(find . -name '*.class'); do
        chmod 700 "$f" && chown ivspcg:ivs "$f"
    done

    update_config

    if [[ "$mmu_type" == "mmu_master" ]]; then
        # 将pcg重启脚本加入到crontab中
        cd /home/ivstool/bin
        sh ./InfoStreamDs_6.0_Crontab_Restart_PCG.sh        
        # pcg启动耗时太长，放后台启动
        start_process &
    fi

    Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "install succeed"
    return 0
}


function install_patch
{
    local ret=0
    local node="$1"

    Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "begin install pcg patch, node: ${node}"

    # MMU主节点安装补丁
    if [ "$node" == "mmu_master" ]; then
        # 安装PCG补丁
        install mmu_master
        ret=$?
    # MMU备节点安装补丁
    elif [ "$node" == "mmu_slave" ]; then
        # 安装PCG补丁
        install mmu_slave
        ret=$?
    else
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "invalid node type ${node}"
        ret=1
    fi
    return $ret
}

function list_installed_patch()
{
    local bin_file_name="MediaServerServiceImpl.class"
    local bin_file_path="/home/ivs_pcg/mss-apache-tomcat/webapps/PCG/WEB-INF/classes/com/huawei/pcg/service/media/impl/MediaServerServiceImpl.class"
    local module="pcg"
    local expect_md5="$(grep -a ${bin_file_name} ./md5sum.txt | awk -F ' ' '{print $1}')"
    local actual_md5="$(md5sum ${bin_file_path} | awk -F ' ' '{print $1}')"
    if [ "$actual_md5" != "$expect_md5" ]; then
        report_module_install_status "${module}" "$STATUS_NOT_INSTALL"
        return $STATUS_NOT_INSTALL
    else
        report_module_install_status "${module}" "$STATUS_INSTALLED"
        return $STATUS_INSTALLED
    fi
}

function main()
{
    local ret=0

    check_param "$@"
    if [ $? -ne 0 ]; then
        show_usage
        return 1
    fi
    Log "${THIS_SCRIPT}:${LINENO}" "[Version:${PACKAGE_NAME}]"
    # 列出已安装的补丁情况
    if [ "$2" == "$OP_LIST_INSTALLED" ]; then
        list_installed_patch "$@"
        return 0
    fi

    echo "InfoStreamDs_6.0_Full_Patch_Package_Pcg install start"
    list_installed_patch
    if [ $? -eq "$STATUS_INSTALLED" ]; then
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "patch already installed, skip it"
        exit 0 
    fi

    install_patch "$@"
    if [ $? -ne 0 ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "install PCG patch failed"
        report_error $patch_name
        ret=1
    fi
    echo "InfoStreamDs_6.0_Full_Patch_Package_Pcg install end"
    return $ret
}

main "$@"