#!/bin/sh
#
# Description: 单机补丁安装脚本 
# Athor: 肖阳阳
# Last Modified: 2021-9-17
# Revision: v0.1

source ./common/InfoStreamDs_6.0_Patch_Common
source ./common.sh

THIS_SCRIPT="$(basename $(pwd))/$(basename $0)"
patch_name="InfoStreamDs_6.0_Full_Patch_Package"

function show_usage
{
    local sep=""
    local hint=""
    local readonly SEP=", "

    echo "usage: sh $0 <node_type> all"
    echo "       sh $0 <node_type> <module> [<module> ...]"
    echo "       sh $0 <node_type> ${OP_LIST_INSTALLED}"

    for node_type in "${node_types[@]}"; do
        hint="${hint}${sep}${node_type}"
        sep="$SEP"
    done
    echo -e "node_type\t: ${hint}"

    hint=""
    sep=""    
    for module in "${common_modules[@]}" "${mmu_modules[@]}"; do
        hint="${hint}${sep}${module}"
        sep="$SEP"
    done
    echo -e "mmu modules\t: ${hint} "

    hint=""
    sep=""
    for module in "${common_modules[@]}"  "${vpu_modules[@]}"; do
        hint="${hint}${sep}${module}"
        sep="$SEP"
    done    
    echo -e "vpu modules\t: ${hint}"
}

# 输入参数检查，参数个数
function check_param
{
    if [ $# -lt 2 ];then
        return 1
    fi
}

function do_install()
{
    local install_dir="$1"
    shift
    local params="$@"
    local scripts=(
        "./install.sh"
        "./start_service.sh"
        "./stop_service.sh"
    )
    local ret=0
    local script_not_exit="FALSE"

    cd "$install_dir"
    for script in ${scripts}; do
        if [ ! -f "$script" ]; then
            Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "error: file ${script} not exist"
            script_not_exit="TRUE"
        fi
    done
    if [ "$script_not_exit" == "TRUE" ]; then
        cd - 1>/dev/null
        echo "${install_dir}:failed" >> ${patch_tag}
        return 1 
    fi

    sh ./stop_service.sh $params || ret=1
    sh ./install.sh $params || ret=1
    sh ./start_service.sh $params || ret=1
    sleep 10s
    cd - 1>/dev/null
    local flag="success"
    [ $ret -eq 1 ] && flag="failed"
    echo "${install_dir}:${flag}" >> ${patch_tag}
    return $ret
}

#
# 用法:
# 1. install_patch <node> all
# 2. install_patch <node> <module> [<module> ...]
# precondition: 
#    (1) <node> must be valid
function install_patch
{
    local node=$1
    shift
    local units=("$@")
    local install_units=()
    local install_succeed="TRUE"

    check_node_and_modules "$node" "${units[@]}"
    if [ $? -ne 0 ]; then
        echo "invalid node type or module. node type: ${node}, modules: ${units[*]}"
        return 1
    fi

    if [ "${units[0]}" == "all" ]; then
        install_units=($(get_node_all_modules "$node"))
    else
        install_units=(${units[@]})
    fi

    log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "begin install all patch. node type: $node, modules: ${install_units[*]}"
    echo "" > ${patch_tag}

    for unit in "${install_units[@]}"; do 
        local dir="$(query_module_dir "$unit")"
        if [ ! -d "$dir" ]; then
            Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "install directory ${dir} not exist, skip installing ${unit}"
            continue
        fi
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "install ${unit} begin"
        do_install "$dir" "$node" &
    done

    # 超时时间10分钟
    local out_time=600
    while [ ${out_time} -gt 0 ]; do
        sleep 1
        let out_time--
        install_units=("${install_units[@]}")
        for ((i=0; i<${#install_units[@]}; i++)); do
            local flag="end"
            local dir="$(query_module_dir "${install_units[$i]}")"
            grep -wq "${dir}" ${patch_tag} || continue
            grep -wq "${dir}:failed" ${patch_tag}
            if [ $? -eq 0 ]; then
                flag="failed"
                install_succeed="FALSE"
            else
                record_module_install_info ${install_units[$i]}
            fi
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "install ${install_units[$i]} ${flag}"
            unset install_units[$i] && break
        done
        [ ${#install_units[@]} -eq 0 ] && break
    done
    [ ${out_time} -eq 0 ] && install_succeed="FALSE" && log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "install ${install_units[@]} timeout."

    [ "$install_succeed" != "TRUE" ] && return 1 || return 0
}

function list_installed_patch()
{
    local node="$1"
    local units=()
    local install_succeed="TRUE"

    units=($(get_node_all_modules "$node"))
    for unit in "${units[@]}"; do 
        local dir="$(query_module_dir "$unit")"
        if [ ! -d "$dir" ]; then
            Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "directory ${dir} not exist"
            install_succeed="FALSE"
            continue
        fi
        if [ ! -f "./${dir}/install.sh" ]; then
            echo "error: ${dir}/install.sh does not exist, skip it"
            install_succeed="FALSE"
            continue
        fi
        cd "$dir"
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" " ${unit} list installed patch begin"
        sh ./install.sh "$node" "$OP_LIST_INSTALLED"
        if [ $? -ne 0 ]; then
            install_succeed="FALSE"
        fi
        cd - 1>/dev/null
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "${unit} list installed patch end"
    done    
    [ "$install_succeed" != "TRUE" ] && return 1 || return 0
}

#
# 清理以前遗留的补丁压缩包和压缩目录
#
function clean_older_patch_pkgs()
{
    local ver="6.0"
    # 补丁解压目录
    for dir in /root/InfoStreamDs_${ver}_Full_Patch_Package_SPH6*/; do
        if [ -d "$dir" ]; then
            rm -fr "$dir"
            Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "remove older patch dir: $dir"
        fi
    done
    # 补丁压缩包
    for file in /root/InfoStreamDs_${ver}_Full_Patch_Package_SPH6*.zip; do
        # 当前补丁压缩包不删除
        if [[ "$file" =~ $VERSION ]]; then
            continue
        fi
        if [ -f "$file" ]; then
            rm -f "$file"
            Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "remove older patch file: $file"
        fi
    done    
}

function main()
{
    local ret=0
    Log "${THIS_SCRIPT}:${LINENO}" "COMMIT_ID" "${COMMIT_ID}" "[Version:${PACKAGE_NAME}]"
    Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "$0 starting..."
    check_param "$@"
    if [ $? -ne 0 ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "wrong param"
        show_usage 
        return 1
    fi

    local node="$1"
    check_node "$node"
    if [ $? -ne 0 ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "invalid node type"
        show_usage 
        return 1
    fi

    # 清理本地老补丁安装包
    clean_older_patch_pkgs

    # 列出本地已安装补丁情况
    if [ "$2" == "$OP_LIST_INSTALLED" ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "list installed patch begin"
        list_installed_patch "$@"
        ret=$?
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "list installed patch end"
        return $ret
    fi

    # 安装补丁
    local ret_flag="install patch succeed"
    install_patch "$@"
    ret=$?
    [ $ret -ne 0 ] && ret_flag="install patch failed" && report_error "$patch_name"
    log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "${ret_flag}"
    echo "${ret_flag}" >> ${patch_tag}
    return $ret
}

if [ "$ENABLE_DEBUG" != "true" ]; then
    main "$@"
fi