#!/bin/sh
#
# Description: 集群补丁安装脚本
# Athor: 肖阳阳
# Last Modified: 2021-9-17
# Revision: v0.1

source ./common_cluster.sh

THIS_SCRIPT="$(basename $(pwd))/$(basename $0)"
patch_name="InfoStreamDs_6.0_Full_Patch_Package_Cluster"
CLUSTER_INFO_FILE="./cluster_info.ini"

function init()
{
    chmod u+x "$SSHPASS_EXE"
    chmod u+x ./bin/scp
}

function show_usage
{
    local sep=""
    local hint=""
    local readonly SEP=", "

    echo "usage: sh $0 <node_type> all"
    echo "       sh $0 <node_type> <module> [<module> ...]"
    echo "       sh $0 <node_type> ${OP_LIST_INSTALLED}"

    for node_type in "${node_types[@]}" "all"; do
        hint="${hint}${sep}${node_type}"
        sep="$SEP"
    done
    echo -e "node_type\t: ${hint}"

    hint=""
    sep=""    
    for module in "${common_modules[@]}" "${mmu_modules[@]}"; do
        hint="${hint}${sep}${module}"
        sep="$SEP"
    done
    echo -e "mmu modules\t: ${hint} "

    hint=""
    sep=""
    for module in "${common_modules[@]}"  "${vpu_modules[@]}"; do
        hint="${hint}${sep}${module}"
        sep="$SEP"
    done    
    echo -e "vpu modules\t: ${hint}"
}

# 输入参数检查，参数个数
function check_param
{
    if [ $# -lt 2 ];then
        return 1
    fi
}

#
# 用法:
# 1. install_patch <cnode> all
# 2. install_patch <cnode> <module> [<module> ...]
# precondition: 
#    (1) <cnode> must be valid
function install_patch
{
    local node=$1
    shift
    local units=("$@")
    local install_nodes=()
    local install_succeed="TRUE"

    check_node_and_modules "$node" "${units[@]}"
    if [ $? -ne 0 ]; then
        echo "invalid node type or module. node type: ${node}, modules: ${units[*]}"
        return 1
    fi
    local mmu_installs=()
    local vpu_installs=()
    local mmu_actual_installs=()
    local vpu_actual_installs=()
    local install_nodes=()

    case "$node" in
        "all")
            mmu_installs=("${units[@]}")
            vpu_installs=("${units[@]}")
            install_nodes=('mmu' 'vpu')
            ;;
        "mmu")
            mmu_installs=("${units[@]}")
            install_nodes=('mmu')
            ;;
        "vpu")
            vpu_installs=("${units[@]}")
            install_nodes=('vpu')
            ;;
    esac

    for unit in "${mmu_installs[@]}"; do
        check_node_and_modules "mmu" "$unit"
        [ $? -eq 0 ] && mmu_actual_installs+=("$unit")
    done

    for unit in "${vpu_installs[@]}"; do
        check_node_and_modules "vpu" "$unit"
        [ $? -eq 0 ] && vpu_actual_installs+=("$unit")
    done
    
    log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "begin install all patch. node type: ${install_nodes[*]}, modules: ${units[*]}"
    if [ "${#mmu_actual_installs[*]}" -ne 0 ]; then
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------install mmu slave begin. ip: ${mmu_slave_ip}"
        easy_remote_exec "$mmu_slave_ip" "rm -f ${patch_tag}; cd ${remote_patch_parent_dir}/${PATCH_DIR} && sh ${INSTALL_SCRIPT} mmu_slave ${mmu_actual_installs[*]}" &
        
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------install mmu_master begin. ip: ${mmu_master_ip}"
        easy_remote_exec "$mmu_master_ip" "rm -f ${patch_tag}; cd ${remote_patch_parent_dir}/${PATCH_DIR} && sh ${INSTALL_SCRIPT} mmu_master ${mmu_actual_installs[*]}" &
    fi

    if [ "${#vpu_actual_installs[*]}" -ne 0 ]; then
        for remote_ip in "${vpu_ips[@]}"; do
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------install vpu begin. ip: ${remote_ip}"
            easy_remote_exec "$remote_ip" "rm -f ${patch_tag}; cd ${remote_patch_parent_dir}/${PATCH_DIR} && sh ${INSTALL_SCRIPT} vpu ${vpu_actual_installs[*]}" &
        done
    fi

    local check_ips=()
    [ "${#mmu_actual_installs[*]}" -ne 0 ] && check_ips=("$mmu_slave_ip" "$mmu_master_ip")
    [ "${#vpu_actual_installs[*]}" -ne 0 ] && check_ips=("${check_ips[@]}" "${vpu_ips[@]}")
    # 超时时间10分钟，实际可能在20-30分钟
    sleep 10
    local out_time=600
    while [ ${out_time} -gt 0 ]; do
        sleep 1
        let out_time--
        check_ips=("${check_ips[@]}")
        for ((i=0; i<${#check_ips[@]}; i++)); do
            local flag="done."
            remote_exec "${check_ips[i]}" "grep -wq 'install patch' ${patch_tag}" || continue
            remote_exec "${check_ips[i]}" "grep -wq 'install patch failed' ${patch_tag}" && flag="failed." && install_succeed="FALSE"
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------install node ${check_ips[$i]}, ${flag}"
            unset check_ips[$i] && break
        done
        [ ${#check_ips[@]} -eq 0 ] && break
    done
    [ ${out_time} -eq 0 ] && install_succeed="FALSE" && log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------install node ${check_ips[@]}, timeout."
    
    [ "$install_succeed" != "TRUE" ] && return 1 || return 0
}

function list_installed_patch()
{
    local node="$1"
    local install_succeed="TRUE"
    local install_nodes=()

    if [ "$node" == "all" ]; then
        install_nodes=("mmu" "vpu")
    else
        install_nodes=("$node")
    fi

    for insatll_node in "${install_nodes[@]}"; do
        case "$insatll_node" in
        "mmu")
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------mmu_master(${mmu_master_ip}) status"
            easy_remote_exec "$mmu_master_ip" "cd ${remote_patch_parent_dir}/${PATCH_DIR} && sh ${INSTALL_SCRIPT} mmu_master ${OP_LIST_INSTALLED}" 
            if [ $? -ne 0 ]; then
                install_succeed="FALSE"
                log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "get mmu_master install status failed. ip: ${mmu_master_ip}"
                log_file ${SSH_OUTPUT_LOG}
            else
                cat ${SSH_OUTPUT_LOG}
            fi

            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------mmu_slave(${mmu_slave_ip}) status"
            easy_remote_exec "$mmu_slave_ip" "cd ${remote_patch_parent_dir}/${PATCH_DIR} && sh ${INSTALL_SCRIPT} mmu_slave ${OP_LIST_INSTALLED}" 
            if [ $? -ne 0 ]; then
                install_succeed="FALSE"
                log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "get mmu_slave install status failed. ip: ${mmu_slave_ip}"
                log_file ${SSH_OUTPUT_LOG}
            else
                cat ${SSH_OUTPUT_LOG}
            fi
            ;;
        "vpu")
            for remote_ip in "${vpu_ips[@]}"; do
            log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "------vpu(${remote_ip}) status"
                easy_remote_exec "$remote_ip" "cd ${remote_patch_parent_dir}/${PATCH_DIR} && sh ${INSTALL_SCRIPT} vpu ${OP_LIST_INSTALLED}" 
                if [ $? -ne 0 ]; then
                    install_succeed="FALSE"
                    log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO" "get vpu install status failed. ip: ${remote_ip}"
                    log_file ${SSH_OUTPUT_LOG}
                else 
                    cat ${SSH_OUTPUT_LOG}
                fi
            done
            ;;
        esac
    done
    [ "$install_succeed" != "TRUE" ] && return 1 || return 0
}


function main()
{
    local ret=0
    Log "${THIS_SCRIPT}:${LINENO}" "COMMIT_ID" "${COMMIT_ID}" "[Version:${PACKAGE_NAME}]"
    Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "$0 starting..."
    check_param "$@"
    if [ $? -ne 0 ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "wrong param"
        show_usage 
        return 1
    fi

    local node="$1"
    check_node "$node"
    if [ $? -ne 0 ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "invalid node type"
        show_usage 
        return 1
    fi

    local cluster_file="./cluster_info.ini"
    read_cluster_info "$cluster_file"
    if [ $? -ne 0 ]; then
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "invalid cluster info, please check $cluster_file"
        return 1
    fi    
    # 列出已安装补丁情况
    if [ "$2" == "$OP_LIST_INSTALLED" ]; then
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "list installed patch begin"
        list_installed_patch "$@"
        ret=$?
        Log "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "list installed patch end"
        return $ret
    fi

    # 安装补丁
    install_patch "$@"
    if [ $? -ne 0 ]; then
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "install patch failed"
        report_error "$patch_name"
        return 1
    else
        log_and_print_console "$THIS_SCRIPT" "$FUNCNAME" "$LINENO"  "install patch succeed"
    fi
    return 0
}

if [ "$ENABLE_DEBUG" != "true" ]; then
    main "$@"
fi