#!/bin/bash

PATH=$PATH:/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin:~/bin
export PATH

input=$*
ppid=$$

ovs_en=0
if [ -f /opt/uvp/evs/user_evs_config ];then
    rpm -q openvswitch > /dev/null
    if [ $? -eq 0 ];then
        ovs_en=1
    fi
fi

#macro for timer controller, 0: no timer, 1: start timer
TIMER=0
if [[ ${TIMER} -ne 0 && ${input[*]/-c/} != "${input[*]}" ]]
then
    timer(){
        sleep 8.5s
        echo "Time up, force to compress logs"
        echo "Stop all log collect process..."
        pstree ${ppid} -p
        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | grep -v ^${ppid}$ | xargs kill -9 >/dev/null 2>&1
        kill -15 $ppid
        exit 0
    }
    echo "Start timer"
    timer &
fi

help_info="
  Usage: hinic_log_collect <-type> [option]
    -h, --help      show help information

    (Required) You must choose one device or all devices that collects logs:
    -a, --all       collect all environment and devices information
    -i, --interface collect environment and specific device information

    (option) The following options are optional:
    -f, --feature   collect specific feature logs, such as ovs
    -c, --chip      collect only chip devices information
    -p, --pangea    collect the information that pangea do not collected
    -n, --network   collect cloud network information
    -l, --log       collection operation path
    -z, --zip       path to store compressed logs

    (reserved)
    -d, --dump      the switch to dump register
"

#PCI_VENDOR_ID_HUAWEI				0x19e5
#HINIC_DEV_ID_1822_PF    			0x1822
#HINIC_DEV_ID_1822_SMTIO 			0x020B
#HINIC_DEV_ID_1822_PANGEA_100GE 	0x0208
#HINIC_DEV_ID_1822_PANGEA_TP_10GE	0x0204
#HINIC_DEV_ID_1822_KR_40GE 			0x020D
#HINIC_DEV_ID_1822_KR_100GE 		0x0205
#HINIC_DEV_ID_1822_KR_25GE			0x0210
#HINIC_DEV_ID_1822_MULTI_HOST		0x0211
#HINIC_DEV_ID_1822_PRD_2x25GE		0x0206
#HINIC_DEV_ID_1822_PRD_2x100GE		0x0200

if [[ -z `lspci -d 19e5:1822` && -z `lspci -d 19e5:0208` && -z `lspci -d 19e5:020b` && -z `lspci -d 19e5:0204` \
&& -z `lspci -d 19e5:020d` && -z `lspci -d 19e5:0205` && -z `lspci -d 19e5:0210` && -z `lspci -d 19e5:0211` \
&& -z `lspci -d 19e5:0206` && -z `lspci -d 19e5:0200` ]];
then
    echo "Info: There is no HW device."
    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
    exit 1
fi

lsmod | grep hinic >/dev/null
if [[ $? -ne 0 ]];
then
    echo "ERROR: NO hinic driver."
    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
    exit 1
fi

#get input parameters
dev_index=''
feature_name=''
dump_register=''
op_type=''

para1=${input//--help/-h}
para2=${para1//--all/-a}
para3=${para2//--interface/-i}
para4=${para3//--feature/-f}
para5=${para4//--dump/-d}
para6=${para5//--chip/-c}
para7=${para6//--pangea/-p}
para8=${para7//--log/-l}
para9=${para8//--zip/-z}
IFS=" " read -r -a arr_input <<< "$para9"

for (( i = 0; i < ${#arr_input[@]}; i++ )); do
    case ${arr_input[i]} in
        '-h' )
            echo "$help_info"
            pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
            exit 1
            ;;
        '-d' )
            dump_register+='dump'
            ;;
        '-f' )
            case ${arr_input[i+1]} in
            'ovs' )
                if [ -z $feature_name ]
                then
                    feature_name+='ovs'
                else
                    printf "\n  [ERROR]You can only choose one feature to collect\n"
                    echo "$help_info"
                    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                    exit 1
                fi
                ;;
            'nic' )
                if [ -z $feature_name ]
                then
                    feature_name+='nic'
                else
                    printf "\n  [ERROR]You can only choose one feature to collect\n"
                    echo "$help_info"
                    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                    exit 1
                fi
                ;;
            'roce' )
                if [ -z $feature_name ]
                then
                    feature_name+='roce'
                else
                    printf "\n  [ERROR]You can only choose one feature to collect\n"
                    echo "$help_info"
                    exit 1
                fi
                ;;
            'dpdk' )
                if [ -z $feature_name ]
                then
                    feature_name+='dpdk'
                else
                    printf "\n  [ERROR]You can only choose one feature to collect\n"
                    echo "$help_info"
                    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                    exit 1
                fi
                ;;
            * )
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            esac
            ;;
        '-p' )
            if [ -z $op_type ]
            then
                op_type="p"
            else
                printf "\n  [ERROR]You can only choose one type of operation\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        '-c' )
            if [ -z $op_type ]
            then
                op_type="c"
            else
                printf "\n  [ERROR]You can only choose one type of operation\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        '-n' )
            if [ -z $op_type ]
            then
                op_type="n"
            else
                printf "\n  [ERROR]You can only choose one type of operation\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        '-a' )
            if [ -z "$dev_index" ]
            then
                dev_index+='all'
            else
                printf "\n  [ERROR]You can only choose one type of device\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        '-i' )
            if [ -z "$dev_index" ]
            then
                case ${arr_input[i+1]} in
                    'hinic'[0-9] )
                        dev_index+=${arr_input[i+1]}
                        ;;
                    'hinic'[0-9][0-9] )
                        dev_index+=${arr_input[i+1]}
                        ;;
                    * )
                        printf "\n  [ERROR]please enter the correct parameters\n"
                        echo "$help_info"
                        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                        exit 1
                esac
            else
                printf "\n  [ERROR]You can only choose one type of device\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        '-l' )
            if [ -z "$HI1822_LOG_PATH" ]
            then
                case ${arr_input[i+1]} in
                    '/'[[:alnum:]]* )
                        HI1822_LOG_PATH=${arr_input[i+1]}
                        final=`echo "${HI1822_LOG_PATH: -1}"`
                        if [ "$final" == '/' ]
                        then
                            HI1822_LOG_PATH=`echo "${HI1822_LOG_PATH%/*}"`
                        fi
                    ;;
                    * )
                        printf "\n  [ERROR]please enter the correct parameters!!\n"
                        echo "$help_info"
                        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                        exit 1
                esac
            else
                printf "\n  [ERROR]You can only choose one type of device!!\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        '-z' )
            if [ -z "$HI1822_LOG_ARCHIV_PATH" ]
            then
                case ${arr_input[i+1]} in
                    '/'[[:alnum:]]* )
                        HI1822_LOG_ARCHIV_PATH=${arr_input[i+1]}
                        final=`echo "${HI1822_LOG_ARCHIV_PATH: -1}"`
                        if [ "$final" == '/' ]
                        then
                            HI1822_LOG_ARCHIV_PATH=`echo "${HI1822_LOG_ARCHIV_PATH%/*}"`
                        fi
                    ;;
                    * )
                        printf "\n  [ERROR]please enter the correct parameters!!\n"
                        echo "$help_info"
                        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                        exit 1
                esac
            else
                printf "\n  [ERROR]You can only choose one type of device!!\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        'ovs' )
            if [ $i = 0 ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            if [ "${arr_input[$i-1]}" != '-f' ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        'nic' )
            if [ $i = 0 ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            if [ "${arr_input[$i-1]}" != '-f' ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        'roce' )
            if [ $i = 0 ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            if [ "${arr_input[$i-1]}" != '-f' ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        'dpdk' )
            if [ $i = 0 ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            if [ "${arr_input[$i-1]}" != '-f' ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        'hinic'[0-9] )
            if [ $i = 0 ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            if [ "${arr_input[$i-1]}" != '-i' ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        'hinic'[0-9][0-9] )
            if [ $i = 0 ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            if [ "${arr_input[$i-1]}" != '-i' ]
                then
                printf "\n  [ERROR]please enter the correct parameters\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        '/'[[:alnum:]]* )
            if [ "${arr_input[$i-1]}" != '-l' ] && [ "${arr_input[$i-1]}" != '-z' ]
            then
                printf "\n  [ERROR]please enter the correct parameters!!\n"
                echo "$help_info"
                exit 1
            fi
            ;;
        * )
            printf "\n  [ERROR]please enter the correct parameters\n"
            echo "$help_info"
            pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
            exit 1
    esac
done

if [ -z "$dev_index" ]
then
    printf "\n  [ERROR]You must choose one device or all devices\n"
    echo "$help_info"
    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
    exit 1
fi

#hinicadm log path
HINICADM_LOG_PATH=/opt/hinic
if [ X${op_type} == 'Xn' ] && [ -z "${HI1822_LOG_PATH}" ]
then
    HI1822_LOG_PATH=/var/log/unic_log
fi

#collect environmental informations
env_type=""
eth_1822=""
dev_name=""

if [ -e /proc/osp/bsp ]; then
    if [ -n "$(cat /proc/osp/bsp | grep -i 'PANGEA')" ]; then
        env_type="storage"
        if [ -z $HI1822_LOG_PATH ]
        then
            HI1822_LOG_PATH=/OSM/log/cur_debug/pangea
        fi

        #pangea archive path
        if [ -z "${HI1822_LOG_ARCHIV_PATH}" ]
        then
            HI1822_LOG_ARCHIV_PATH=/OSM/coffer_log/log/cur_debug/pangea
        fi

    fi
elif [ -e /etc/EulerLinux.conf ] && [ -n "$(cat /etc/EulerLinux.conf | grep -i 'uvp')" ]; then
    env_type="uvp"
    if [ -z $HI1822_LOG_PATH ]
    then
        HI1822_LOG_PATH=/opt/hinic
    fi
else
    env_type="server"
    if [ -z $HI1822_LOG_PATH ]
    then
        HI1822_LOG_PATH=/opt/hinic
    fi
fi

eth_all=`ip link show | grep "<" | awk -F ':' '{print $2}'`
for eth_tmp in $eth_all
do
    ret=`ethtool -i "$eth_tmp" 2>/dev/null | grep -o 'hinic'`
    if [[ $ret =~ "hinic" ]]
    then
    eth_1822=${eth_1822}" "${eth_tmp}
    fi
done

dev_info=`hinicadm info`
dev_list=`echo "$dev_info" | grep -o 'hinic[0-9]\{1,2\}'`
echo "Environment type: $env_type"
echo "Eth list: $eth_1822"
echo "Device list: $dev_list"
echo "Device index: $dev_index"
echo "Operation type: ${op_type}"

if [ -z "${dev_list}" ]
then
    printf "\n  [ERROR]There is a serious problem\n"
    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
    exit 1
fi

compress_log(){
    if [[ $# -ne 3 ]]; then
        echo "Compression parameter error"
        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
        exit 1
    fi

    echo "$@" | grep -E "[[:space:]]/[[:space:]]|/bin"
    if [[ $? -eq 0 ]]; then
        echo "Compression path error"
        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
        exit 1
    fi

    hi1822_log_path=$1
    hinicadm_log_path=$2
    date_str=$3


    cd "${hi1822_log_path}" || exit
    tar_path="${hi1822_log_path}"

    echo "Start compress logs..."
    chmod 400 ./"${date_str}"/*
    tar -zcvf log_"${date_str}".tar.gz "${date_str}" >> /dev/null

    chmod 400 log_"${date_str}".tar.gz

    if [[ $env_type == "storage" ]]; then
        rm -rf "${hinicadm_log_path:?}"/*
    fi

    if [ "$HI1822_LOG_ARCHIV_PATH" ]
    then
        if [ -d "$HI1822_LOG_ARCHIV_PATH" ]
        then
            echo Archive directory "$HI1822_LOG_ARCHIV_PATH" exists
        else
            mkdir -p "$HI1822_LOG_ARCHIV_PATH"
        fi
        if [[ $env_type == "storage" ]]; then
            cp -f "${hi1822_log_path}"/log_"${date_str}".tar.gz ${HI1822_LOG_ARCHIV_PATH}/
        else
            mv -f "${hi1822_log_path}"/log_"${date_str}".tar.gz ${HI1822_LOG_ARCHIV_PATH}/
        fi
        tar_path=${HI1822_LOG_ARCHIV_PATH}
    fi

    rm -rf "${hi1822_log_path:?}"/"${date_str}"

    echo "Compress logs done"
    echo "Compressed logs path: ${tar_path}/log_${date_str}.tar.gz"
    echo "Log collection end"
    pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
}

collect_chip_logs(){
    dev_name=$1
    hinicadm_log_path=$2
    hi1822_log_path=$3
    date_str=$4

    hinicadm log -i "${dev_name}" -a
    echo "Moving chip logs to ${hi1822_log_path}/${date_str}/"

    logs=(ucode_ram_"${dev_name}" ucode_flash_"${dev_name}" ucode_lastword_flash_"${dev_name}"  up_ram_"${dev_name}" up_flash_"${dev_name}" up_lastword_flash_"${dev_name}")

    for i in `seq 10`
    do

        for log in ${logs[*]}
        do
            num=`find "${hinicadm_log_path}" -type f -name "${log}*.log" | wc -l`
            if [ "$num" -eq 0 ]
            then
                continue
            fi
            #log found
            LASTLOGFILE=`find "${hinicadm_log_path}" -type f -name "${log}*.log" | sort | tail -1`
            if [ "$LASTLOGFILE" ]; then
                # shellcheck disable=SC2086
                mv $LASTLOGFILE "${hi1822_log_path}"/"${date_str}" >> /dev/null
            fi
            # shellcheck disable=SC2206
            logs=(${logs[@]/$log/})

        done
        if [[ ${#logs[*]} -eq 0 ]];then
            break
        fi

        sleep 0.2
    done

    echo "Collect ${dev_name} chip logs done"
}

collect_p_log(){
    dev_name=$1
    hi1822_log_path=$2
    date_str=$3
    hinicadm_log_path=$4

    echo "Collecting ${dev_name} firmware log...."
    hinicadm version -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_version.log
    if [ $? -ne 0 ]
    then
        printf "\n  [ERROR]There is a serious problem,the tool may not be installed\n"
        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
        exit 1
    fi

    #chip logs
    collect_chip_logs "${dev_name}" "${hinicadm_log_path}" "${hi1822_log_path}" "${date_str}"&

    #counter
    hinicadm counter -i "${dev_name}" -t 0 -x 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 0 -x 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 17 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 18 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 20 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 36 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 37 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 38 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 2 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log &

    hinicadm event -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_event.log &
    hinicadm info >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_info.log
    hinicadm info -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_info.log &
    hinicadm mode -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_mode.log &

    hinicadm temperature -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_temperature.log &
    hinicadm mac -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_mac.log &
    hinicadm qos -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_qos.log &
    hinicadm table -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_table.log &

    hinicadm hilink_param -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_param.log
    hinicadm hilink_param -i "${dev_name}" -t 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_param.log
    port_num_info=`hinicadm info -i "${dev_name}" | grep "port num" | grep -o '[0-9]\{1,3\}'`
    port_num=`echo "$port_num_info" | grep -o "[0-9]"`
    if [ "$port_num" != "" ]
    then
        for (( i = 0; i < $port_num; i++ )); do
            hinicadm hilink_dump -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_dump.log
            hinicadm hilink_port -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_port.log
            hinicadm hilink_count -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_count.log
            hinicadm fec -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_fec.log &
            hinicadm autoneg -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_autoneg.log &
            hinicadm lro -i "${dev_name}" -x $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_lro.log &
            tp_eth_flag=`hinicadm info -i "${dev_name}" | grep "TP_ETH_10GE"`
            if [ "$tp_eth_flag" == "" ]
            then
                hinicadm sfp -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_sfp.log
                hinicadm sfp -i "${dev_name}" -p $i -a >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_sfp.log &
            fi
        done
    fi

    arr_eth=$(echo "$eth_1822" | tr " " "\n")
    for eth in ${arr_eth[*]}
    do
        hinicadm rate -i "$eth" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_pf_rate.log &
    done
    service_mode=`hinicadm info -i ${dev_name}| grep -i "Service"  | awk -F ": " '{print $2}'`;
    if [[ $service_mode -eq 3 ]]
    then
        hinicadm counter -i "${dev_name}" -t 1 -x 25 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
        hinicadm counter -i "${dev_name}" -t 1 -x 26 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
        hinicadm toeconn -i "${dev_name}" -c >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toeconn -i "${dev_name}" -l >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toeconn -i "${dev_name}" -e >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toestat -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_stats.log &
        hinicadm toestat -i "${dev_name}" -t 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_stats.log &
        for eth in ${arr_eth[*]}
        do
            hinicadm toectrl -i "${dev_name}" -n "$eth" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_ctrl.log &
        done
    fi

    wait
    echo "Collect chip ${dev_name} related logs done"
}

collect_interrupts_stat_snmp_log(){
    hi1822_log_path=$1
    date_str=$2
    cat /proc/interrupts >> "${hi1822_log_path}"/"${date_str}"/interrupts_1.log
    cat /proc/stat >> "${hi1822_log_path}"/"${date_str}"/stat_1.log
    cat /proc/net/snmp >> "${hi1822_log_path}"/"${date_str}"/snmp_1.log
    sleep 3
    cat /proc/interrupts >> "${hi1822_log_path}"/"${date_str}"/interrupts_2.log 2>&1
    cat /proc/stat >> "${hi1822_log_path}"/"${date_str}"/stat_2.log 2>&1
    cat /proc/net/snmp >> "${hi1822_log_path}"/"${date_str}"/snmp_2.log 2>&1
}

collect_p_host_log(){
    hi1822_log_path=$1
    date_str=$2
    hinicadm_log_path=$3

    echo "Collection operation path: ${hi1822_log_path}/${date_str}/"
    echo "Collecting host related logs..."

    #host info
    top -bn 1 >> "${hi1822_log_path}"/"${date_str}"/top.log &

    collect_interrupts_stat_snmp_log "${hi1822_log_path}" "${date_str}"&

    uname -a >> "${hi1822_log_path}"/"${date_str}"/os_info.log
    vmstat >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
    ps euf >> "${hi1822_log_path}"/"${date_str}"/ps.log
    ps aux >> "${hi1822_log_path}"/"${date_str}"/ps.log &
    cat /proc/cpuinfo >> "${hi1822_log_path}"/"${date_str}"/cpuinfo.log &
    cat /proc/meminfo >> "${hi1822_log_path}"/"${date_str}"/meminfo.log &
    lspci -vvv >> "${hi1822_log_path}"/"${date_str}"/lspci_vvv.log 2>&1 &
    lspci -xxx >> "${hi1822_log_path}"/"${date_str}"/lspci_xxx.log 2>&1 &

    #interface info
    ifconfig -a >> "${hi1822_log_path}"/"${date_str}"/ifconfig.log &
    arr_eth=$(echo "$eth_1822" | tr " " "\n")
    for eth in ${arr_eth[*]}
    do
        echo "ethtool -S $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -S "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -k $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -k "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -c $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -c "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -a $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -a "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -x $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -x "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -i $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -i "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -g $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -g "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -l $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -l "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
		if [[ -e /sys/class/net/"$eth"/device/numa_node ]]; then
			cat /sys/class/net/"$eth"/device/numa_node >> "${hi1822_log_path}"/"${date_str}"/numa_node.log &
		fi
    done

    free -m >> "${hi1822_log_path}"/"${date_str}"/free.log &

    #kernel info
    uname -a | grep aarch64
    if [[ $? -eq 0 ]];
    then
        if [[ -e /boot/efi/EFI/euleros/grub.cfg ]]; then
            cat /boot/efi/EFI/euleros/grub.cfg >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
        fi
    else
        if [[ -e /boot/grub2/grub.cfg ]]; then
            cat /boot/grub2/grub.cfg >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
        fi
    fi

    #interrupts
    # shellcheck disable=SC2207
    # shellcheck disable=SC2140
    irq_num=(`cat /proc/interrupts |grep qp | awk -F "":"" '{print $1}'`)
    for irq in ${irq_num[*]}
    do
        echo "irq $irq:" >> "${hi1822_log_path}"/"${date_str}"/smp_affinity.log
        cat /proc/irq/"${irq}"/smp_affinity >> "${hi1822_log_path}"/"${date_str}"/smp_affinity.log &
    done
    wait

    LASTLOGFILE=`find "$hinicadm_log_path" -type f -name "operations.log" | sort | tail -1`
    if [ "$LASTLOGFILE" ]; then
        # shellcheck disable=SC2086
        mv $LASTLOGFILE "${hi1822_log_path}"/"${date_str}" >> /dev/null
    fi
    wait
    echo "Collect host related logs done"
}

collect_c_log(){
    dev_name=$1
    hi1822_log_path=$2
    date_str=$3
    hinicadm_log_path=$4

    echo "Collecting ${dev_name} firmware log..."

    hinicadm version -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_version.log
    if [ $? -ne 0 ]
    then
        printf "\n  [ERROR]There is a serious problem,the tool may not be installed\n"
        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
        exit 1
    fi

    #chip logs
    collect_chip_logs "${dev_name}" "${hinicadm_log_path}" "${hi1822_log_path}" "${date_str}"&

    #counter
    hinicadm counter -i "${dev_name}" -t 0 -x 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 0 -x 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 17 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 18 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 20 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 36 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 37 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log
    hinicadm counter -i "${dev_name}" -t 1 -x 38 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log &

    hinicadm event -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_event.log &
    hinicadm info >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_info.log
    hinicadm info -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_info.log &
    hinicadm mode -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_mode.log &

    hinicadm hilink_param -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_param.log
    hinicadm hilink_param -i "${dev_name}" -t 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_param.log
    port_num_info=`hinicadm info -i "${dev_name}" | grep "port num" | grep -o '[0-9]\{1,3\}'`
    port_num=`echo "$port_num_info" | grep -o "[0-9]"`
    if [ "$port_num" != "" ]
    then
        for (( i = 0; i < $port_num; i++ )); do
            hinicadm hilink_dump -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_dump.log
            hinicadm hilink_count -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_count.log
        done
    fi

    service_mode=`hinicadm info -i ${dev_name}| grep -i "Service"  | awk -F ": " '{print $2}'`;
    if [[ $service_mode -eq 3 ]]
    then
        hinicadm toeconn -i "${dev_name}" -c >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toeconn -i "${dev_name}" -l >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toeconn -i "${dev_name}" -e >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toestat -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_stats.log &
        hinicadm toestat -i "${dev_name}" -t 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_stats.log &
        arr_eth=$(echo "$eth_1822" | tr " " "\n")
        for eth in ${arr_eth[*]}
        do
            hinicadm toectrl -i "${dev_name}" -n "$eth" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_ctrl.log &
        done
    fi

    wait
    echo "Collect chip ${dev_name} related logs done"
}

collect_c_host_log(){
    hi1822_log_path=$1
    date_str=$2
    hinicadm_log_path=$3

    echo "Collection operation path: ${hi1822_log_path}/${date_str}/"
    echo "Collecting host related logs..."

    #host info
    uname -a >> "${hi1822_log_path}"/"${date_str}"/os_info.log
    vmstat >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
    ps euf >> "${hi1822_log_path}"/"${date_str}"/ps.log
    ps aux >> "${hi1822_log_path}"/"${date_str}"/ps.log &
    cat /proc/cpuinfo >> "${hi1822_log_path}"/"${date_str}"/cpuinfo.log &
    cat /proc/meminfo >> "${hi1822_log_path}"/"${date_str}"/meminfo.log &
    lspci -vvv >> "${hi1822_log_path}"/"${date_str}"/lspci_vvv.log &
    lspci -xxx >> "${hi1822_log_path}"/"${date_str}"/lspci_xxx.log &

    #interface info
    ifconfig -a >> "${hi1822_log_path}"/"${date_str}"/ifconfig.log &
    arr_eth=$(echo "$eth_1822" | tr " " "\n")
    for eth in ${arr_eth[*]}
    do
        echo "ethtool -S $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -S "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -k $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -k "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1 &
    done

    LASTLOGFILE=`find "$hinicadm_log_path" -type f -name "operations.log" | sort | tail -1`
    if [ "$LASTLOGFILE" ]; then
        # shellcheck disable=SC2086
        mv $LASTLOGFILE "${hi1822_log_path}"/"${date_str}" >> /dev/null
    fi
    wait
    echo "Collect host related logs done"
}

#collect device informations
collect_device_log(){
    dev_name=`echo "$1" | grep -o 'hinic[0-9]\{1,2\}'`
    hi1822_log_path=$2
    date_str=$3
    hinicadm_log_path=$4

    echo "Collecting ${dev_name} firmware log..."
    hinicadm version -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_version.log
    if [ $? -ne 0 ]
    then
        printf "\n  [ERROR]There is a serious problem,the tool may not be installed\n"
        pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
        exit 1
    fi

    #chip logs
    collect_chip_logs "${dev_name}" "${hinicadm_log_path}" "${hi1822_log_path}" "${date_str}" &

    #tile_io
    hinicadm tile_io -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_tile_io.log

    hinicadm event -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_event.log
    hinicadm fm_show -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_fm_show.log
    hinicadm info >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_info.log
    hinicadm info -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_info.log
    hinicadm temperature -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_temperature.log
    hinicadm mode -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_mode.log
    hinicadm sdi_mode -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_sdi_mode.log
    hinicconfig "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_config.log
    hinicadm mac -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_mac.log
    hinicadm qos -i "${dev_name}" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_qos.log
    hinicadm table -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_table.log
    hinicadm sdi_cfg -i "${dev_name}" -t cpath_vlan >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_cpath_vlan.log
    hinicadm sdi_cfg -i "${dev_name}" -t reset_scheme >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_reset_scheme.log

    hinicadm hilink_param -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_param.log
    hinicadm hilink_param -i "${dev_name}" -t 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_param.log
    arr_0_macro=(0 1 2 3 4 5 6 7)
    for i in ${arr_0_macro[*]}; do
        hinicadm serdes -i "${dev_name}" -t 1 -m "$i" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_serdes.log
    done
    ( hinicadm counter -i "${dev_name}";echo "Counter collect success." ) >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log &
    port_num_info=`hinicadm info -i "${dev_name}" | grep "port num" | grep -o '[0-9]\{1,3\}'`
    port_num=`echo "$port_num_info" | grep -o "[0-9]"`
    if [ "$port_num" != "" ]
    then
        for (( i = 0; i < $port_num; i++ )); do
            hinicadm hilink_dump -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_dump.log
            hinicadm hilink_port -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_port.log
            hinicadm hilink_count -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_hilink_count.log
            hinicadm fec -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_fec.log
            hinicadm autoneg -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_autoneg.log
            hinicadm lro -i "${dev_name}" -x $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_lro.log &
            tp_eth_flag=`hinicadm info -i "${dev_name}" | grep "TP_ETH_10GE"`
            if [ "$tp_eth_flag" == "" ]
            then
                hinicadm sfp -i "${dev_name}" -p $i >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_sfp.log
                hinicadm sfp -i "${dev_name}" -p $i -a >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_port_sfp.log
            fi
        done
    fi

    arr_eth=$(echo "$eth_1822" | tr " " "\n")
    for eth in ${arr_eth[*]}
    do
        hinicadm rate -i "$eth" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_pf_rate.log
    done
    service_mode=`hinicadm info -i ${dev_name}| grep -i "Service"  | awk -F ": " '{print $2}'`;
    if [[ $service_mode -eq 3 ]]
    then
        hinicadm toeconn -i "${dev_name}" -c >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toeconn -i "${dev_name}" -l >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toeconn -i "${dev_name}" -e >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_conn.log &
        hinicadm toestat -i "${dev_name}" -t 0 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_stats.log &
        hinicadm toestat -i "${dev_name}" -t 1 >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_stats.log &
        for eth in ${arr_eth[*]}
        do
            hinicadm toectrl -i "${dev_name}" -n "$eth" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_toe_ctrl.log &
        done
    fi

    with_full_eye=`hinicadm -h | grep -o 'full_eye'`
    if [ "$with_full_eye" != "" ]
    then
        arr_0_macro=(0 1 2 3)
        arr_0_lane=(0 1 2 3)
        for i in ${arr_0_macro[*]}; do
            for j in ${arr_0_lane[*]}; do
                echo "full_eye type 0 macro $i lane $j:" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_full_eye.log
                hinicadm full_eye -i "${dev_name}" -t 0 -m "$i" -l "$j" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_full_eye.log
            done
        done
        arr_1_macro=(4 5)
        arr_1_lane=(1 3 4 5)
        for i in ${arr_1_macro[*]}; do
            for j in ${arr_1_lane[*]}; do
                echo "full_eye type 1 macro $i lane $j:" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_full_eye.log
                hinicadm full_eye -i "${dev_name}" -t 1 -m "$i" -l "$j" >> "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_full_eye.log
            done
        done
    fi
}

#waiting for collection to complete
wait_for_complete(){
    hi1822_log_path=$1
    date_str=$2
    shift 2
    i=0
    index=0
    flag=( "|" "/" "-" "\\" )
    echo "Waiting for counter finished in devices: $*"
    echo "Logging is in progress, please wait..."

    check_line="Counter collect success"
    for dev_name in "$@"
    do
        cnt_size='0'
        counter_check_res=
        while [[ $cnt_size = '0' || -z ${counter_check_res} ]]
        do
            if [ $i -le 600 ]
            then
                let index=index%4
                printf "Progress: [%d%%][%s]\r" "`echo $((i/10))`" "${flag[$index]}"
                let i++
                let index++

            fi
            cnt_size=`test -e "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log &&du -b "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log | awk '{print $1}' || echo 0`
            counter_check_res=`cat "${hi1822_log_path}"/"${date_str}"/"${dev_name}"_counter.log | grep "${check_line}"`
            sleep 0.2
        done

    done
    if [ $i -le 600 ]; then
        printf "Progress: [100%%]   \n"
    else
        printf "\n"
    fi

}

collect_os_info(){
    hi1822_log_path=$1
    date_str=$2
    hinicadm_log_path=$3

    sys_log=messages
    if [ ! -e /var/log/messages ] && [ -e /var/log/syslog ]; then
        sys_log=syslog
    fi

    case $env_type in
        "server" )
            echo "Collection operation path: ${hi1822_log_path}/${date_str}/"
            cp -r /var/log/$sys_log "${hi1822_log_path}"/"${date_str}" &
            uname -a | grep aarch64
            if [[ $? -eq 0 ]];
            then
                POWERONFILE=`find /var/log/ -type f -name "messages.*.gz" | sort | tail -3`
                if [[ $POWERONFILE != "" ]];then
                    # shellcheck disable=SC2086
                    cp $POWERONFILE "${hi1822_log_path}"/"${date_str}" &
                fi
            else
                POWERONFILE=`find /var/log/ -type f -name "messages-2*.*" | sort | tail -3`
                if [[ $POWERONFILE != "" ]];then
                    # shellcheck disable=SC2086
                    cp $POWERONFILE "${hi1822_log_path}"/"${date_str}" &
                fi
            fi
            CRASHDATEFILE=`find /var/crash/ -type f -name "*dmesg.txt" | sort | tail -1`
            if [[ $CRASHDATEFILE != "" ]];then
                # shellcheck disable=SC2086
                cp $CRASHDATEFILE "${hi1822_log_path}"/"${date_str}" &
            fi

            cat /etc/*release | sort | uniq >> "${hi1822_log_path}"/"${date_str}"/lsb_release.log &

            lscpu >> "${hi1822_log_path}"/"${date_str}"/lscpu.log &
            service iptables status >> "${hi1822_log_path}"/"${date_str}"/iptables.log  2>/dev/null &
            service irqbalance status >> "${hi1822_log_path}"/"${date_str}"/irqbalance.log  2>/dev/null &
            ;;
        "uvp" )
            echo "Collection operation path: ${hi1822_log_path}/${date_str}/"
            cp -r /var/log/$sys_log "${hi1822_log_path}"/"${date_str}" &
            POWERONFILE=`find /var/log/ -type f -name "logdump-*.*" | sort -n -t '-' -k3 | tail -3`
            # shellcheck disable=SC2086
            cp $POWERONFILE "${hi1822_log_path}"/"${date_str}" &

            cat /etc/*release | sort | uniq >> "${hi1822_log_path}"/"${date_str}"/lsb_release.log &

            lscpu >> "${hi1822_log_path}"/"${date_str}"/lscpu.log
            service iptables status >> "${hi1822_log_path}"/"${date_str}"/iptables.log 2>/dev/null &
            service irqbalance status >> "${hi1822_log_path}"/"${date_str}"/irqbalance.log 2>/dev/null &

            if [ $ovs_en -eq 1 ];then
                cat /opt/uvp/evs/user_evs_config >> "${hi1822_log_path}"/"${date_str}"/user_evs_config.log &
                cp /var/log/openvswitch/ovs-vswitchd.log "${hi1822_log_path}"/"${date_str}" &
                cp /var/log/openvswitch/ovs-operations.log "${hi1822_log_path}"/"${date_str}" &
                cp /var/log/openvswitch/ovsdb-server.log "${hi1822_log_path}"/"${date_str}" &
                rpm -q openvswitch-vhostdp > /dev/null
                if [[ $? -eq 0 ]];
                then
                    if [ -f /var/log/vhostagent/vhostagent.log ];then
                        cp /var/log/vhostagent/vhostagent.log "${hi1822_log_path}"/"${date_str}" &
                    fi
                    ovs-appctl list-commands | grep "vhostagent/memory-show" > /dev/null
                    if [ $? -eq 0 ];then
                        ovs-appctl vhostagent/memory-show >> "${hi1822_log_path}"/"${date_str}"/ovs-memory-show.log &
                        if [ -f /var/log/vhostdp/vhostdp.log ];then
                            cp /var/log/vhostdp/vhostdp.log "${hi1822_log_path}"/"${date_str}" &
                        fi
                    fi
                fi
                ovs-vsctl show >> "${hi1822_log_path}"/"${date_str}"/ovs-vsctl-show.log
                typepath="${hi1822_log_path}"/"${date_str}"/ovs-vsctl-show.log
                typetmp=dpdkbond
                line=`awk '/'$typetmp'/{print NR}' "$typepath"`
                if [[ "${line}" != "" ]];
                then
                    start=`expr "$line" - 3`
                    dpdk_port=`sed -n ''"$start"','"$line"'p' "$typepath" | grep Port | awk -F " " '{print $2}' | sed 's/\"//g'`
                    ovs-appctl dpdk-bond/show "$dpdk_port" >> "${hi1822_log_path}"/"${date_str}"/dpdk-bond-show.log &
                    ovs-appctl dpif-dpdk/dump-eth-xstat "$dpdk_port" >> "${hi1822_log_path}"/"${date_str}"/dump-eth-xstat.log &
                fi

                ovs-appctl list-commands | grep hwbond-slave-show > /dev/null
                if [ $? -eq 0 ];then
                    typeovs=hwbond
                    line=`awk '/'$typeovs'/{print NR}' "$typepath"`
                    if [[ "${line}" != "" ]];
                    then
                        start=`expr "$line" - 3`
                        ovs_port=`sed -n ''"$start"','"$line"'p' "$typepath" | grep Port | awk -F " " '{print $2}' | sed 's/\"//g'`
                        ovs-appctl hwoff/hwbond-slave-show "$ovs_port" >> "${hi1822_log_path}"/"${date_str}"/hwbond-slave-show.log &
                    fi
                fi

                ovs-appctl list-commands | grep dpif-dpdk > /dev/null
                if [ $? -eq 0 ];then
                    ovs-appctl dpif-dpdk/dump-dpdk-mem >> "${hi1822_log_path}"/"${date_str}"/dump-dpdk-mem.log &
                    ovs-appctl dpif-dpdk/show-cpu-ports >> "${hi1822_log_path}"/"${date_str}"/show-cpu-ports.log &
                fi
            fi
            if [ -f /var/log/dpdk/dpdk.log ];then
                cp /var/log/dpdk/dpdk.log "${hi1822_log_path}"/"${date_str}" &
            fi
            cat /etc/uvp_version >> "${hi1822_log_path}"/"${date_str}"/uvp_version.log &
            if [ -f /etc/gcn_version ]; then
                cat /etc/gcn_version >> "${hi1822_log_path}"/"${date_str}"/gcn_version.log &
            fi
            cp -r /var/log/logdump "${hi1822_log_path}"/"${date_str}" &
            cp -r /var/log/idump "${hi1822_log_path}"/"${date_str}" &
            cp -r /var/crash "${hi1822_log_path}"/"${date_str}" &
            if [ -f /var/log/sysmonitor/hinic_monitor.log ];then
                cp /var/log/sysmonitor/hinic_monitor.log "${hi1822_log_path}"/"${date_str}" &
            fi
            if [ -f /opt/hinic/hinic_monitor/alarm_state ];then
                cp /opt/hinic/hinic_monitor/alarm_state "${hi1822_log_path}"/"${date_str}" &
            fi
            if [ -f /opt/hinic/hinic_monitor/hinicadm_temp_1 ];then
                cp /opt/hinic/hinic_monitor/hinicadm_temp_1 "${hi1822_log_path}"/"${date_str}" &
            fi
            if [ -f /opt/hinic/hinic_monitor/hinicadm_temp ];then
                cp /opt/hinic/hinic_monitor/hinicadm_temp "${hi1822_log_path}"/"${date_str}" &
            fi
            if [ -d /var/log/unic/ ];then
                cp -r /var/log/unic/* "${HI1822_LOG_PATH}"/"${DATE_STR}" &
                cp /etc/unic*_version "${HI1822_LOG_PATH}"/"${DATE_STR}" &
            fi
            ;;
        "storage" )
            mkdir -p "$hinicadm_log_path"
            echo "Collection operation path: ${hi1822_log_path}/${date_str}/"
            cp -r /OSM/coffer_log/log/his_debug/message "${hi1822_log_path}"/"${date_str}" &
            cp /OSM/coffer_log/log/cur_debug/messages "${hi1822_log_path}"/"${date_str}" &
            cp /OSM/conf/Match_version_file.cfg "${hi1822_log_path}"/"${date_str}" &
            POWERONFILE=`find /OSM/coffer_log/log/his_debug/log_debug/ -type f -name "his_tar_*.tgz" | sort | tail -3`
            # shellcheck disable=SC2086
            cp $POWERONFILE "${hi1822_log_path}"/"${date_str}"
            ;;
    esac

    echo "Collecting OS information..."
    collect_interrupts_stat_snmp_log "${hi1822_log_path}" "${date_str}" &
    uname -a >> "${hi1822_log_path}"/"${date_str}"/os_info.log

    uname -a | grep aarch64
    if [[ $? -eq 0 ]];
    then
        if [[ -e /boot/efi/EFI/euleros/grub.cfg ]]; then
            cat /boot/efi/EFI/euleros/grub.cfg >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
        fi
    else
        if [[ -e /boot/grub2/grub.cfg ]]; then
            cat /boot/grub2/grub.cfg >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
        fi
    fi

    if [[ $env_type == "storage" ]]; then
        getver >> "${hi1822_log_path}"/"${date_str}"/os_info.log 2>&1
    fi

    vmstat >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
    ps euf >> "${hi1822_log_path}"/"${date_str}"/ps.log
    ps aux >> "${hi1822_log_path}"/"${date_str}"/ps.log &
    cat /proc/cpuinfo >> "${hi1822_log_path}"/"${date_str}"/cpuinfo.log &
    top -bn 1 >> "${hi1822_log_path}"/"${date_str}"/top.log &
    cat /proc/meminfo >> "${hi1822_log_path}"/"${date_str}"/meminfo.log &
    free -m >> "${hi1822_log_path}"/"${date_str}"/free.log &
    lspci -vvv >> "${hi1822_log_path}"/"${date_str}"/lspci_vvv.log 2>&1 &
    lspci -xxx >> "${hi1822_log_path}"/"${date_str}"/lspci_xxx.log 2>&1 &
    ifconfig -a >> "${hi1822_log_path}"/"${date_str}"/ifconfig.log &

    # shellcheck disable=SC2207
    # shellcheck disable=SC2140
    irq_num=(`cat /proc/interrupts |grep qp | awk -F "":"" '{print $1}'`)
    for irq in ${irq_num[*]}
    do
        echo "irq $irq:" >> "${hi1822_log_path}"/"${date_str}"/smp_affinity.log
        cat /proc/irq/"${irq}"/smp_affinity >> "${hi1822_log_path}"/"${date_str}"/smp_affinity.log
    done

    arr_eth=$(echo "$eth_1822" | tr " " "\n")
    for eth in ${arr_eth[*]}
    do
        echo "ethtool -S $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -S "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -k $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -k "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -c $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -c "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -a $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -a "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -x $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -x "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -i $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -i "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -g $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -g "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -l $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -l "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1 &
		if [[ -e /sys/class/net/"$eth"/device/numa_node ]]; then
			cat /sys/class/net/"$eth"/device/numa_node >> "${hi1822_log_path}"/"${date_str}"/numa_node.log &
		fi
    done
    wait

    LASTLOGFILE=`find "$hinicadm_log_path" -type f -name "operations.log" | sort | tail -1`
    if [ "$LASTLOGFILE" ];then
        # shellcheck disable=SC2086
        mv $LASTLOGFILE "${hi1822_log_path}"/"${date_str}" >> /dev/null
    fi
    echo "OS information collection succeeded"
}

collect_n_host_log(){
    hi1822_log_path=$1
    date_str=$2
    hinicadm_log_path=$3

    echo "Collection operation path: ${hi1822_log_path}/${date_str}/"
    echo "Collecting host related logs..."

    top -bn 1 >> "${hi1822_log_path}"/"${date_str}"/top.log &

    collect_interrupts_stat_snmp_log "${hi1822_log_path}" "${date_str}"&
    #host info
    uname -a >> "${hi1822_log_path}"/"${date_str}"/os_info.log
    vmstat >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
    ps euf >> "${hi1822_log_path}"/"${date_str}"/ps.log
    ps aux >> "${hi1822_log_path}"/"${date_str}"/ps.log &
    cat /proc/cpuinfo >> "${hi1822_log_path}"/"${date_str}"/cpuinfo.log &
    cat /proc/meminfo >> "${hi1822_log_path}"/"${date_str}"/meminfo.log &
    lspci -vvv >> "${hi1822_log_path}"/"${date_str}"/lspci_vvv.log &
    lspci -xxx >> "${hi1822_log_path}"/"${date_str}"/lspci_xxx.log &

    case $env_type in
        "server" )
            lscpu >> "${hi1822_log_path}"/"${date_str}"/lscpu.log &
            cat /etc/*release | sort | uniq >> "${hi1822_log_path}"/"${date_str}"/lsb_release.log &
            service iptables status >> "${hi1822_log_path}"/"${date_str}"/iptables.log  2>/dev/null &
            service irqbalance status >> "${hi1822_log_path}"/"${date_str}"/irqbalance.log  2>/dev/null &
            ;;
        "uvp" )
            lscpu >> "${hi1822_log_path}"/"${date_str}"/lscpu.log &
            cat /etc/*release | sort | uniq >> "${hi1822_log_path}"/"${date_str}"/lsb_release.log &
            service iptables status >> "${hi1822_log_path}"/"${date_str}"/iptables.log  2>/dev/null &
            service irqbalance status >> "${hi1822_log_path}"/"${date_str}"/irqbalance.log  2>/dev/null &
            cat /etc/uvp_version >> "${hi1822_log_path}"/"${date_str}"/uvp_version.log &

            if [ $ovs_en -eq 1 ];then
                cat /opt/uvp/evs/user_evs_config >> "${hi1822_log_path}"/"${date_str}"/user_evs_config.log &
                ovs-vsctl show >> "${hi1822_log_path}"/"${date_str}"/ovs-vsctl-show.log
                typepath="${hi1822_log_path}"/"${date_str}"/ovs-vsctl-show.log
                typetmp=dpdkbond
                line=`awk '/'$typetmp'/{print NR}' "$typepath"`
                if [[ "${line}" != "" ]];
                then
                    start=`expr "$line" - 3`
                    dpdk_port=`sed -n ''"$start"','"$line"'p' "$typepath" | grep Port | awk -F " " '{print $2}' | sed 's/\"//g'`
                    ovs-appctl dpdk-bond/show "$dpdk_port" >> "${hi1822_log_path}"/"${date_str}"/dpdk-bond-show.log &
                    ovs-appctl dpif-dpdk/dump-eth-xstat "$dpdk_port" >> "${hi1822_log_path}"/"${date_str}"/dump-eth-xstat.log &
                fi
                typeovs=hwbond
                line=`awk '/'$typeovs'/{print NR}' "$typepath"`
                if [[ "${line}" != "" ]];
                then
                    start=`expr "$line" - 3`
                    ovs_port=`sed -n ''"$start"','"$line"'p' "$typepath" | grep Port | awk -F " " '{print $2}' | sed 's/\"//g'`
                    ovs-appctl hwoff/hwbond-slave-show "$ovs_port"  >> "${hi1822_log_path}"/"${date_str}"/hwbond-slave-show.log &
                fi
                ovs-appctl list-commands | grep dpif-dpdk > /dev/null
                if [ $? -eq 0 ];then
                    ovs-appctl dpif-dpdk/dump-dpdk-mem >> "${hi1822_log_path}"/"${date_str}"/dump-dpdk-mem.log &
                    ovs-appctl dpif-dpdk/show-cpu-ports >> "${hi1822_log_path}"/"${date_str}"/show-cpu-ports.log &
                fi
                ovs-appctl list-commands | grep hwoff-flow-agent > /dev/null
                if [ $? -eq 0 ];then
                    ovs-appctl hwoff-flow-agent/error-stats >> "${hi1822_log_path}"/"${date_str}"/ovs-error-stats.log &
                fi
                rpm -q openvswitch-vhostdp > /dev/null
                if [[ $? -eq 0 ]];
                then
                    ovs-appctl list-commands | grep "vhostagent/memory-show" > /dev/null
                    if [ $? -eq 0 ];then
                        ovs-appctl vhostagent/memory-show >> "${hi1822_log_path}"/"${date_str}"/ovs-memory-show.log &
                    fi
                fi
            fi
			if [ -f /var/log/sysmonitor/hinic_monitor.log ];then
                cp /var/log/sysmonitor/hinic_monitor.log "${hi1822_log_path}"/"${date_str}" &
            fi
			if [ -f /opt/hinic/hinic_monitor/alarm_state ];then
                cp /opt/hinic/hinic_monitor/alarm_state "${hi1822_log_path}"/"${date_str}" &
            fi
			if [ -f /opt/hinic/hinic_monitor/hinicadm_temp_1 ];then
                cp /opt/hinic/hinic_monitor/hinicadm_temp_1 "${hi1822_log_path}"/"${date_str}" &
            fi
			if [ -f /opt/hinic/hinic_monitor/hinicadm_temp ];then
                cp /opt/hinic/hinic_monitor/hinicadm_temp "${hi1822_log_path}"/"${date_str}" &
            fi
            ;;
        "storage" )
            ;;
    esac

    #interface info
    ifconfig -a >> "${hi1822_log_path}"/"${date_str}"/ifconfig.log &
    arr_eth=$(echo "$eth_1822" | tr " " "\n")
    for eth in ${arr_eth[*]}
    do
        echo "ethtool -S $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -S "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -k $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -k "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -c $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -c "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -a $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -a "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -x $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -x "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -i $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -i "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -g $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -g "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
        echo "ethtool -l $eth:" >>  "${hi1822_log_path}"/"${date_str}"/ethtool_info.log
        ethtool -l "$eth" >> "${hi1822_log_path}"/"${date_str}"/ethtool_info.log 2>&1
		if [[ -e /sys/class/net/"$eth"/device/numa_node ]]; then
			cat /sys/class/net/"$eth"/device/numa_node >> "${hi1822_log_path}"/"${date_str}"/numa_node.log &
		fi
    done

    free -m >> "${hi1822_log_path}"/"${date_str}"/free.log &

    uname -a | grep aarch64
    if [[ $? -eq 0 ]];
    then
        if [[ -e /boot/efi/EFI/euleros/grub.cfg ]]; then
            cat /boot/efi/EFI/euleros/grub.cfg >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
        fi
    else
        if [[ -e /boot/grub2/grub.cfg ]]; then
            cat /boot/grub2/grub.cfg >> "${hi1822_log_path}"/"${date_str}"/os_info.log &
        fi
    fi

    #interrupts
    # shellcheck disable=SC2207
    # shellcheck disable=SC2140
    irq_num=(`cat /proc/interrupts |grep qp | awk -F "":"" '{print $1}'`)
    for irq in ${irq_num[*]}
    do
        echo "irq $irq:" >> "${hi1822_log_path}"/"${date_str}"/smp_affinity.log
        cat /proc/irq/"${irq}"/smp_affinity >> "${hi1822_log_path}"/"${date_str}"/smp_affinity.log &
    done

    LASTLOGFILE=`find "$hinicadm_log_path" -type f -name "operations.log" | sort | tail -1`
    if [ "$LASTLOGFILE" ]; then
        # shellcheck disable=SC2086
        mv $LASTLOGFILE "${hi1822_log_path}"/"${date_str}" >> /dev/null
    fi
    wait
    echo "Collect host related logs done"
}

DATE_STR=`date +%Y_%m_%d_%H_%M_%S`
mkdir -p "${HI1822_LOG_PATH}"/"${DATE_STR}"
echo "User:" >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log
who am i >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log
id >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log
echo "Start time:" >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log
date -u >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log

if [ X$op_type != 'Xn' ]
then
    trap 'compress_log ${HI1822_LOG_PATH} ${HINICADM_LOG_PATH} ${DATE_STR};exit 0' TERM
fi

if [ X${op_type} == 'Xp' ];
then
    process_p() {
        #collect host logs
        collect_p_host_log "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
        # shellcheck disable=SC2206
        arr=(${dev_list})
        #collect chip logs
        case "$dev_index" in
        'all' )
            for ss in ${arr[*]}
            do
                collect_p_log "$ss" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
            done
            ;;
        * )
            if [[ ${arr[*]/"$dev_index"/} != "${arr[*]}" ]]
            then
                collect_p_log "$dev_index" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
            else
                printf "\n  [ERROR]please enter the correct device name\n"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        esac
        wait
    }
    process_p &
    wait $!
    echo "Option p log collection succeeded"
elif [ X${op_type} == 'Xc' ];
then
    process_c(){
        #collect host logs
        collect_p_host_log "${HI1822_LOG_PATH}" "${DATE_STR}" &
        # shellcheck disable=SC2206
        arr=(${dev_list})
        #collect chip logs
        case "$dev_index" in
        'all' )
            for ss in ${arr[*]}
            do
                collect_p_log "$ss" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
            done
            ;;
        * )
            if [[ ${arr[*]/"$dev_index"/} != "${arr[*]}" ]]
            then
                collect_p_log "$dev_index" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
            else
                printf "\n  [ERROR]please enter the correct device name\n"
                pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
                exit 1
            fi
            ;;
        esac
        wait
    }
    process_c &
    wait $!
    echo "Option c log collection succeeded"
elif [ X${op_type} == 'Xn' ];
then
    collect_n_host_log "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
    # shellcheck disable=SC2206
    arr=(${dev_list})
    case "$dev_index" in
        'all' )
            for ss in ${arr[*]}
            do
                collect_device_log "$ss" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
            done
            # shellcheck disable=SC2086
            wait_for_complete "${HI1822_LOG_PATH}" "${DATE_STR}" ${arr[*]}
            ;;
        * )

            if [[ ${arr[*]/"$dev_index"/} != "${arr[*]}" ]]
            then
                collect_device_log "$dev_index" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
            else
                printf "\n  [ERROR]please enter the correct device name\n"
                exit 1
            fi
            IFS=" " read -r -a arry <<< "$dev_index"
            # shellcheck disable=SC2086
            wait_for_complete "${HI1822_LOG_PATH}" "${DATE_STR}" ${arry[*]}
            ;;
    esac
    wait
    echo "Operation logs path: ${HI1822_LOG_PATH}/${DATE_STR}/"
    echo "Option n log collection succeeded"
else
    #collect os informations
    collect_os_info "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &

    #collect feature informations
    case $feature_name in
    'ovs' )
        ;;
    'dpdk' )
        ;;
    'nic' )
        ;;
    'roce' )
        ;;
    * )
        ;;
    esac

	# shellcheck disable=SC2206
    arr=(${dev_list})
    case "$dev_index" in
    'all' )
        for ss in ${arr[*]}
        do
            collect_device_log "$ss" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
        done
        wait
        # shellcheck disable=SC2086
        wait_for_complete "${HI1822_LOG_PATH}" "${DATE_STR}" ${arr[*]}
        ;;
    * )
        if [[ ${arr[*]/"$dev_index"/} != "${arr[*]}" ]]
        then
            collect_device_log "$dev_index" "${HI1822_LOG_PATH}" "${DATE_STR}" ${HINICADM_LOG_PATH} &
        else
            printf "\n  [ERROR]please enter the correct device name\n"
            pstree ${ppid} -p | awk -F"[()]" '{print $2}' | xargs kill -9 2>/dev/null 1>/dev/null
            exit 1
        fi
        IFS=" " read -r -a arry <<< "$dev_index"
        wait
        # shellcheck disable=SC2086
        wait_for_complete "${HI1822_LOG_PATH}" "${DATE_STR}" ${arry[*]}
        ;;
    esac
    wait
    echo "All log collection succeeded"
fi

echo "End time:" >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log
date -u >> "${HI1822_LOG_PATH}"/"${DATE_STR}"/collect_info.log

kill -TERM $ppid
