#include <linux/pci.h>

/**
 * pcie_get_mps - get PCI Express maximum payload size
 * @dev: PCI device to query
 *
 * Returns maximum payload size in bytes
 */
int pcie_get_mps(struct pci_dev *dev)
{
	u16 ctl;

	pcie_capability_read_word(dev, PCI_EXP_DEVCTL, &ctl);

	return 128 << ((ctl & PCI_EXP_DEVCTL_PAYLOAD) >> 5);
}
EXPORT_SYMBOL(pcie_get_mps);

/**
 * pcie_set_mps - set PCI Express maximum payload size
 * @dev: PCI device to query
 * @mps: maximum payload size in bytes
 *    valid values are 128, 256, 512, 1024, 2048, 4096
 *
 * If possible sets maximum payload size
 */
int pcie_set_mps(struct pci_dev *dev, int mps)
{
	u16 v;
#ifndef HAVE_PCI_DEV_PCIE_MPSS
	int pos;
	u16 reg16;
#endif

	if (mps < 128 || mps > 4096 || !is_power_of_2(mps))
		return -EINVAL;

	v = ffs(mps) - 8;

#ifdef HAVE_PCI_DEV_PCIE_MPSS
	if (v > dev->pcie_mpss)
		return -EINVAL;
#else
	pos = pci_find_capability(dev, PCI_CAP_ID_EXP);
	if (!pos)
	return -EINVAL;
	pci_read_config_word(dev, pos + PCI_EXP_DEVCAP, &reg16);
	if (v > (reg16 & PCI_EXP_DEVCAP_PAYLOAD))
		return -EINVAL;
#endif
	v <<= 5;

	return pcie_capability_clear_and_set_word(dev, PCI_EXP_DEVCTL,
						  PCI_EXP_DEVCTL_PAYLOAD, v);
}
EXPORT_SYMBOL(pcie_set_mps);
