/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_cli_script.h
*
* @purpose   CLI Text Based Configuration
*
* @component OPEN
*
* @create    4/26/2014
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_CLI_SCRIPT_H
#define OPENAPI_CLI_SCRIPT_H

#include "openapi_common.h"

#define OPENAPI_FILE_NAME_LENGTH_MAX   96
#define OPENAPI_CLI_MAX_STRING_LENGTH  255

/*********************************************************************
* @purpose  Apply CLI text based configuration 
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    txtCfgFile         @b{(input)}  File Name 
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
* @param    outFile            @b{(input)}  Output File Name
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
*
* @returns  OPEN_E_NONE        if default configuration applied 
* @returns  OPEN_E_FAIL        if invalid file name is passed
* @returns  OPEN_E_PARAM       if invalid parameter is passed
*
* @notes   Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiTxtCfgApply(openapiClientHandle_t *client_handle,
                                open_buffdesc *txtCfgFile, open_buffdesc *outFile);

/*********************************************************************
* @purpose  Generate running configuration into specified file
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    txtCfgFile         @b{(input)}  File Name 
* @param    -                  Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
*
* @returns  OPEN_E_NONE        if default configuration applied 
* @returns  OPEN_E_FAIL        if invalid file name is passed
* @returns  OPEN_E_PARAM       if invalid parameter is passed
*
* @notes   Calling this API will not change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiRunningCfgGenerate(openapiClientHandle_t *client_handle,
                                       open_buffdesc *txtCfgFile);

/*********************************************************************
* @purpose  Save running configuration into startup configuration file
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
*
* @returns  OPEN_E_NONE        if default configuration applied 
* @returns  OPEN_E_FAIL        if invalid file name is passed
* @returns  OPEN_E_PARAM       if invalid parameter is passed
*
* @notes   Calling this API will not change the running configuration of the switch.
*          Saves the configuration to the switch and when switch reboots the
*          saved configuration is applied.
*
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiSaveConfig(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Validate given configuration file
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    txtCfgFile         @b{(input)}  File Name 
* @param    -                  Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
* @param    outFile            @b{(input)}  Output File Name 
* @param    -                  Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
*
* @returns  OPEN_E_NONE        if the configuration is valid
* @returns  OPEN_E_FAIL        if the configuration is invalid
* @returns  OPEN_E_PARAM       if invalid parameter is passed
* @returns  OPEN_E_INTERNAL    if internal error has occurred
*
* @notes   Calling this API will not change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiTxtCfgValidate(openapiClientHandle_t *client_handle,
                                   open_buffdesc *txtCfgFile, open_buffdesc *outFile);

/*********************************************************************
* @purpose  Get the maximum file name length.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    maxLen          @b{(output)} Maximum length for a file name.
*
* @returns  OPEN_E_NONE     if file name maximum length is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.

* @notes    Calling this API will not change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.4
*
* @end
*********************************************************************/
open_error_t openapiFileNameMaxLengthGet(openapiClientHandle_t *client_handle,
                                         uint32_t *maxLen);

/*********************************************************************
* @purpose  gracefully Apply CLI text based configuration 
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    txtCfgFile         @b{(input)}  File Name 
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
* @param    outFile            @b{(input)}  Output File Name
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
*
* @returns  OPEN_E_NONE        if default configuration applied 
* @returns  OPEN_E_FAIL        if invalid file name is passed
* @returns  OPEN_E_PARAM       if invalid parameter is passed
*
* @notes   Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiTxtCfgNDCApply(openapiClientHandle_t *client_handle,
                                open_buffdesc *txtCfgFile, open_buffdesc *outFile);

/*********************************************************************
* @purpose  Generate configuration difference between running 
*           configuration and input configuration file
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    txtCfgFile         @b{(input)}  File Name
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
* @param    outFile            @b{(input)}  Output File Name
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
*
* @returns  OPEN_E_NONE        if default configuration applied
* @returns  OPEN_E_FAIL        if invalid file name is passed
* @returns  OPEN_E_PARAM       if invalid parameter is passed
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiTxtCfgNDCGenerate(openapiClientHandle_t *client_handle,
                                      open_buffdesc *txtCfgFile, 
                                      open_buffdesc *outFile);

/*********************************************************************
* @purpose  Execute given show CLI command and redirect its output to the  
*           specified file.
*
* @param    client_handle      @b{(input)}  Client handle from
*                                           registration API
* @param    showCliCommand     @b{(input)}  Show CLI command
*                              Maximum value @ref OPENAPI_CLI_MAX_STRING_LENGTH
* @param    fileName           @b{(input)}  Output file name
*                              Maximum value @ref OPENAPI_FILE_NAME_LENGTH_MAX
*
* @returns  OPEN_E_NONE        if default configuration applied
* @returns  OPEN_E_FAIL        if invalid file name is passed
* @returns  OPEN_E_PARAM       if invalid parameter is passed
* @returns  OPEN_E_UNAVAIL     if CLI package is not enabled.
*
* @notes   
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiCliShowExecuteToFile(openapiClientHandle_t *client_handle,
                                         open_buffdesc         *showCliCommand,
                                         open_buffdesc         *fileName);
#endif
