/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dot1s.h
*
* @purpose   Spanning Tree Configuration and Status 
*
* @component OPEN
*
* @create    10/25/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DOT1S_H_INCLUDED
#define OPENAPI_DOT1S_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_DOT1S_FORCE_VERSION_DOT1D    = 0, /**< IEEE 802.1d Spanning Tree */
  OPEN_DOT1S_FORCE_VERSION_UNUSED   = 1, /**< unused */
  OPEN_DOT1S_FORCE_VERSION_DOT1W    = 2, /**< IEEE 802.1w Rapid Spanning Tree */
  OPEN_DOT1S_FORCE_VERSION_DOT1S    = 3, /**< IEEE 802.1s Multiple Spanning Tree */
  OPEN_DOT1S_FORCE_VERSION_LAST          /**< must be last */
}OPEN_DOT1S_FORCE_VERSION_t;

typedef enum
{
  OPEN_DOT1S_PORT_GUARD_LOOP    = 0, /**< Port Guard Loop */
  OPEN_DOT1S_PORT_GUARD_ROOT    = 1, /**< Port Guard Root */
  OPEN_DOT1S_PORT_GUARD_NONE    = 2  /**< Port Guard None */
}OPEN_DOT1S_PORT_GUARD_t;

typedef enum
{
  OPEN_DOT1S_MGMT_STR_CONFIG_NAME = 0 /**< String type Configuration name */
}OPEN_DOT1S_MGMT_STR_t;

typedef enum
{
  OPEN_DOT1S_PORT_DISCARDING        = 1, /**< Port state Discarding */
  OPEN_DOT1S_PORT_LEARNING          = 2, /**< Port state Learning */
  OPEN_DOT1S_PORT_FORWARDING        = 3, /**< Port state Forwarding */
  OPEN_DOT1S_PORT_DISABLED          = 4, /**< Port state Disabled */
  OPEN_DOT1S_PORT_MANUAL_FWD        = 5, /**< Port state Manual Forward */
  OPEN_DOT1S_PORT_NOT_PARTICIPATE   = 6  /**< Port state Not Participating */
}OPEN_DOT1S_PORT_FWD_STATE_t;

typedef enum
{
  OPEN_DOT1S_ROLE_DISABLED          = 0, /**< DisabledPort */
  OPEN_DOT1S_ROLE_ROOT              = 1, /**< RootPort */
  OPEN_DOT1S_ROLE_DESIGNATED        = 2, /**< DesignatedPort */
  OPEN_DOT1S_ROLE_ALTERNATE         = 3, /**< AlternatePort */
  OPEN_DOT1S_ROLE_BACKUP            = 4, /**< BackupPort */
  OPEN_DOT1S_ROLE_MASTER            = 5  /**< MasterPort */

}OPEN_DOT1S_PORT_ROLE_t;

#define OPENAPI_DOT1S_CST_ID 0
#define OPENAPI_DOT1S_MSTID_MAX 4094

#define OPEN_DOT1S_VLAN_ID_LEN        2
#define OPEN_DOT1S_MAC_ID_LEN         OPEN_MAC_ADDR_LEN
#define OPEN_DOT1S_VID_MAC_ADDR_LEN  (OPEN_DOT1S_VLAN_ID_LEN + OPEN_DOT1S_MAC_ID_LEN)

/** DOT1S general bridge identifier formed by combining the
 *  bridge priority, instance identifier, and MAC address into
 *  an 8-octet hex value. */

typedef struct
{
  uint8_t octets[OPEN_DOT1S_VID_MAC_ADDR_LEN];
} OPEN_DOT1S_BRIDGEID_t;


/*********************************************************************
* @purpose  Gets the Spanning Tree Management string length. This length is used to
*           allocate and initialize the open_buffdesc type argument buffers
*           used by various DOT1S OpEN APIs.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    strType         @b{(input)}   string type
* @param    strLen          @b{(output)}  string length
*
* @returns  OPEN_E_NONE        String length Get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed. 
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1sMgmtStrLenGet(openapiClientHandle_t *client_handle, 
                                       OPEN_DOT1S_MGMT_STR_t strType,
                                       uint32_t *strLen);

/*********************************************************************
* @purpose  Get the maximum number of Spanning Tree VLANs.
* 
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    vlanMax          @b{(output)} Maximum VLAN
*
* @returns  OPEN_E_NONE      if SNTP priority is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    none
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1VlanMaxGet(openapiClientHandle_t *client_handle,
                                   uint32_t *vlanMax);

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Operational Mode
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    dot1sMode       @b{(input)}   spanning tree operational mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SModeSet(openapiClientHandle_t *client_handle,
                                 OPEN_CONTROL_t dot1sMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Operational Mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    dot1sMode       @b{(output)}  spanning tree operational mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SModeGet(openapiClientHandle_t *client_handle,
                                 OPEN_CONTROL_t *dot1sMode);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Filter mode for an interface
*
* @param    client_handle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
* @param    dot1sBPDUFilterMode     @b{(input)}   spanning tree BPDU filter mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFilterModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t dot1sBPDUFilterMode);

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Filter mode for an interface
*
* @param    client_handle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
* @param    dot1sBPDUFilterMode     @b{(output)}  spanning tree BPDU filter mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFilterModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t *dot1sBPDUFilterMode);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Flood mode for an interface
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sBPDUFloodMode    @b{(input)}   spanning tree BPDU Flood mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFloodModeSet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_CONTROL_t dot1sBPDUFloodMode);

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Flood mode for an interface
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sBPDUFloodMode     @b{(output)}  spanning tree BPDU Flood mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUFloodModeGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_CONTROL_t *dot1sBPDUFloodMode);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Guard Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUGuard     @b{(input)}   spanning tree BPDU guard
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUGuardModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t dot1sBPDUGuard);

/*********************************************************************
* @purpose  Gets the Spanning Tree BPDU Guard Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUGuard     @b{(output)}  spanning tree BPDU guard
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUGuardModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t *dot1sBPDUGuard);

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Filter Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUFilter    @b{(input)}   spanning tree BPDU filter
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUFilterModeSet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t dot1sBPDUFilter);

/*********************************************************************
* @purpose  Gets the Spanning Tree BPDU Filter Mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sBPDUFilter    @b{(output)}  spanning tree BPDU filter
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiDot1SBPDUFilterModeGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t *dot1sBPDUFilter);

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Name
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    dot1sConfigName       @b{(input)}   spanning tree configuration name
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* @notes  Maximum size for dot1sConfigName is determined by calling
*         openapiDot1sMgmtStrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigNameSet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *dot1sConfigName);

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Name
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    dot1sConfigName       @b{(output)}  spanning tree configuration name
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Maximum size for dot1sConfigName is determined by calling
*         openapiDot1sMgmtStrLenGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigNameGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *dot1sConfigName);

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Revision
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    dot1sConfigRev      @b{(input)}   spanning tree configuration revision
* @param    -                   Minimum value @ref OPENAPI_DOT1S_MIN_REVISION_VALUE
* @param    -                   Maximum value @ref OPENAPI_DOT1S_MAX_REVISION_VALUE 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigRevisionSet(openapiClientHandle_t *client_handle,
                                           uint32_t dot1sConfigRev);

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Revision
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sConfigRev     @b{(output)}  spanning tree configuration revision
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SConfigRevisionGet(openapiClientHandle_t *client_handle,
                                           uint32_t *dot1sConfigRev);

/*********************************************************************
* @purpose  Configures Spanning Tree Force Revision
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sForceVersion  @b{(input)}   spanning tree force version
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SForceVersionSet(openapiClientHandle_t *client_handle,
                                         OPEN_DOT1S_FORCE_VERSION_t dot1sForceVersion);

/*********************************************************************
* @purpose  Gets the Spanning Tree Force Revision
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sForceVersion  @b{(output)}  spanning tree force version
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SForceVersionGet(openapiClientHandle_t *client_handle,
                                         OPEN_DOT1S_FORCE_VERSION_t *dot1sForceVersion);

/*********************************************************************
* @purpose  Configures Spanning Tree Guard mode
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sGuardMode        @b{(input)}   spanning tree Guard mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfGuardModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                                          OPEN_DOT1S_PORT_GUARD_t dot1sGuardMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Guard mode
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    ifNum              @b{(input)}   ID of the interface
* @param    dot1sGuardMode     @b{(output)}  spanning tree Guard mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfGuardModeGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_DOT1S_PORT_GUARD_t *dot1sGuardMode);

/*********************************************************************
* @purpose  Creates/Deletes Spanning Tree MST Instance
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId @b{(input)}   spanning tree MST instance Id
* @param    -                  Minimum value @ref OPENAPI_DOT1S_MSTID_MIN
* @param    -                  Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
* @param    action             @b{(input)}   create/delete instance
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceSet(openapiClientHandle_t *client_handle,
                                        uint32_t dot1sMSTInstanceId,
                                        OPEN_ACTION_t action);

/*********************************************************************
* @purpose  Gets First Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    instanceId                @b{(output)}  first spanning tree MST instance Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        No next instance found.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceFirstGet(openapiClientHandle_t *client_handle,
                                             uint32_t *instanceId);

/*********************************************************************
* @purpose  Gets Next Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    currInstance              @b{(input)}   current spanning tree MST instance Id
* @param    nextInstance              @b{(output)}  next spanning tree MST instance Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        No next instance found.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceNextGet(openapiClientHandle_t *client_handle,
                                            uint32_t currInstance,
                                            uint32_t *nextInstance);

/*********************************************************************
* @purpose  Configures Spanning Tree MST Instance priority
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance    @b{(input)}   spanning tree MST instance
* @param    -                   Minimum value @ref OPENAPI_DOT1S_MSTID_MIN
* @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
* @param    priority            @b{(input)}   instance priority
* @param    -                   Minimum value @ref OPENAPI_DOT1S_PRIORITY_MIN  
* @param    -                   Maximum value @ref OPENAPI_DOT1S_PRIORITY_MAX 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstancePrioritySet(openapiClientHandle_t *client_handle, 
                                                uint32_t dot1sMSTInstance,
                                                uint32_t priority);

/*********************************************************************
* @purpose  Gets Spanning Tree MST Instance priority
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance          @b{(input)}   spanning tree MST instance
* @param    priority                  @b{(output)}  instance priority
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstancePriorityGet(openapiClientHandle_t *client_handle, 
                                                uint32_t dot1sMSTInstance,
                                                uint16_t *priority);

/*********************************************************************
* @purpose  Adds/Deletes Spanning Tree MST Instance-VLAN association
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance    @b{(input)}   spanning tree MST instance
* @param    -                   Minimum value @ref OPENAPI_DOT1S_MSTID_MIN
* @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
* @param    vlan                @b{(input)}   VLAN
* @param    -                   Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID  
* @param    -                   Maximum value openapiDot1VlanMaxGet()
* @param    action              @b{(input)}   create/delete association
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceVLANSet(openapiClientHandle_t *client_handle, 
                                            uint32_t dot1sMSTInstance,
                                            uint32_t vlan,
                                            OPEN_ACTION_t action);
/*********************************************************************
* @purpose  Gets First VLAN Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    VLANId                    @b{(output)}  VLAN Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceFirstVLANGet(openapiClientHandle_t *client_handle,
                                                 uint32_t dot1sInstance,
                                                 uint32_t *VLANId);

/*********************************************************************
* @purpose  Gets Next VLAN Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    currVLANId                @b{(input)}   current VLAN Id
* @param    nextVLANId                @b{(output)}  next VLAN Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceNextVLANGet(openapiClientHandle_t *client_handle,
                                                uint32_t dot1sInstance,
                                                uint32_t currVLANId, uint32_t *nextVLANId);

/*********************************************************************
* @purpose  Gets First Forwarding Database Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    FId                       @b{(output)}  Forwarding Database Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceFirstFIdGet(openapiClientHandle_t *client_handle,
                                                uint32_t dot1sInstance,
                                                uint32_t *FId);

/*********************************************************************
* @purpose  Gets Next Forwarding Database Id for the Spanning Tree MST Instance
*
* @param    client_handle             @b{(input)}   client handle from registration API
* @param    dot1sInstance             @b{(input)}   spanning tree MST instance
* @param    currFId                   @b{(input)}   current Forwarding Database Id
* @param    nextFId                   @b{(output)}  next Forwarding Database Id
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        no next VLAN found
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SMSTInstanceNextFIdGet(openapiClientHandle_t *client_handle,
                                               uint32_t dot1sInstance,
                                               uint32_t currFId, uint32_t *nextFId);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Bridge Identifier for a specific instance.
 *
 * @param    client_handle    @b{(input)}   client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}   spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
 * @param    id               @b{(output)}   bridge identifier 
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceBridgeIdentifierGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t dot1sMSTInstance, 
                                                        OPEN_DOT1S_BRIDGEID_t *id);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Time since the last Topology Change for a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX 
 * @param    timeVal          @b{(output)}   time value, in seconds 
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(openapiClientHandle_t *client_handle, 
                                                               uint32_t dot1sMSTInstance, 
                                                               uint32_t *timeVal);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST count of Topology Changes in a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    count            @b{(output)}   number of TCNs received
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceTopologyChangeCountGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t mstId, 
                                                           uint32_t *count);

/*****************************************************************************
 * @purpose  Get if a toplogy change is in progress for any port assigned to the MST instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    change           @b{(output)}   OPEN_TRUE, a change in progress
 *                                           OPEN_FALSE, no change in progress.
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceTopologyChangeStateGet(openapiClientHandle_t *client_handle, 
                                                           uint32_t dot1sMSTInstance, 
                                                           OPEN_BOOL_t *change);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Designated Root Identifier in a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    rootId           @b{(output)}   designated root identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceDesignatedRootIDGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t dot1sMSTInstance,
                                                        OPEN_DOT1S_BRIDGEID_t *rootId);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Designated Root Path Cost for a specific instance
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    cost             @b{(output)}   designated root path cost
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceRootPathCostGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t dot1sMSTInstance, 
                                                    uint32_t *cost);

/*****************************************************************************
 * @purpose  Get the Spanning Tree MST Root Port ID in a specific instance.
 *
 * @param    client_handle    @b{(input)}    client handle from registration API
 * @param    dot1sMSTInstance @b{(input)}    spanning tree MST instance
 * @param    -                   Minimum value @ref OPENAPI_DOT1S_CST_ID
 * @param    -                   Maximum value @ref OPENAPI_DOT1S_MSTID_MAX
 * @param    port             @b{(output)}   port ID of the root port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    Use a dot1sMSTInstance value of @ref OPENAPI_DOT1S_CST_ID to access
 *           the CST instance instead of an MST instance.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sMSTInstanceRootPortIDGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t dot1sMSTInstance, 
                                                  uint32_t *port);

/*****************************************************************************
 * @purpose   Get the max age for the CIST.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     age            @b{(output)}   the current CIST max age
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTMaxAgeGet(openapiClientHandle_t *client_handle, 
                                       uint32_t *age);

/*****************************************************************************
 * @purpose   Get the Max Hop count for the device.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     count          @b{(output)}   max hop count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sBridgeMaxHopGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *count);

/*****************************************************************************
 * @purpose   Get the Tx Hold count for the device.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     count          @b{(output)}   Tx hold count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sBridgeTxHoldCountGet(openapiClientHandle_t *client_handle, 
                                              uint32_t *count);

/*****************************************************************************
 * @purpose   Get the CIST Bridge Forward Delay.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     delay          @b{(output)}   forward delay for the CIST
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTRootFwdDelayGet(openapiClientHandle_t *client_handle, 
                                             uint32_t *delay);

/*****************************************************************************
 * @purpose   Get the CIST Hello Time of a specific port.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     ifNum          @b{(input)}    interface number
 * @param     helloTime      @b{(output)}   hello time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTHelloTimeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, 
                                          uint32_t *helloTime);

/*****************************************************************************
 * @purpose   Get the CIST Bridge Hold Time.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     holdTime       @b{(output)}   hold time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTBridgeHoldTimeGet(openapiClientHandle_t *client_handle, 
                                               uint32_t *holdTime);

/*****************************************************************************
 * @purpose   Get the CIST Regional Root Identifier.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     rootId         @b{(output)}   regional root identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTRegionalRootIDGet(openapiClientHandle_t *client_handle, 
                                               OPEN_DOT1S_BRIDGEID_t *rootId);

/*****************************************************************************
 * @purpose   Get the CIST Regional Root Path Cost.
 *
 * @param     client_handle  @b{(input)}    client handle from registration API
 * @param     cost           @b{(output)}   regional root path cost
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1sCISTRegionalRootPathCostGet(openapiClientHandle_t *client_handle, 
                                                     uint32_t *cost);

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Admin Mode on an interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    dot1sMode       @b{(input)}   spanning tree mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum, 
                                     OPEN_CONTROL_t dot1sMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Admin Mode on an interface
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    dot1sMode       @b{(output)}  spanning tree admin mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfModeGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum, 
                                     OPEN_CONTROL_t *dot1sMode);

/*********************************************************************
* @purpose  Gets Spanning Tree Port Forwarding State
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
* @param    dot1sPortFwdState      @b{(output)}  spanning tree port forwarding state
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfPortFwdStateGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             uint32_t dot1sInstance,
                                             OPEN_DOT1S_PORT_FWD_STATE_t *dot1sPortFwdState);

/*********************************************************************
* @purpose  Gets Spanning Tree Port Role
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
* @param    dot1sPortRole          @b{(output)}  spanning tree port role
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfPortRoleGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         uint32_t dot1sInstance,
                                         OPEN_DOT1S_PORT_ROLE_t *dot1sPortRole);

/*********************************************************************
* @purpose  Configures the Spanning Tree MST Port Administrative Edge Port mode of an interface for the CIST
*
* @param    client_handle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
* @param    loopInconsistentState  @b{(output)}  loop inconsistent state (a boolean value)
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfPortLoopInconsistentStateGet(openapiClientHandle_t *client_handle,
                                                          uint32_t ifNum, 
                                                          uint32_t dot1sInstance,
                                                          OPEN_BOOL_t *loopInconsistentState);

/*********************************************************************
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sEdgePortMode     @b{(input)}   spanning tree edge port mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfEdgePortModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             OPEN_CONTROL_t dot1sEdgePortMode);

/*********************************************************************
* @purpose  Gets the Spanning Tree MST Port Administrative Edge Port mode of an
*           interface for the CIST
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    dot1sEdgePortMode     @b{(output)}  spanning tree edge port mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfEdgePortModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             uint32_t *dot1sEdgePortMode);

/*********************************************************************
* @purpose  Gets the MSTP Hello time of an interface for the CIST
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    helloTime             @b{(output)}  spanning tree port hello time
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfHelloTimeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum,
                                          uint32_t *helloTime); 

/*********************************************************************
* @purpose  Gets the MSTP BPDU guard effect of an interface for the CIST
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    mode                  @b{(output)}  spanning tree BPDU guard effect mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1SIntfBPDUGuardEffectGet(openapiClientHandle_t *client_handle, 
                                                uint32_t ifNum,
                                                OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Get the MSTP Port Identifier of a specific port in a
*           specific instance
*
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance  @b{(input)}   spanning tree MST instance
* @param    ifNum             @b{(input)}   ID of the interface
* @param    portID            @b{(output)}  port ID of the interface
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @comments an MSTID of zero will represent the CIST
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMSTIntfPortIDGet(openapiClientHandle_t *client_handle,
                                          uint32_t dot1sMSTInstance,
                                          uint32_t ifNum, 
                                          open_buffdesc *portID);  

/*********************************************************************
* @purpose  Set the MSTP Port Priority in a specific instance
*          
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance  @b{(input)}  spanning tree MST instance
* @param    ifNum             @b{(input)}  ID of the interface
* @param    priority          @b{(input)}  priority of the interface MSTP 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @comments an MSTID of zero will represent the CIST
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1SMstiPortPrioritySet(openapiClientHandle_t *client_handle,
                                             uint32_t dot1sMSTInstance,
                                             uint32_t ifNum,
                                             uint32_t priority);

/*********************************************************************
* @purpose  Get the MSTP Port Priority in a specific instance
*          
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance  @b{(input)}   spanning tree MST instance
* @param    ifNum             @b{(input)}   ID of the interface
* @param    priority          @b{(output)}  priority of the interface MSTP 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @comments an MSTID of zero will represent the CIST
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDot1SMstiPortPriorityGet(openapiClientHandle_t *client_handle, 
                                             uint32_t dot1sMSTInstance,
                                             uint32_t ifNum,
                                             uint32_t *priority);

/*********************************************************************
* @purpose  Get the MSTP Port Path Cost Mode in a specific instance
*
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance  @b{(input)}   spanning tree MST instance
* @param    ifNum             @b{(input)}   ID of the interface
* @param    costMode          @b{(output)}  current mode of the MSTP parameter
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @comments an MSTID of zero will represent the CIST
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiPortPathCostModeGet(openapiClientHandle_t *client_handle,  
                                                 uint32_t dot1sMSTInstance,
                                                 uint32_t ifNum,
                                                 OPEN_CONTROL_t *costMode);

/*********************************************************************
* @purpose  Set the MSTP Port Path Cost Mode in a specific instance
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    dot1sMSTInstance  @b{(input)} spanning tree MST instance
* @param    ifNum             @b{(input)} ID of the interface
* @param    pCost             @b{(input)} path cost of the MSTP parameter
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @comments an MSTID of zero will represent the CIST
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiPortPathCostSet(openapiClientHandle_t *client_handle,  
                                             uint32_t dot1sMSTInstance,
                                             uint32_t ifNum,
                                             uint32_t pCost);

/*********************************************************************
* @purpose  Gets the MSTP operational Port Path Cost in a specific instance
*
* @param    client_handle     @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance  @b{(input)}   spanning tree MST instance
* @param    ifNum             @b{(input)}   ID of the interface
* @param    portCost          @b{(output)}  port path cost of the MSTP parameter
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @comments an MSTID of zero will represent the CIST
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiOperPortPathCostGet(openapiClientHandle_t *client_handle,
                                                 uint32_t dot1sMSTInstance,
                                                 uint32_t ifNum,
                                                 uint32_t *portCost);

/*********************************************************************
* @purpose  Gets the CIST External Port Path Cost Mode on an interface 
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    ifNum             @b{(input)}  ID of the interface
* @param    costMode          @b{(output)} CIST path cost of the interface
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistExtPortPathCostModeGet(openapiClientHandle_t *client_handle,
                                                    uint32_t ifNum, 
                                                    OPEN_CONTROL_t *costMode);
                             
/*********************************************************************
* @purpose  Gets the CIST External Port Path Cost on an interface 
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    ifNum             @b{(input)}  ID of the interface
* @param    portCost          @b{(output)} CIST path cost of the interface
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistExtPortPathCostGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, 
                                                uint32_t *portCost);

/*********************************************************************
* @purpose  Sets the CIST External Port Path Cost on an interface
*
* @param    client_handle     @b{(input)} client handle from registration API
* @param    ifNum             @b{(input)} ID of the interface
* @param    portCost          @b{(input)} CIST path cost of the interface
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistExtPortPathCostSet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, 
                                                uint32_t portCost);

/*********************************************************************
* @purpose  Get the Spanning Tree MST Instance Designated Root Bridge 
*           Identifier on a interface.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    rootID           @b{(output)} MSTP Designated Root Bridge Identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiPortDesignatedRootIDGet(openapiClientHandle_t *client_handle,
                                                     uint32_t dot1sMSTInstance, 
                                                     uint32_t ifNum, 
                                                     open_buffdesc *rootID);

/*********************************************************************
* @purpose  Get the MSTP Designated Bridge cost in a specific
*           instance for a particular interface 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    dCost            @b{(output)} MSTP Designated Bridge cost
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiDesignatedCostGet(openapiClientHandle_t *client_handle, 
                                               uint32_t dot1sMSTInstance,
                                               uint32_t ifNum, 
                                               uint32_t *dCost);

/*********************************************************************
* @purpose  Get the MSTP Designated Bridge Identifier in a specific
*           instance for a particular interface 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    bridgeID         @b{(output)} MSTP Designated Bridge Identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiDesignatedBridgeIDGet(openapiClientHandle_t *client_handle,
                                                   uint32_t dot1sMSTInstance, 
                                                   uint32_t ifNum, 
                                                   open_buffdesc *bridgeID);

/*********************************************************************
* @purpose  Gets the MSTP Designated Port Identifier in a specific
*           instance for a particular interface 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    portID           @b{(output)} MSTP Designated Bridge Identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiDesignatedPortIDGet(openapiClientHandle_t *client_handle,
                                                 uint32_t dot1sMSTInstance, 
                                                 uint32_t ifNum, 
                                                 open_buffdesc *portID);

/*********************************************************************
* @purpose  Gets the CIST Topology Change Acknowledge parameter for a 
*           specific port in the CIST
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} Topology Change Acknowledge
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistPortTopologyChangeAckGet(openapiClientHandle_t *client_handle, 
                                                      uint32_t ifNum, 
                                                      OPEN_BOOL_t *val);

/*********************************************************************
* @purpose  Gets the MSTP Root Port HelloTime paramter for the CIST 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    helloTime        @b{(output)} Root port Hello time  
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistHelloTimeGet(openapiClientHandle_t *client_handle,   
                                          uint32_t *helloTime);

/*********************************************************************
* @purpose  Gets the CIST Operational Edge Port parameter of a
*           specific port 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    status           @b{(output)} Operational Edge Port status
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistPortOperEdgeGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, 
                                             OPEN_BOOL_t *status);

/*********************************************************************
* @purpose  Gets the CIST Operational Point to Point Mac parameter
*           of a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    status           @b{(output)} Operational Point to Point
*                                         Mac parameter
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistPortOperPointToPointGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, 
                                                     OPEN_BOOL_t *status);  

/*********************************************************************
* @purpose  Gets the CIST Regional Root Bridge Identifier for a particular interface 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    bridgeID         @b{(output)} MSTP Designated Bridge Identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistPortRegionalRootIDGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum,   
                                                   open_buffdesc *bridgeID);

/*********************************************************************
* @purpose  Gets the CIST port path cost 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    portCost         @b{(output)} CIST port path cost
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SCistPortPathCostGet(openapiClientHandle_t *client_handle, 
                                             uint32_t ifNum, 
                                             uint32_t *portCost); 

/*********************************************************************
* @purpose  Gets the MSTP Instance Loop Inconsistent State on an interface
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    status           @b{(output)} MSTP Loop Inconsistent State 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SMstiPortLoopInconsistentStateGet(openapiClientHandle_t *client_handle,
                                                          uint32_t dot1sMSTInstance,  
                                                          uint32_t ifNum, 
                                                          OPEN_BOOL_t *status); 

/*********************************************************************
* @purpose  Gets the MSTP Statistics for the number of times this interface
*           has transitioned into loop inconsistent state
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    loopCount        @b{(output)} Loop Inconsistent State Start stat 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t 
openapiDot1SPortStatsTransitionsIntoLoopInconsistentStateGet(
                 openapiClientHandle_t *client_handle,
                 uint32_t dot1sMSTInstance, 
                 uint32_t ifNum, 
                 uint32_t *loopCount);

/*********************************************************************
* @purpose  Gets the MSTP Statistics for the number of times this interface 
*           has transitioned out of loop inconsistent state
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    dot1sMSTInstance @b{(input)}  spanning tree MST instance
* @param    ifNum            @b{(input)}  ID of the interface
* @param    loopCount        @b{(output)} Loop Inconsistent State End stat 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t 
openapiDot1SPortStatsTransitionsOutOfLoopInconsistentStateGet(
      openapiClientHandle_t *client_handle,
      uint32_t dot1sMSTInstance, 
      uint32_t ifNum, 
      uint32_t *loopCount);

/*********************************************************************
* @purpose  Clears the port stats
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsClear(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum);

/*********************************************************************
* @purpose  Gets the MSTP Administrative Restricted Tcn parameter of a
*           specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    mode             @b{(output)} TCN guard status
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortRestrictedTcnGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_BOOL_t *mode);

/*********************************************************************
* @purpose  Set the MSTP Administrative Restricted Tcn parameter of a
*           specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(input)}  TCN guard status
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortRestrictedTcnSet(openapiClientHandle_t *client_handle, 
                                              uint32_t ifNum, 
                                              OPEN_BOOL_t val);

/*********************************************************************
* @purpose  Gets the MSTP Administrative Auto Edge Port parameter of a
*           specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    mode             @b{(output)} Auto Edge Port mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortAutoEdgeGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         OPEN_BOOL_t *mode);

/*********************************************************************
* @purpose  Set the MSTP Administrative Auto Edge Port parameter of a
*           specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    mode             @b{(input)}  Auto Edge Port mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortAutoEdgeSet(openapiClientHandle_t *client_handle, 
                                         uint32_t ifNum, 
                                         OPEN_BOOL_t mode);

/*********************************************************************
* @purpose  Gets the MSTP Up Time of the port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    upTime           @b{(output)} port up time
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortUpTimeGet(openapiClientHandle_t *client_handle,  
                                       uint32_t ifNum, 
                                       open_buffdesc *upTime);  

/*********************************************************************
* @purpose  Gets the MSTP Statistics for STP BPDUs Sent on a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} STP BPDUs sent
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsSTPBPDUsSentGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t ifNum, 
                                                  uint32_t *val); 

/*********************************************************************
* @purpose  Gets the MSTP Statistics for STP BPDUs Received on a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} STP BPDUs received
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsSTPBPDUsReceivedGet(openapiClientHandle_t *client_handle, 
                                                      uint32_t ifNum, 
                                                      uint32_t *val); 

/*********************************************************************
* @purpose  Gets the MSTP Statistics for RSTP BPDUs Sent on a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} RSTP BPDUs sent
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsRSTPBPDUsSentGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t ifNum, 
                                                   uint32_t *val); 

/*********************************************************************
* @purpose  Gets the MSTP Statistics for RSTP BPDUs Received on a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} RSTP BPDUs received
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsRSTPBPDUsReceivedGet(openapiClientHandle_t *client_handle, 
                                                       uint32_t ifNum, 
                                                       uint32_t *val); 

/*********************************************************************
* @purpose  Gets the MSTP Statistics for MSTP BPDUs Sent on a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} MSTP BPDUs sent
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsMSTPBPDUsSentGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t ifNum, 
                                                   uint32_t *val); 

/*********************************************************************
* @purpose  Gets the MSTP Statistics for MSTP BPDUs Received on a specific port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    val              @b{(output)} MSTP BPDUs received
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end 
*********************************************************************/
open_error_t openapiDot1SPortStatsMSTPBPDUsReceivedGet(openapiClientHandle_t *client_handle, 
                                                       uint32_t ifNum, 
                                                       uint32_t *val); 

#endif /* OPENAPI_DOT1S_H_INCLUDED */
