/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dot1x.h
*
* @purpose   IEEE 802.1x Configuration and Status
*
* @component OpEN
*
* @create    11/16/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DOT1X_H_INCLUDED
#define OPENAPI_DOT1X_H_INCLUDED

#include "openapi_common.h"
#include "openapi_user_manager.h"

/* Port authorization mode */
typedef enum
{
  OPEN_DOT1X_PORT_FORCE_UNAUTHORIZED = 1,
  OPEN_DOT1X_PORT_AUTO               = 2,
  OPEN_DOT1X_PORT_FORCE_AUTHORIZED   = 3,
  OPEN_DOT1X_PORT_NA                 = 4,
} OPEN_DOT1X_PORT_CONTROL_t;

/* Port authorization status */
typedef enum
{
  OPEN_DOT1X_PORT_STATUS_AUTHORIZED   = 1,
  OPEN_DOT1X_PORT_STATUS_UNAUTHORIZED = 2,
  OPEN_DOT1X_PORT_STATUS_NA           = 3,
} OPEN_DOT1X_PORT_STATUS_t;

/*  PAE State Machine (APM) states */
typedef enum
{
  OPEN_DOT1X_APM_INITIALIZE     = 1,
  OPEN_DOT1X_APM_DISCONNECTED   = 2,
  OPEN_DOT1X_APM_CONNECTING     = 3,
  OPEN_DOT1X_APM_AUTHENTICATING = 4,
  OPEN_DOT1X_APM_AUTHENTICATED  = 5,
  OPEN_DOT1X_APM_ABORTING       = 6,
  OPEN_DOT1X_APM_HELD           = 7,
  OPEN_DOT1X_APM_FORCE_AUTH     = 8,
  OPEN_DOT1X_APM_FORCE_UNAUTH   = 9,
} OPEN_DOT1X_APM_STATES_t;

/* Backend  State Machine (BAM) states */
typedef enum
{
  OPEN_DOT1X_BAM_REQUEST    = 1,
  OPEN_DOT1X_BAM_RESPONSE   = 2,
  OPEN_DOT1X_BAM_SUCCESS    = 3,
  OPEN_DOT1X_BAM_FAIL       = 4,
  OPEN_DOT1X_BAM_TIMEOUT    = 5,
  OPEN_DOT1X_BAM_IDLE       = 6,
  OPEN_DOT1X_BAM_INITIALIZE = 7,
} OPEN_DOT1X_BAM_STATES_t;

/* Vlan Assignment Mode */
typedef enum
{
  OPEN_DOT1X_DEFAULT_ASSIGNED_VLAN = 1,
  OPEN_DOT1X_RADIUS_ASSIGNED_VLAN  = 2,
  OPEN_DOT1X_UNAUTHENTICATED_VLAN  = 3,
  OPEN_DOT1X_GUEST_VLAN            = 4,
  OPEN_DOT1X_VOICE_VLAN            = 5,
  OPEN_DOT1X_MONITOR_MODE_VLAN     = 6,
  OPEN_DOT1X_NOT_ASSIGNED          = 7,
} OPEN_DOT1X_VLAN_ASSIGNED_MODE_t;

/* Port host mode */
typedef enum
{
  OPEN_AUTHMGR_INVALID_HOST_MODE = 0,
  OPEN_AUTHMGR_SINGLE_AUTH_MODE = 1,
  OPEN_AUTHMGR_MULTI_HOST_MODE = 2,
  OPEN_AUTHMGR_MULTI_DOMAIN_AUTH_MODE = 3,              
  OPEN_AUTHMGR_MULTI_AUTH_MODE = 4,
  OPEN_AUTHMGR_MULTI_DOMAIN_HOST_MODE = 5,               
} OPEN_AUTHMGR_HOST_CONTROL_t;

/* MAB Auth Type */
typedef enum
{
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_INVALID = 0,
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_EAP_MD5 = 1,
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_PAP = 2,
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_CHAP = 3,
} OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t;

/*********************************************************************
* @purpose  Gets the dot1x global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} Administrative mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xSystemAuthControlModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the dot1x global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  Administrative mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xSystemAuthControlModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the dot1x default authentication method.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} authentication method
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xDefaultAuthenMethodGet(openapiClientHandle_t *client_handle, OPEN_USER_MGR_AUTH_METHOD_t *method);

/*********************************************************************
* @purpose  Sets the dot1x default authentication method.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  authentication method
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xDefaultAuthenMethodSet(openapiClientHandle_t *client_handle, OPEN_USER_MGR_AUTH_METHOD_t method);

/*********************************************************************
* @purpose  Gets the dot1x port configured control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} control mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortControlModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DOT1X_PORT_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the dot1x port configured control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(input)}  control mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortControlModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DOT1X_PORT_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the dot1x port guest vlan ID.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortGuestVlanGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *vlanId);

/*********************************************************************
* @purpose  Sets the dot1x port guest vlan ID.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(input)}  VLAN ID
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortGuestVlanSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t vlanId);

/*********************************************************************
* @purpose  Sets the dot1x port MAC authentication bypass enable mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mabEnabled      @b{(output)} OPEN_TRUE if MAB enabled, else OPEN_FALSE
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortMabEnabledGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t *mabEnabled);

/*********************************************************************
* @purpose  Sets the dot1x port MAC authentication bypass enable mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mabEnabled      @b{(input)}  Set to OPEN_TRUE to enable MAB, OPEN_FALSE to disable
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortMabEnabledSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t mabEnabled);

/*********************************************************************
* @purpose  Gets the dot1x port operating control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Operating control mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortOperatingControlModeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, OPEN_DOT1X_PORT_CONTROL_t *mode);

/*********************************************************************
* @purpose  Gets the dot1x port status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    status          @b{(output)} Status
*
* @returns  OPEN_E_NONE        Status get successful.
* @returns  OPEN_E_FAIL        Status get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortStatusGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum, OPEN_DOT1X_PORT_STATUS_t *status);

/*********************************************************************
* @purpose  Gets the dot1x port PAE state.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    state           @b{(output)} State
*
* @returns  OPEN_E_NONE        State get successful.
* @returns  OPEN_E_FAIL        State get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortPaeStateGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, OPEN_DOT1X_APM_STATES_t *state);

/*********************************************************************
* @purpose  Gets the dot1x port back end authorization state.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    state           @b{(output)} State
*
* @returns  OPEN_E_NONE        State get successful.
* @returns  OPEN_E_FAIL        State get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortBackendAuthStateGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum, OPEN_DOT1X_BAM_STATES_t *state);

/*********************************************************************
* @purpose  Gets the dot1x port operational MAB mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortOperationalMabModeGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Gets the dot1x port assigned VLAN.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlan            @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        VLAN ID get successful.
* @returns  OPEN_E_FAIL        VLAN ID get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortVlanAssignedGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, uint32_t *vlan);

/*********************************************************************
* @purpose  Gets the dot1x port assigned VLAN reason.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    reason          @b{(output)} Reason
*
* @returns  OPEN_E_NONE        Reason get successful.
* @returns  OPEN_E_FAIL        Reason get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xPortVlanAssignedReasonGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, OPEN_DOT1X_VLAN_ASSIGNED_MODE_t *reason);

/*********************************************************************
* @purpose  Gets the first dot1x logical interface identifier for a given
*           physical port.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID for the physical port
* @param    logicalIfNum    @b{(output)} logical interface identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed or no logical port found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortFirstGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *logicalIfNum);

/*********************************************************************
* @purpose  Gets the next dot1x logical interface identifier on the
*           same physical port as the given logical interface identifier.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    logicalIfNum      @b{(input)}  logical interface identifier
* @param    nextLogicalIfNum  @b{(output)} next logical interface identifier
*                                          on the same physical port as the
*                                          given logical interface identifier
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed or no next logical port found.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortNextGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, uint32_t *nextLogicalIfNum);

/*********************************************************************
* @purpose  Gets the MAC address of the dot1x supplicant present on the
*           specified logical interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    macAddr         @b{output)}  supplicant's MAC address, buffdesc must
*                                        provide at least OPEN_MAC_ADDR_LEN bytes
*                                        of space
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortSupplicantMacAddrGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, open_buffdesc *macAddr);

/*********************************************************************
* @purpose  Gets the dot1x assigned VLAN ID and reason for the
*           specified logical interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlan            @b{output)}  VLAN ID
* @param    reason          @b{output)}  reason the VLAN was assigned
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiDot1xLogicalPortVlanAssignmentGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, uint32_t *vlan, OPEN_DOT1X_VLAN_ASSIGNED_MODE_t *reason);

/*********************************************************************
* @purpose  Sets the authentication host mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(input)}  Interface host mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationHostModeSet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,                                    
                                                     OPEN_AUTHMGR_HOST_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the authentication host mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Interface host mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationHostModeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_AUTHMGR_HOST_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the authentication type to be used by MAB for the 
*           specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Authentication type
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiMabPortMABAuthTypeSet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val);

/*********************************************************************
* @purpose  Gets the authentication host mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Authentication type
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiMabPortMABAuthTypeGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t *val);

/*********************************************************************
* @purpose  Sets the authentication periodic (reauth) enabled status for 
*           the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Authentication periodic status 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationPeriodicSet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_BOOL_t val);

/*********************************************************************
* @purpose  Gets the authentication periodic (reauth) enabled status
*           for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Authentication periodic status 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationPeriodicGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_BOOL_t *val);

/*********************************************************************
* @purpose  Sets the reauthentication period for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Reauthentication period 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodSet(openapiClientHandle_t *client_handle,
                                                         uint32_t ifNum,
                                                         uint32_t val);

/*********************************************************************
* @purpose  Gets the reauthentication period for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Reauthentication period 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodGet(openapiClientHandle_t *client_handle,
                                                         uint32_t ifNum,
                                                         uint32_t *val);

/*********************************************************************
* @purpose  Sets the flag to get reauthentication period from 
*           server/user for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Flag to get reauthentication period 
*                                        from server/user 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodFromServerSet(openapiClientHandle_t *client_handle,
                                                                   uint32_t ifNum,
                                                                   OPEN_BOOL_t val);

/*********************************************************************
* @purpose  Gets the flag to get reauthentication period from 
*           server/user for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Flag to get reauthentication period 
*                                        from server/user 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodServerGet(openapiClientHandle_t *client_handle,
                                                               uint32_t ifNum,
                                                               OPEN_BOOL_t *val);

/*********************************************************************
* @purpose  Sets the authentication manager global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  Administrative mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAdminModeSet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the authentication manager global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} Administrative mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAdminModeGet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t *mode);
#endif /* OPENAPI_DOT1X_H_INCLUDED */
