/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dynamic_arp_inspection.h
*
* @purpose   Dynamic ARP table configuration and status.
*
* @component OPEN
*
* @create    12/16/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DYNAMIC_ARP_INSPECTION_H_INCLUDED
#define OPENAPI_DYNAMIC_ARP_INSPECTION_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

#define OPEN_ARP_ACL_NAME_LEN_MIN 1      /**< smallest allowable ACL name */
#define OPEN_ARP_ACL_NAME_LEN_MAX 31     /**< largest allowable ACL name */

/** OpEN uses this enumeration to define ARP ACL Rule Actions */
typedef enum
{
  OPEN_ARP_ACL_RULE_ACTION_PERMIT = 1, /**< PERMIT action */
  OPEN_ARP_ACL_RULE_ACTION_DENY   = 2  /**< DENY action */
} OPEN_ARP_ACL_RULE_ACTION_t;

/*********************************************************************
* @purpose  API to Check if an ARP ACL exists
*
* @param    client_handle   @b{(input)} client handle from registration API.
* @param    aclName         @b((input)) ACL Name
*
* @returns  OPEN_E_NONE   ACL exists.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   ACL does not exist.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpAclGet(openapiClientHandle_t *client_handle,
                              open_buffdesc *aclName);

/*********************************************************************
* @purpose  API not supported
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    aclName         @b((input))   ACL Name
* @param    ipAddr          @b((input))   IP address
* @param    macAddr         @b((input))   MAC address
* @param    ipAddrNext      @b((output))  Next IP address
* @param    macAddrNext     @b((output))  Next MAC address
*
* @returns  OPEN_E_UNAVAIL    Functionality unavailable
*
* @notes    This function was deprecated in API version 1.18. Please use
*           @ref openapiArpAclRuleNextGetV2 in new code.
*
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.14
* @deprecated-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiArpAclRuleNextGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *aclName, 
                                        open_inet_addr_t ipAddr, 
                                        open_buffdesc *macAddr,
                                        open_inet_addr_t *ipAddrNext, 
                                        open_buffdesc *macAddrNext);

/*********************************************************************
* @purpose  API to get the next ARP ACL Rule in the given ACL
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    aclName         @b((input))   ACL Name
* @param    seqNo           @b((input))   Rule Sequence number
* @param    seqNoNext       @b((output))  Sequence number of next rule
* @param    ipAddrNext      @b((output))  IP address in next rule
* @param    macAddrNext     @b((output))  MAC address in next rule
* @param    actionNext      @b((output))  Action in next rule
*
* @returns  OPEN_E_NONE       Next rule returned.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       End of entries, or some other error occurred.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiArpAclRuleNextGetV2(openapiClientHandle_t *client_handle,
                                        open_buffdesc *aclName, 
                                        uint32_t seqNo, 
                                        uint32_t *seqNoNext, 
                                        open_inet_addr_t *ipAddrNext, 
                                        open_buffdesc *macAddrNext,
                                        OPEN_ARP_ACL_RULE_ACTION_t *actionNext);

/*********************************************************************
* @purpose  API to get the next ARP ACL entry
*
* @param    client_handle   @b{(input)} client handle from registration API.
* @param    aclName         @b((input))  ACL Name
* @param    aclNameNext     @b((output)) Next ACL Name
*
* @returns  OPEN_E_NONE       Next rule returned.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       End of entries, or some other error occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpAclNextGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *aclName, 
                                  open_buffdesc *aclNameNext);

/*********************************************************************
* @purpose Set the global dynamic ARP inspection source MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(input)}  Source MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to set data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiSourceMacValidateModeSet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose Get the global dynamic ARP inspection source MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(output)} Source MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to get data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiSourceMacValidateModeGet(openapiClientHandle_t *client_handle,
                                                OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose Set the global dynamic ARP inspection destination MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(input)}  Destination MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to set data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiDestMacValidateModeSet(openapiClientHandle_t *client_handle,
                                              OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose Get the global dynamic ARP inspection destination MAC validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(output)} Destination MAC validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to get data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiDestMacValidateModeGet(openapiClientHandle_t *client_handle,
                                              OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose Set the global dynamic ARP inspection IP address validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(input)}  IP address validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to set data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiIpAddrValidateModeSet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose Get the global dynamic ARP inspection IP address validate mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API.
* @param    mode              @b{(output)} IP address validate mode.
*
* @returns  OPEN_E_NONE       Operation succeeded.
* @returns  OPEN_E_FAIL       Unable to get data value.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end                                   
*********************************************************************/
open_error_t openapiDaiIpAddrValidateModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_CONTROL_t *mode);

#endif /* OPENAPI_DYNAMIC_ARP_INSPECTION_H_INCLUDED */
