/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_instru_bhd.h
*
* @purpose   Instrumentation BHD Specific APIs
*
* @component OPEN
*
* @create    7/19/2016
*
* @end
*
**********************************************************************/


#ifndef OPENAPI_BHD_H_INCLUDED
#define OPENAPI_BHD_H_INCLUDED
#include <string.h>
#include <stdio.h>
#include <stdbool.h>

#include <netinet/in.h>
#include "openapi_common.h"
#include "openapi_instru_nos.h"

#define OPEN_BHD_MAX_PACKET_SIZE                OPEN_INSTRU_MAX_PACKET_SIZE

/* Maximum number of ports supported to participate in BHD */
#define BVIEW_BHD_MAX_SUPPORTED_PORTS           OPEN_ASIC_MAX_PORTS

/* Maximum size of the sampled packet reported in asynchronous BHD event reports */ 
#define BVIEW_BHD_MAX_UNITS                     1

#define BVIEW_BHD_WATERMARK_MIN                 100
#define BVIEW_BHD_WATERMARK_MAX                 10000

#define BVIEW_BHD_SAMPLE_PERIODICITY_MIN        1
#define BVIEW_BHD_SAMPLE_PERIODICITY_MAX        (60*60)

#define BVIEW_BHD_SAMPLE_COUNT_MIN              0
#define BVIEW_BHD_SAMPLE_COUNT_MAX              30

#define BVIEW_BHD_VLAN_MIN                      1
#define BVIEW_BHD_VLAN_MAX                      4094

#define BVIEW_BHD_SOURCE_UDP_PORT_MIN           1
#define BVIEW_BHD_SOURCE_UDP_PORT_MAX           65535

#define BVIEW_BHD_DEST_UDP_PORT_MIN             1
#define BVIEW_BHD_DEST_UDP_PORT_MAX             65535

#define BVIEW_BHD_SAMP_POOL_SIZE_MIN            1024
#define BVIEW_BHD_SAMP_POOL_SIZE_MAX            100000

#define OPEN_BHD_MAX_PORT_LIST_STR              256
#define OPEN_BHD_MAX_PORT_STR                   8

#define OPEN_BHD_SFLOW_AGENT_NUM                1
#define OPEN_BHD_SFLOW_SESSION                  1

/* Black Holed packet Sampling methods*/
typedef enum 
{
  /* Agent sampling */	
  OPEN_BHD_AGENT_SAMPLING = (1 << 0),
  /* sFlow sampling */
  OPEN_BHD_SFLOW_SAMPLING = (1 << 1),
  OPEN_BHD_MAX_SAMPLING_METHOD

}OPEN_BHD_PKT_SAMPLING_METHOD_t;

#define BVIEW_BHD_DEFAULT_PKT_SAMPLING_METHOD  BVIEW_BHD_AGENT_SAMPLING

/* Sampling Parameters */
typedef union
{
  struct open_agent_sampling_params
  {
    /* This represents the traffic rate above 
       which traffic is considered as Black holed. 
       Sampling would start only after the water-mark level is crossed  */
    uint32_t water_mark;
    /* Time interval in seconds. */
    uint32_t sample_periodicity;
    /* Number of samples to be sent with in sample_periodicity interval */
    uint32_t sample_count;

  } agent_sampling_params;

  struct open_sflow_sampling_params 
  {
    /* vlan-id  of sflow encapsulation header */
    uint32_t encap_vlan_id;
    /* Destination IP address sflow encapsulation header */
    struct in_addr encap_dest_ip_addr;
    /* Source udp port number of sflow encapsulation header */
    uint16_t encap_src_udp_port;
    /* Destination udp port number of sflow encapsulation header */
    uint16_t encap_dest_udp_port;
    /* Mirror Port number on which sflow encapsulated 
       sample packet is sent out                      */
    uint32_t mirror_port;
    char mirror_port_str[OPEN_BHD_MAX_PORT_STR];

    /* Represents the packet pool size for sampling. 
       One packet is sampled for each pool. Minimum is 1024 */
    uint32_t sample_pool_size;

  } sflow_sampling_params;

}OPEN_BHD_PKT_SAMPLING_PARAMS_t;


/* Sampling configuration */
typedef struct 
{
  /* sampling method */
  OPEN_BHD_PKT_SAMPLING_METHOD_t sampling_method;
  /* Sampling Parameters */
  OPEN_BHD_PKT_SAMPLING_PARAMS_t sampling_params; 

} OPEN_BHD_PKT_SAMPLING_CONFIG_t;

typedef struct 
{
  /* List of ports participating in BHD  */
  OPEN_PORT_MASK_t black_hole_port_mask;
  uint8_t          black_hole_port_list_str[OPEN_BHD_MAX_PORT_LIST_STR];

  /* Pkt sampling configuration */
  OPEN_BHD_PKT_SAMPLING_CONFIG_t sampling_config;

} OPEN_BLACK_HOLE_CONFIG_t;

/*sFlow sampling status per port */
typedef struct 
{
  /* Ingress port number */
  int32_t port;
  char portStr[OPEN_BHD_MAX_PORT_STR];
  /* sFlow sampling is enabled or not */ 
  bool sflowSamplingEnabled;
  /* Total number of packets sampled since sflow sampling is enabled */  
  uint32_t sampled_pkt_count;
  /* Total number of packets black holed since sflow sampling is enabled */  
  uint32_t black_holed_pkt_count;

} OPEN_BHD_PORT_SFLOW_SAMPLING_STATUS_t;

/*********************************************************************
* @purpose  BHD feature sflow configuration get function
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    port          @b{(input)}         port number
* @param    port_list     @b{(input)}         Spine port list 
* @param    status        @b{(output)}        Sflow sampling status
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
*
* @end
*********************************************************************/
open_error_t openapiBhdSflowSamplingStatusGet(openapiClientHandle_t *clientHandle,
                                              int32_t asic,
                                              int32_t port,
                                              open_buffdesc *port_list,
                                              OPEN_BHD_PORT_SFLOW_SAMPLING_STATUS_t  *status);


/*********************************************************************
* @purpose  BHD feature, cancel black hole
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    sFlow         @b{(input)}         Flag to remove BHD sflow config. 
* @param    port_list     @b{(input)}         Spine port list 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
*
* @end
*********************************************************************/
open_error_t openapiBhdBlackHoleCancel(openapiClientHandle_t *clientHandle,
                                       int32_t asic, int32_t sFlow, open_buffdesc *port_list);

/*********************************************************************
* @purpose  BHD feature, configure black hole
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    config        @b{(input)}         Black hole configuration 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
*
* @end
*********************************************************************/
open_error_t openapiBhdBlackHoleConfigure(openapiClientHandle_t *clientHandle,
                                          int32_t asic,
                                          OPEN_BLACK_HOLE_CONFIG_t  *config);

/*********************************************************************
* @purpose  BHD feature, asic sampling capabilities get 
*
* @param    clientHandle           @b{(input)}         OpenAPI connection Handle
* @param    asic                   @b{(input)}         Asic/Unit number 
* @param    sampling_capabilities  @b{(output)}        Sampling capabilities 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
*
* @end
*********************************************************************/
open_error_t openapiBhdAsicCapabilitiesGet(openapiClientHandle_t *clientHandle,
                                           int32_t asic,
                                           int32_t *sampling_capabilities);

/*********************************************************************
* @purpose  BHD feature, Black holed packet count get 
*
* @param    clientHandle           @b{(input)}         OpenAPI connection Handle
* @param    asic                   @b{(input)}         Asic/Unit number 
* @param    count                  @b{(output)}        Black holed packet count 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
*
* @end
*********************************************************************/
open_error_t openapiBhdPacketCountGet(openapiClientHandle_t *clientHandle,
                                      int32_t asic,
                                      uint64_t *count);
#endif
